<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\Countries;
use Nette\Application\UI\Form;
use Nette\Application\UI\ITemplateFactory;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderAddressTexts;
use EshopOrders\Model\Orders;
use EshopOrders\Model\OrderAddresss;

class OrderAddressForm extends BaseControl
{
	/** @var int @persistent */
	public $orderId;

	/** @var int @persistent */
	public $orderAddressType;

	/** @var Order */
	public $order;

	/** @var OrderAddress */
	public $orderAddress;

	/** @var Orders */
	protected $ordersService;

	/** @var Countries */
	protected $countriesService;

	public function __construct(Orders $orders, Countries $countries)
	{
		$this->ordersService    = $orders;
		$this->countriesService = $countries;
	}

	public function render()
	{
		$this->template->address = $this->orderAddress;
		$this->template->render($this->getTemplateFile('OrderAddressForm', 'Order'));
	}

	public function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->orderId && $this->orderAddressType && !$this->orderAddress) {
			$this->setOrderAddress($this->orderId, $this->orderAddressType);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */


	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addGroup('1');
		$form->addText('firstName', 'default.firstName')->setMaxLength(255);
		$form->addText('lastName', 'default.lastName')->setRequired()->setMaxLength(255);
		$form->addText('street', 'default.street')->setMaxLength(255);
		$form->addText('city', 'default.city')->setMaxLength(255);
		$form->addText('postal', 'default.zipNumber')->setMaxLength(255);
		$countries = $this->countriesService->getAllNameColumn();
		$form->addSelect('country', 'eshopOrders.orderPage.country', $countries);

		$form->addGroup('2');
		$form->addText('email', 'default.email')->setMaxLength(255);
		$form->addText('phone', 'default.phone')->setMaxLength(255);
		$form->addText('company', 'eshopOrders.orderPage.company')->setMaxLength(255);
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')->setMaxLength(255);
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber')->setMaxLength(255);

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->orderAddress) {
				$orderAddress = $this->orderAddress;
				$flashMessage = 'eshopOrders.addressForm.edited';
			} else {
				throw new Exception("order address must be set");
			}

			$orderAddress->setFirstName($values->firstName);
			$orderAddress->setLastName($values->lastName);
			$orderAddress->setStreet($values->street);
			$orderAddress->setCity($values->city);
			$orderAddress->setPostal($values->postal);
			$orderAddress->setEmail($values->email);
			$orderAddress->setPhone($values->phone);
			$orderAddress->setCompany($values->company);
			$orderAddress->setIdNumber($values->idNumber);
			$orderAddress->setVatNumber($values->vatNumber);

			if ($values->country)
				$orderAddress->setCountry($this->countriesService->getReference($values->country));

			$this->em->persist($orderAddress)->flush();

			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	public function setOrderAddress($orderId, $addressType = 'delivery')
	{
		$this->order            = $this->ordersService->get($orderId);
		$this->orderId          = $orderId;
		$this->orderAddressType = $addressType;
		if ($addressType == 'delivery') {
			$this->orderAddress = $this->order->getAddressDelivery();
		} else {
			$this->orderAddress = $this->order->getAddressInvoice();
		}

		if ($this->orderAddress) {
			$i = $this->orderAddress;
			$this['form']->setDefaults([
				'firstName' => $i->getFirstName(),
				'lastName'  => $i->getLastName(),
				'street'    => $i->getStreet(),
				'city'      => $i->getCity(),
				'postal'    => $i->getPostal(),
				'country'   => $i->getCountry() ? $i->getCountry()->getId() : null,
				'email'     => $i->getEmail(),
				'phone'     => $i->getPhone(),
				'company'   => $i->getCompany(),
				'idNumber'  => $i->getIdNumber(),
				'vatNumber' => $i->getVatNumber(),
			]);
		}
	}
}
