<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Payment;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Payments;

class PaymentForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	/** @var Payment */
	public $payment;

	/** @var Payments */
	protected $paymentsService;

	public function __construct(Payments $payments)
	{
		$this->paymentsService = $payments;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'default.name')->setRequired()->setMaxLength(255);
		$form->addText('ident', 'eshopOrders.paymentForm.itemIdent')->setMaxLength(255);
		$form->addText('text', 'default.text')->setMaxLength(255);
		$form->addBool('isPublished', 'default.isActive')->setDefaultValue(1);
		$form->addInteger('price', 'eshopOrders.default.priceSimple');
		$form->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
		$form->addInteger('availableFrom', 'eshopOrders.default.availableFrom');
		$form->addInteger('availableTo', 'eshopOrders.default.availableTo');
		$form->addFilesManager('image', 'default.image');

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			PaymentForm::class . '::createForm'
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$payment      = $this->paymentsService->get($this->id);
				$flashMessage = 'eshopOrders.paymentForm.edited';
			} else {
				$payment      = new Payment();
				$flashMessage = 'eshopOrders.paymentForm.added';
			}

			$payment->setName($values->name);
			$payment->setIdent($values->ident);
			$payment->setText($values->text);
			$payment->isPublished = $values->isPublished;
			$payment->setPrice((int) $values->price);
			$payment->setFreeFrom((int) $values->freeFrom);
			$payment->setAvailableFrom((int) $values->availableFrom);
			$payment->setAvailableTo((int) $values->availableTo);
			$payment->image = $values->image;

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->getPresenter(false) ? $this->template : null,
				$this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $payment;
			$this->eventDispatcher->dispatch($event, PaymentForm::class . '::formSuccess');

			$this->em->persist($payment)->flush();
			$form->addCustomData('paymentId', $payment->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setPayment(int $id): void
	{
		$this->id      = $id;
		$this->payment = $p = $this->paymentsService->get($id);

		if (!$this->payment)
			throw new InvalidArgumentException();

		$d = [
			'name'          => $p->getName(),
			'ident'         => $p->getIdent(),
			'text'          => $p->getText(),
			'isPublished'   => $p->isPublished,
			'price'         => $p->getPrice(),
			'freeFrom'      => $p->getFreeFrom(),
			'availableFrom' => $p->getAvailableFrom(),
			'availableTo'   => $p->getAvailableTo(),
			'image'         => $p->image,
		];

		$this['form']->setDefaults($d);

		$this->eventDispatcher->dispatch(new SetFormDataEvent($this['form'], $this->payment), PaymentForm::class . '::setPayment');
	}
}
