<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Spedition;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\ExportHelper;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\Entities\SpeditionPriceLevel;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\AdminModule\Model\Speditions;

class SpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	public ?Spedition $spedition = null;

	/** @var Speditions */
	protected Speditions $speditionsService;

	/** @var ExportHelper */
	protected ExportHelper $exportHelper;

	/** @var GroupsCustomers */
	protected GroupsCustomers $groupCustomers;

	public function __construct(Speditions $speditions, ExportHelper $exportHelper, GroupsCustomers $groupCustomers)
	{
		$this->speditionsService = $speditions;
		$this->exportHelper      = $exportHelper;
		$this->groupCustomers    = $groupCustomers;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'default.name')
			->setRequired()
			->setMaxLength(255);
		$form->addText('ident', 'eshopOrders.paymentForm.itemIdent')
			->setDescription('eshopOrders.speditionForm.identDescription')
			->setMaxLength(255);
		$form->addText('code1', 'eshopOrders.speditionForm.code1')
			->setDescription('eshopOrders.speditionForm.code1Description')
			->setMaxLength(255)
			->setNullable();
		$form->addText('text', 'default.text')
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isActive')
			->setDefaultValue(1);
		$form->addBool('isPickup', 'eshopOrders.speditionForm.isPickup')
			->setDefaultValue(0);
		$form->addBool('allowOversized', 'eshopOrders.spedition.allowOversized')
			->setDefaultValue(0);
		$form->addFilesManager('image', 'default.image');
		$form->addSelect('zboziId', 'eshopOrders.speditionForm.zboziId', [null => ''] + $this->exportHelper->getZboziSpeditions());
		$form->addSelect('heurekaId', 'eshopOrders.speditionForm.heurekaId', [null => ''] + $this->exportHelper->getHeurekaSpeditions())
			->setDescription('eshopOrders.speditionForm.heurekaIdDescription');

		$form->addInteger('price', 'eshopOrders.default.priceSimple');
		$form->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
		$form->addInteger('availableFrom', 'eshopOrders.default.availableFrom')
			->setDefaultValue(0);
		$form->addInteger('availableTo', 'eshopOrders.default.availableTo')
			->setDefaultValue(999999);

		if (EshopOrdersConfig::load('enablePriceLevels', false)) {
			$container = $form->addContainer('priceLevels', 'eshopOrders.spedition.priceLevels');
			foreach ($this->groupCustomers->getOptionsForSelect() as $k => $v) {
				$group = $container->addContainer($k, $this->t('eshopOrders.spedition.priceLevel', ['group' => $v]));
				$group->addInteger('price', 'eshopOrders.default.priceSimple');
				$group->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
				$group->addInteger('availableFrom', 'eshopOrders.default.availableFrom');
				$group->addInteger('availableTo', 'eshopOrders.default.availableTo');
			}
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			SpeditionForm::class . '::createForm'
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$spedition    = $this->speditionsService->get($this->id);
				$flashMessage = 'eshopOrders.speditionForm.edited';
			} else {
				$spedition    = new Spedition();
				$flashMessage = 'eshopOrders.speditionForm.added';
			}

			$spedition->setName($values->name);
			$spedition->setIdent($values->ident);
			$spedition->setText($values->text);
			$spedition->isPublished = $values->isPublished;
			$spedition->setPrice((int) $values->price);
			$spedition->setFreeFrom((int) $values->freeFrom);
			$spedition->setAvailableFrom((int) $values->availableFrom);
			$spedition->setAvailableTo((int) $values->availableTo);
			$spedition->image          = $values->image;
			$spedition->zboziId        = $values->zboziId ?: null;
			$spedition->heurekaId      = $values->heurekaId ?: null;
			$spedition->code1          = $values->code1 ?: null;
			$spedition->isPickup       = $values->isPickup;
			$spedition->allowOversized = $values->allowOversized ? 1 : 0;

			if (EshopOrdersConfig::load('enablePriceLevels', false)) {
				$current = $spedition->getPriceLevels()->toArray();
				foreach ($values->priceLevels as $groupId => $vals) {
					if (!$vals->price)
						continue;

					$priceLevel = $current[$groupId] ?? null;
					if (!$priceLevel) {
						$priceLevel = new SpeditionPriceLevel($spedition, $this->groupCustomers->getReference($groupId), $vals->price);
						$spedition->getPriceLevels()->set($groupId, $priceLevel);
					}

					$priceLevel->setPrice($vals->price);
					$priceLevel->freeFrom      = $vals->freeFrom ?: null;
					$priceLevel->availableFrom = $vals->availableFrom ?: 0;
					$priceLevel->availableTo   = $vals->availableTo ?: 999999;

					unset($current[$groupId]);
					$this->em->persist($priceLevel);
				}

				foreach ($current as $priceLevel)
					$this->em->remove($priceLevel);
			}

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->getPresenter(false) ? $this->template : null,
				$this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $spedition;
			$this->eventDispatcher->dispatch($event, SpeditionForm::class . '::formSuccess');

			$this->em->persist($spedition)->flush();
			$form->addCustomData('speditionId', $spedition->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setSpedition(int $id): void
	{
		$this->id        = $id;
		$this->spedition = $s = $this->speditionsService->get($id);

		if (!$s)
			throw new InvalidArgumentException();

		$d = [
			'name'           => $s->getName(),
			'ident'          => $s->getIdent(),
			'code1'          => $s->code1,
			'text'           => $s->getText(),
			'isPublished'    => $s->isPublished,
			'price'          => $s->getPrice(),
			'freeFrom'       => $s->getFreeFrom(),
			'availableFrom'  => $s->getAvailableFrom(),
			'availableTo'    => $s->getAvailableTo(),
			'image'          => $s->image,
			'zboziId'        => $s->zboziId,
			'heurekaId'      => $s->heurekaId,
			'isPickup'       => $s->isPickup ? 1 : 0,
			'allowOversized' => $s->allowOversized ? 1 : 0,
		];

		if (EshopOrdersConfig::load('enablePriceLevels', false)) {
			foreach ($s->getPriceLevels()->toArray() as $groupId => $priceLevel) {
				/** @var SpeditionPriceLevel $priceLevel */
				$d['priceLevels'][$groupId] = [
					'price'         => $priceLevel->getPrice(),
					'freeFrom'      => $priceLevel->freeFrom,
					'availableFrom' => $priceLevel->availableFrom,
					'availableTo'   => $priceLevel->availableTo,
				];
			}
		}

		$this['form']->setDefaults($d);

		$this->eventDispatcher->dispatch(new SetFormDataEvent($this['form'], $this->spedition), SpeditionForm::class . '::setSpedition');
	}
}
