<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\WebColors;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;

/**
 * @method Order|object|null getReference($id)
 * @method Order[]|null getAll()
 * @method Order|null get($id)
 */
class Orders extends BaseEntityService
{
	protected $entityClass = Order::class;

	/** @var EventDispatcher */
	protected EventDispatcher $eventDispatcher;

	/** @var WebColors */
	protected WebColors $webColors;

	public function __construct(EventDispatcher $eventDispatcher, WebColors $webColors)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->webColors       = $webColors;
	}

	/**
	 * @return OrderExpedition[]
	 */
	public function getOrdersForExpedition(): array
	{
		$data = [];

		$qb = $this->getEr()->createQueryBuilder('o')
			->select('o.id, oss.ident, GROUP_CONCAT(IDENTITY(oStat.status)) as hidden statusString')
			->innerJoin('o.orderStatuses', 'oStat')
			->innerJoin('o.spedition', 'os')
			->leftJoin('os.spedition', 'oss')
			->groupBy('o.id')
			->orderBy('o.id', 'desc');

		foreach (EshopOrdersConfig::load('expeditionOrdersGrid.status') as $v) {
			$qb->andHaving("statusString LIKE :$v")
				->setParameter($v, "%$v");
		}

		foreach ($qb->getQuery()->getResult() as $row) {
			$data[] = [
				'id'      => $row['id'],
				'service' => $row['ident'],
			];
		}

		$event = new Event([
			'data'       => $data,
			'expedition' => [],
		]);
		$this->eventDispatcher->dispatch($event, Orders::class . '::processOrdersForExpedition');

		foreach ($event->data['expedition'] as &$expedition) {
			/** @var OrderExpedition $expedition */
			$expedition->siteColors = $this->webColors->getColors($expedition->order->site->getIdent());
		}

		return $event->data['expedition'];
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	public function sendExpeditionData(array $ids): array
	{
		$event = new Event([
			'data'       => $this->getOrderIdsWithSpeditionCode($ids),
			'ok'         => [],
			'okCount'    => 0,
			'errors'     => [],
			'errorCount' => 0,
		]);

		$this->eventDispatcher->dispatch($event, Orders::class . '::sendExpeditionData');

		return [
			'okCount'    => $event->data['okCount'],
			'errorCount' => $event->data['errorCount'],
		];
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	public function generateLabel(array $ids)
	{
		$event = new Event([
			'data'       => $this->getOrderIdsWithSpeditionCode($ids),
			'ok'         => [],
			'okCount'    => 0,
			'errors'     => [],
			'errorCount' => 0,
			'files'      => [],
		]);

		$this->eventDispatcher->dispatch($event, Orders::class . '::generateLabel');

		return [
			'okCount'    => $event->data['okCount'],
			'errorCount' => $event->data['errorCount'],
			'files'      => $event->data['files'],
		];
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	protected function getOrderIdsWithSpeditionCode(array $ids): array
	{
		$result = [];
		$qb     = $this->getEr()->createQueryBuilder('o')
			->select('o.id, oss.ident')
			->innerJoin('o.spedition', 'os')
			->leftJoin('os.spedition', 'oss')
			->andWhere('o.id IN (:ids)')
			->setParameter('ids', $ids);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$result[] = [
				'id'      => $row['id'],
				'service' => $row['ident'],
			];
		}

		return $result;
	}
}
