<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Nette\Utils\ArrayHash;

class CartDetailItem extends BaseControl
{
	/** @var Carts */
	protected $cartsService;

	/** @var int */
	protected $itemId;

	public function __construct(Carts $carts)
	{
		$this->cartsService = $carts;
	}

	public function render(): void
	{
		$this->template->item = $this->getItem();
		$this->template->render($this->getTemplateFile(null, 'CartDetailItem'));
	}

	protected function createComponentForm(): ?BaseForm
	{
		$form = $this->createForm();
		$item = $this->getItem();

		$form->addText('quantity', null)
			->addRule($form::FILLED)
			->setHtmlAttribute('data-add-to-cart-quantity-input', 'instant')
			->setHtmlAttribute('data-item-id', $item->getId())
			->setDefaultValue($item->quantity);
		$form->addHidden('itemId', (string) $item->getId());

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnSuccess($form, ArrayHash $values): bool
	{
		try {
			$cartItems = $this->cartsService->getCurrentCart()->getCartItems();
			$this->eventDispatcher->dispatch(new UpdatedCartItemEvent(
				(int) $values->itemId,
				(int) $values->quantity,
				(int) $cartItems[$values->itemId]->quantity,
				), 'eshopOrders.cartUpdateItem');
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageWarning('eshopOrders.cart.countUpdateErr');
			$this->getPresenter()->redrawControl('flashes');

			return false;
		}

		return true;
	}

	protected function getItem(): ?CartItem
	{
		return $this->cartsService->getCartItem($this->itemId);
	}

	public function setItem(int $itemId): self
	{
		$this->itemId = $itemId;

		return $this;
	}
}

