<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Order;

use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\Entities\Site;
use Core\Model\Images\ImageHelper;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\PhoneInput;
use EshopCatalog\FrontModule\Model\Dao\BankAccount;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use EshopOrders\Model\PaymentSpeditions;
use EshopOrders\Model\Statuses;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Users\Model\Http\UserStorage;
use Core\Model\Templating\Filters\Price as PriceFilter;
use EshopOrders\FrontModule\Model\Dao;

class OrderForm extends BaseControl
{
	const PERSONAL_COLUMNS = ['company', 'firstName', 'lastName', 'phone', 'email', 'country', 'street', 'city',
		'postal', 'idNumber', 'vatNumber'];

	/** @var Statuses */
	protected $statusesService;

	/** @var CartHelper */
	protected $cartHelperService;

	/** @var IOrderCartDetailFactory */
	protected $orderCartDetailFactory;

	/** @var IPaySpedSummaryFactory */
	protected $paySpedSummaryFactory;

	/** @var IOrderSummaryFactory */
	protected $orderSummaryFactory;

	/** @var Orders */
	private $ordersService;

	/** @var Carts */
	private $cartsService;

	/** @var Speditions */
	private $speditionsService;

	/** @var Payments */
	private $paymentsService;

	/** @var PaymentSpeditions */
	protected $paymentSpeditionsService;

	/** @var Customers */
	private $customersService;

	/** @var Countries */
	protected $countriesService;

	/** @var SessionSection */
	protected $sessionSection;

	/** @var Spedition[] */
	protected $speditions;

	/** @var Payment[] */
	protected $payments;

	/** @var PaymentSpedition[] */
	private $paymentSpeditions;

	/** @var array|null */
	protected ?array $paymentsSpeditions = null;

	protected ?array $cCountries = null;

	/** @var Cart */
	protected $cart;

	/** @var array */
	protected $orderPageConfig;

	/** @var UserStorage */
	protected $userStorage;

	/** @var Session */
	protected $session;

	/** @var ProductsFacade */
	protected $products;

	/** @var Invoices */
	protected $invoices;

	/** @var InvoiceConfigRepository */
	protected $invoiceConfigRepository;

	/** @var Sites */
	protected $sitesService;

	/** @var PriceFilter */
	protected $priceFilter;

	protected Sellers $sellers;

	protected bool $allowNewsletterFlag = false;

	public function __construct($data, IOrderCartDetailFactory $orderCartDetailFactory, IPaySpedSummaryFactory $paySpedSummaryFactory, IOrderSummaryFactory $orderSummaryFactory, Orders $orders, Carts $carts, CartHelper $cartHelper, Speditions $speditions, Payments $payments, PaymentSpeditions $paymentSpeditions, Customers $customersService, Statuses $statuses, Countries $countries, Session $session, UserStorage $userStorage, ProductsFacade $products, Invoices $invoices, InvoiceConfigRepository $invoiceConfigRepository, Sites $sites, PriceFilter $priceFilter, Sellers $sellers)
	{
		$this->ordersService            = $orders;
		$this->cartsService             = $carts;
		$this->cartHelperService        = $cartHelper;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->customersService         = $customersService;
		$this->statusesService          = $statuses;
		$this->countriesService         = $countries;
		$this->userStorage              = $userStorage;
		$this->session                  = $session;
		$this->products                 = $products;
		$this->invoices                 = $invoices;
		$this->invoiceConfigRepository  = $invoiceConfigRepository;
		$this->sitesService             = $sites;
		$this->priceFilter              = $priceFilter;
		$this->sellers                  = $sellers;

		$this->orderCartDetailFactory = $orderCartDetailFactory;
		$this->paySpedSummaryFactory  = $paySpedSummaryFactory;
		$this->orderSummaryFactory    = $orderSummaryFactory;

		$this->sessionSection  = $session->getSection('eshopOrdersOrderForm');
		$this->orderPageConfig = $data['orderPage'];

		$this->cart = $this->cartsService->getCurrentCart();
	}

	public function render()
	{
		$siteIdent  = $this->sitesService->getCurrentSite()->getIdent();
		$this->cart = $this->cartsService->getCurrentCart();
		$this->template->setFile($this->getTemplateFile());

		$this->template->visibleCountSpedition = $this->orderPageConfig['visibleCountSpedition'];
		$this->template->visibleCountPayment   = $this->orderPageConfig['visibleCountPayment'];

		$this->template->paymentSpeditions = $this->getPaymentSpeditions();
		$this->template->speditions        = $this->getSpeditions();
		$this->template->payments          = $this->getPayments();
		$this->template->countries         = $this->getCountries();
		$this->template->cart              = $this->cart;

		$form                     = $this['form'];
		$this->template->thisForm = $form;

		$this->template->termsAndConditionsNavId = (int) $this->settings->get('eshopCatalog' . ucfirst($siteIdent) . 'TermsAndConditionsNavId', 0);
		$this->template->gdprNavId               = (int) $this->settings->get('eshopCatalog' . ucfirst($siteIdent) . 'GdprNavId', 0);

		$paymentSpeditions                           = $this->getPaymentsSpeditionsCombinations();
		$this->template->paymentSpeditionsStructured = $paymentSpeditions['structured'];
		$this->template->paymentSpeditionsReversed   = $paymentSpeditions['reversed'];

		$this->template->orderData = $this->sessionSection->orderFormData;

		$this->template->render();
	}

	public function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->sessionSection->orderFormData) {
			$this->setOrderData($this->sessionSection->orderFormData);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleSaveStep2()
	{
		$request = $this->getPresenter()->getHttpRequest();
		$data    = $this->sessionSection->orderFormData;

		$data['speditionCountry'] = $request->getPost('speditionCountry');
		foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v)
			$data[$k] = (int) $request->getPost($v)[$data['speditionCountry']];

		$this->eventDispatcher->dispatch(new SaveOrderFormDataEvent($data, $this['form']), 'eshopOrders.saveOrderFormDataStep2');
		$this->sessionSection->orderFormData = $data;

		$this->redrawControl('step3');
		$this->redrawControl('orderCartDetail');
		$this->redrawControl('orderSummary');
	}

	public function handleSaveStep3()
	{
		$request = $this->getPresenter()->getHttpRequest();
		$data    = $this->sessionSection->orderFormData;

		foreach (self::PERSONAL_COLUMNS as $k) {
			$data[$k]       = $request->getPost($k);
			$data[$k . '2'] = $request->getPost($k . '2');
		}

		$countries            = $this->countriesService->getAllNameColumn();
		$data['countryText']  = $countries[$request->getPost('country')] ?? null;
		$data['country2Text'] = $countries[$request->getPost('country2')] ?? null;

		foreach (['useAddrDeli', 'speditionCountry'] as $k) {
			$data[$k] = $request->getPost($k);
		}

		$speditionCountry = $data['speditionCountry'];
		if ($data['useAddrDeli'] === 'on') {
			if ($data['country2'] != $speditionCountry)
				$this['form']->addError('eshopOrdersFront.orderPage.badSpeditionCountry');
		} else {
			if ($data['country'] != $speditionCountry)
				$this['form']->addError('eshopOrdersFront.orderPage.badSpeditionCountry');
		}

		$this->eventDispatcher->dispatch(new SaveOrderFormDataEvent($data, $this['form']), 'eshopOrders.saveOrderFormDataStep3');
		$this['form']['validatePersonal']->validate();
		$this->sessionSection->orderFormData = $data;

		$this->redrawControl('step3');
		$this->redrawControl('orderSummary');
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */

	/**
	 * @param bool $allow
	 */
	public function allowNewsletterFlag(bool $allow = true): void { $this->allowNewsletterFlag = $allow; }

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{

		$form = $this->createForm();
		$form->setAjax();
		$cart      = $this->cartsService->getCurrentCart();
		$countries = $this->countriesService->getAllNameColumn();

		$form->addHidden('currency', 'CZK');
		$form->addGroup('speditions');

		$form->addSelect('speditionCountry', 'eshopOrders.orderPage.country', $this->getCountries())
			->setTranslator(null)
			->setDefaultValue(key($this->getCountries()))
			->setRequired();

		$speditionsContainer = $form->addContainer('speditions');
		$paymentsContainer   = $form->addContainer('payments');
		$paySpedCombinations = $this->getPaymentsSpeditionsCombinations();

		foreach ($this->getSpeditions() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['structured'][$country][$row->getId()]))
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format((float) $row->getPrice());
			}

			$speditionsContainer->addRadioList($country, 'eshopOrders.orderPage.spedition', $list)
				->addConditionOn($form['speditionCountry'], BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED);
		}

		foreach ($this->getPayments() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['reversed'][$country][$row->getId()]))
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format((float) $row->getPrice());
			}

			$paymentsContainer->addRadioList($country, 'eshopOrders.orderPage.payment', $list)
				->addConditionOn($form['speditionCountry'], BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED);
		}

		$form->addSubmit('validateSpedition', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form['speditionCountry'],
				$form['payments'],
				$form['speditions'],
			])->onClick[] = [$this, 'handleSaveStep2'];

		$form->addGroup('personal');
		$form->addText('firstName', 'eshopOrders.orderPage.name')->setRequired();
		$form->addText('lastName', 'eshopOrders.orderPage.lastName')->setRequired();
		$form->addPhone('phone', 'eshopOrders.orderPage.phone')
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addEmail('email', 'eshopOrders.orderPage.email')->setRequired();
		$form->addText('street', 'eshopOrders.orderPage.street')->setRequired();
		$form->addText('city', 'eshopOrders.orderPage.city')->setRequired();
		$form->addText('postal', 'eshopOrders.orderPage.postal')->setRequired();
		$form->addSelect('country', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);
		$form->addCheckbox('companyOrder', 'eshopOrdersForm.orderPage.companyOrder');
		$form->addText('company', 'eshopOrders.orderPage.company')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');

		//fakturacni adresa
		$useAddrDeli = $form->addCheckbox('useAddrDeli', 'eshopOrders.orderPage.useAddrDeli');
		$form->addText('company2', 'eshopOrders.orderPage.company');
		$form->addText('firstName2', 'eshopOrders.orderPage.name')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('lastName2', 'eshopOrders.orderPage.lastName')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('email2', 'eshopOrders.orderPage.email')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)
			->addRule(BaseForm::EMAIL)
			->setRequired();
		$form->addPhone('phone2', 'eshopOrders.orderPage.phone')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addText('street2', 'eshopOrders.orderPage.street')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('city2', 'eshopOrders.orderPage.city')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('postal2', 'eshopOrders.orderPage.postal')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addSelect('country2', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null)
			->addConditionOn($form['useAddrDeli'], BaseForm::EQUAL, true)
			->addRule(BaseForm::EQUAL, 'eshopOrdersFront.orderPage.badSpeditionCountry', $form['speditionCountry']);


		$form['country']->addRule(function($item, $args) {
			$data   = $this->sessionSection->orderFormData;
			$values = $args->getValues();

			if ($data['useAddrDeli'] == false && $data['disableDeliveryAddress'] === true || $data['useAddrDeli'] === true)
				return true;
			else if ($values->useAddrDeli == true)
				return true;
			else
				return $values['speditionCountry'] === $item->getValue();
		}, 'eshopOrdersFront.orderPage.badSpeditionCountry', $form);

		$validateFields = [];
		foreach (self::PERSONAL_COLUMNS as $v) {
			$validateFields[] = $form[$v];

			if (!in_array($v, ['idNumber', 'vatNumber']))
				$validateFields[] = $form[$v . '2'];
		}

		$form->addSubmit('validatePersonal', 'eshopOrders.orderPage.nextStep')
			->setValidationScope($validateFields)
			->onClick[] = [$this, 'handleSaveStep3'];

		$form->addTextArea('messageAlt', 'eshopOrders.orderPage.message')->setAttribute('class', ['messageInput']);
		$form->addTextArea('message', 'eshopOrders.orderPage.message');

		$form->addCheckbox('agreedTerms', 'eshopOrders.orderPage.agreedTerms')->setRequired();
		$form->addCheckbox('agreedQuestioning', 'eshopOrders.orderPage.agreedQuestioning')->setDefaultValue(1);

		if ($this->allowNewsletterFlag)
			$form->addCheckbox('agreedNewsletter', 'eshopOrdersFront.orderPage.agreedNewsletter')
				->setDefaultValue(1);

		$form->addSubmit('submit', 'eshopOrders.orderPage.send')
			->setHtmlAttribute('class', ['submitButton', 'eshopOrdersNext4'])
			->setHtmlId('orderFormSubmit');

		if ($this->userStorage->getIdentity()) {
			$customer = $this->customersService->getByUser($this->userStorage->getIdentity());

			if ($customer) {
				$addrDeli = $customer->getAddressDelivery();
				$addrInvo = $customer->getAddressInvoice();

				if ($addrInvo)
					$form->setDefaults($addrInvo->toArray());
				if ($addrDeli) {
					$def = [];
					foreach ($addrDeli->toArray() as $k => $v)
						$def[$k . '2'] = $v;
					$form->setDefaults($def);
				}
			}
		}

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		$orderFormEvent           = new OrderFormEvent($form);
		$orderFormEvent->template = $this->template;
		$this->eventDispatcher->dispatch($orderFormEvent, 'eshopOrders.createOrderForm');

		return $form;
	}

	public function setOrderData(array $orderData): void
	{
		$this->eventDispatcher->dispatch(new SetOrderFormDataEvent($orderData, $this['form']), 'eshopOrders.setOrderFormData');

		$form = $this['form'];
		$d    = [
			'messageAlt'        => $orderData['messageAlt'],
			'message'           => $orderData['message'],
			'useAddrDeli'       => $orderData['useAddrDeli'],
			'agreedTerms'       => $orderData['agreedTerms'],
			'agreedQuestioning' => $orderData['agreedQuestioning'],
			'agreedNewsletter'  => $orderData['agreedNewsletter'],
		];

		$spedCountry = $orderData['speditionCountry'];
		if ($spedCountry) {
			if (array_key_exists($spedCountry, $form['speditionCountry']->getItems()))
				$d['speditionCountry'] = $spedCountry;

			foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v)
				if ($orderData[$k] && isset($form[$v][$spedCountry]) && array_key_exists($orderData[$k], $form[$v][$spedCountry]->getItems()))
					$d[$k][$spedCountry] = $orderData[$k];
		}

		foreach (self::PERSONAL_COLUMNS as $k) {
			if ($orderData[$k])
				$d[$k] = $orderData[$k];

			if (!in_array($k, ['idNumber', 'vatNumber']) && $orderData[$k . '2'])
				$d[$k . '2'] = $orderData[$k . '2'];
		}

		$form->setDefaults($d);
	}

	public function formOnValidate(BaseForm $form, $a): bool
	{
		$values                              = $form->getValues();
		$cart                                = $this->cartsService->getCurrentCart();
		$cartItems                           = $cart->getCartItems();
		$this->template->formSummaryMessages = [];

		if (empty($cartItems)) {
			$form->addError('eshopOrders.orderPage.emptyCart');
			$this->template->formSummaryMessages = ['eshopOrders.orderPage.emptyCartData'];
		}

		$spedCountry = $values->speditionCountry;
		$spedition   = $values->speditions[$spedCountry] ?? null;
		$payment     = $values->payments[$spedCountry] ?? null;

		$paymentsSpeditions = $this->getPaymentsSpeditionsCombinations();
		$paySped            = $paymentsSpeditions['structured'][$spedCountry][$spedition] ?? null;
		if (!$paySped || !in_array($payment, $paySped))
			$form->addError('eshopOrdersFront.orderPage.invalidPaymentSpeditionCombination');

		$this->eventDispatcher->dispatch(new OrderFormEvent($form), 'eshopOrders.orderFormValidate');

		if ($form->hasErrors()) {
			$errs = [];
			foreach ($form->getComponents(true) as $c) {
				if ($c->getErrors())
					$errs[$c->getHtmlId()] = $c->getErrors();
			}

			$this->getPresenter()->payload->orderFormErrors  = $errs;
			$this->getPresenter()->payload->orderFormIsValid = false;

			foreach ($form->getErrors() as $e)
				$this->template->formSummaryMessages[] = $e;
			$this->redrawControl('formMessages');

			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');

			return false;
		} else {
			$this->getPresenter()->payload->orderFormIsValid = true;

			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');
		}

		return true;
		// nebude potreba take validovat jestli doprava a platba jsou aktivni a plati pro cenu kosiku?
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values)
	{
		if (!$form['submit']->isSubmittedBy()) {
			return;
		}

		try {
			/** @var Site $currentSite */
			$currentSite = $this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent());
			$order       = new Order($currentSite);

			$order->setMessage($values->message ?: $values->messageAlt);
			$order->setAgreedTerms($values->agreedTerms);

			if ($values->agreedQuestioning) {
				$orderFlagQuestioning = new OrderFlag(OrderFlag::TYPE_QUESTIONING, true, $order);
				$this->em->persist($orderFlagQuestioning);
				$order->addFlag($orderFlagQuestioning);
			}
			if ($values->agreedNewsletter) {
				$orderFlagNewsletter = new OrderFlag(OrderFlag::TYPE_NEWSLETTER, true, $order);
				$this->em->persist($orderFlagNewsletter);
				$order->addFlag($orderFlagNewsletter);
			}

			$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
			$addrInv->setFirstName($values->firstName);
			$addrInv->setLastName($values->lastName);
			$addrInv->setEmail($values->email);
			$addrInv->setPhone($values->phone);
			$addrInv->setStreet($values->street);
			$addrInv->setCity($values->city);
			$addrInv->setPostal($values->postal);
			if ($values->country) {
				$addrInv->setCountry($this->countriesService->getReference($values->country));
			}

			if ($values->companyOrder) {
				$addrInv->setCompany($values->company);
				$addrInv->setIdNumber($values->idNumber);
				$addrInv->setVatNumber($values->vatNumber);
			}
			$this->em->persist($addrInv);

			$order->setAddressInvoice($addrInv);

			if ($values->useAddrDeli) {
				$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
				$addrDeli->setFirstName($values->firstName2);
				$addrDeli->setLastName($values->lastName2);
				$addrDeli->setEmail($values->email2);
				$addrDeli->setPhone($values->phone2);
				$addrDeli->setStreet($values->street2);
				$addrDeli->setCity($values->city2);
				$addrDeli->setPostal($values->postal2);
				if ($values->country2)
					$addrDeli->setCountry($this->countriesService->getReference($values->country2));
				$addrDeli->setCompany($values->company2);
				$this->em->persist($addrDeli);

				$order->setAddressDelivery($addrDeli);
			}

			$customer = $this->getOrCreateCustomer(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName(),
				$order->getAddressInvoice()->getLastName(),
				$order->getAddressInvoice()->getPhone());

			if ($customer) {
				$order->setCustomer($customer);
				$customerDeli = $customer->getAddressDelivery();
				$customerInvo = $customer->getAddressInvoice();

				if ($addrInv) {
					if (!$customerInvo)
						$customerInvo = new CustomerAddress($customer);
					$customerInvo->fillFromOrderAddress($addrInv);
					$customer->setAddressInvoice($customerInvo);
					$this->em->persist($customerInvo);
				}

				if ($addrDeli) {
					if (!$customerDeli)
						$customerDeli = new CustomerAddress($customer);
					$customerDeli->fillFromOrderAddress($addrDeli);
					$customer->setAddressDelivery($customerDeli);
					$this->em->persist($customerDeli);
				}
			}

			$cart       = $this->cartsService->getCurrentCart();
			$cartItems  = $cart->getCartItems();
			$orderItems = $this->ordersService->fillOrderItems($cartItems, $order);
			$order->setOrderItems($orderItems);

			$orderPayment = new OrderPayment($this->paymentsService->getReference((int) $values->payments[$values->speditionCountry]), $order);
			$payment      = $this->paymentsService->get((int) $values->payments[$values->speditionCountry]);
			$orderPayment->setPrice($payment->getPriceActual($cart, true));
			$order->setPayment($orderPayment);

			$orderSpedition = new OrderSpedition($this->speditionsService->getReference((int) $values->speditions[$values->speditionCountry]), $order);
			$spedition      = $this->speditionsService->get((int) $values->speditions[$values->speditionCountry]);
			$orderSpedition->setPrice($spedition->getPriceActual($cart, true));
			$order->setSpedition($orderSpedition);

			$statusCreated      = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);

			foreach ($cart->getGifts() as $gift) {
				$orderGift        = new OrderGift($order, $this->em->getReference(Product::class, $gift->productId), $gift->name);
				$orderGift->ean   = $gift->ean;
				$orderGift->code1 = $gift->code1;

				$this->em->persist($orderGift);
			}

			//			foreach ($cart->getDiscounts() as $k => $v) {
			//				$discount = new OrderDiscount($v->id, $order);
			//				$discount->setValue($v->amount);
			//				$discount->setType($v->type);
			//				$discount->setPrice($cart->calculateDiscount($v));
			//				$discount->setName($this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol);
			//				$this->em->persist($discount);
			//				$order->addOrderDiscount($discount);
			//			}

			// ------------------------------------ INVOICE ------------------------------------------------------------
			if ($invoiceIdent = $this->invoiceConfigRepository->generateIdent($currentSite->getIdent())) {

				/** @var \EshopCatalog\FrontModule\Model\Dao\Seller $seller */
				$seller  = $this->sellers->getSellerForSite($currentSite->getIdent());
				$invoice = new Invoice($this->invoiceConfigRepository->getDueDate($currentSite->getIdent()), $invoiceIdent, $currentSite);
				$order->setInvoice($invoice);

				// invoice payment
				$invoicePayment        = new Invoice\Payment();
				$invoicePayment->name  = $orderPayment->getName();
				$invoicePayment->price = $orderPayment->getPrice();

				// invoice spedition
				$invoiceSpedition        = new Invoice\Spedition();
				$invoiceSpedition->name  = $orderSpedition->getName();
				$invoiceSpedition->price = $orderSpedition->getPrice();

				// customer address
				$customerAddress          = new Invoice\Address();
				$customerAddress->city    = $addrInv->getCity();
				$customerAddress->country = $addrInv->getCountry() ? $addrInv->getCountry()->getName() : null;
				$customerAddress->postal  = $addrInv->getPostal();
				$customerAddress->street  = $addrInv->getStreet();

				// customer
				$invoiceCustomer            = new Invoice\Customer($customerAddress);
				$invoiceCustomer->phone     = $addrInv->getPhone();
				$invoiceCustomer->email     = $addrInv->getEmail();
				$invoiceCustomer->company   = $addrInv->getCompany();
				$invoiceCustomer->firstName = $addrInv->getFirstName();
				$invoiceCustomer->lastName  = $addrInv->getLastName();
				$invoiceCustomer->idNumber  = $addrInv->getIdNumber();
				$invoiceCustomer->vatNumber = $addrInv->getVatNumber();

				// bank
				$bank = new Invoice\Bank();
				// supplier address
				$supplierAddress = new Invoice\Address();
				if ($seller && $seller->getBankAccount()) {
					/** @var BankAccount $bankAccount */
					$bankAccount       = $seller->getBankAccount();
					$bank->bankAccount = $bankAccount->numberPart1;
					$bank->bankCode    = $bankAccount->numberPart2;
					$bank->iban        = $bankAccount->iban;
					$bank->swift       = $bankAccount->swift;

					$supplierAddress->street = $seller->street;
					$supplierAddress->postal = $seller->postal;
					$supplierAddress->city   = $seller->city;

					// supplier
					$supplier             = new Invoice\Supplier($bank, $supplierAddress);
					$supplier->email      = $seller->email;
					$supplier->vatNumber  = $seller->dic;
					$supplier->idNumber   = $seller->ic;
					$supplier->name       = $seller->name;
					$supplier->isPayerVat = $seller->dic ? true : false;
				} else {
					$bank->bankAccount = EshopOrdersConfig::load('invoice.bank.bankAccount');
					$bank->bankCode    = EshopOrdersConfig::load('invoice.bank.bankCode');
					$bank->iban        = EshopOrdersConfig::load('invoice.bank.iban');
					$bank->swift       = EshopOrdersConfig::load('invoice.bank.swift');

					$supplierAddress->street = EshopOrdersConfig::load('invoice.supplier.street');
					$supplierAddress->postal = EshopOrdersConfig::load('invoice.supplier.postCode');
					$supplierAddress->city   = EshopOrdersConfig::load('invoice.supplier.city');

					// supplier
					$supplier             = new Invoice\Supplier($bank, $supplierAddress);
					$supplier->email      = EshopOrdersConfig::load('invoice.supplier.email');
					$supplier->vatNumber  = EshopOrdersConfig::load('invoice.supplier.taxIdentificationNumber');
					$supplier->idNumber   = EshopOrdersConfig::load('invoice.supplier.companyIdentificationNumber');
					$supplier->name       = EshopOrdersConfig::load('invoice.supplier.name');
					$supplier->isPayerVat = EshopOrdersConfig::load('invoice.isPayerVat');
				}
				$bank->variableSymbol = $invoiceIdent;

				// invoice data
				$invoiceData           = new Invoice\InvoiceData($invoiceCustomer, $supplier, $invoiceSpedition, $invoicePayment, $invoice);
				$invoiceData->lang     = $this->translator->getLocale();
				$invoiceData->currency = $values->currency;

				// products
				$invoiceProducts = [];
				foreach ($orderItems as $p) {
					$product = $this->products->getProduct($p->getProductId());

					if ($product === null) {
						continue;
					}

					$invoiceProduct            = new Invoice\Product($invoiceData);
					$invoiceProduct->productId = $p->getProductId();
					$invoiceProduct->name      = $p->getOrderItemText($this->translator->getLocale())->getName();
					$invoiceProduct->price     = $p->getPrice();
					$invoiceProduct->quantity  = $p->getQuantity();
					$invoiceProduct->vatRate   = (int) $p->getVatRate();

					if ($product->getGallery()) {
						$cover = $product->getGallery()->getCover();

						// create base64 image miniature
						if ($cover && file_exists(WWW_DIR . $cover->getFilePath())) {
							$invoiceProduct->imageBase64 = ImageHelper::createBase64Miniature(WWW_DIR . $cover->getFilePath(), EshopOrdersConfig::load('invoice.productMiniature.width'), null, \Nette\Utils\Image::SHRINK_ONLY);
						}
					}

					$invoiceProducts[] = $invoiceProduct;
				}

				// discounts
				$invoiceDiscounts = [];
				foreach ($cart->getDiscounts() as $k => $v) {
					$invoiceDiscount        = new Invoice\Discount($invoiceData);
					$invoiceDiscount->price = $cart->calculateDiscount($v);
					$invoiceDiscount->name  = $this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol;
					$invoiceDiscount->type  = $v->type;
					$invoiceDiscount->value = $v->amount;
					$invoiceDiscount->code  = $v->id;

					$invoiceDiscounts[] = $invoiceDiscount;
				}

				$invoiceData->discounts = $invoiceDiscounts;
				$invoiceData->products  = $invoiceProducts;
				$invoice->invoiceData   = $invoiceData;
			}

			// ------------------------------------ END OF INVOICE -----------------------------------------------------

			$this->em->getConnection()->beginTransaction();
			try {
				$eventOrder                 = new OrderEvent($order);
				$eventOrder->addrDelivery   = $addrDeli;
				$eventOrder->addrInvoice    = $addrInv;
				$eventOrder->orderSpedition = $orderSpedition;
				$eventOrder->orderPayment   = $orderPayment;
				$eventOrder->orderItems     = $orderItems;
				$eventOrder->formData       = $values;
				$this->eventDispatcher->dispatch($eventOrder, 'eshopOrders.orderBeforeSave');
				//ulozit objednavku
				$this->em->persist($order);
				//ulozit k ni adresy
				$this->em->persist($addrInv);
				if ($addrDeli)
					$this->em->persist($addrDeli);
				//polozky doprava a platba
				$this->em->persist($orderSpedition);
				$this->em->persist($orderPayment);
				//ulozit k ni polozky
				foreach ($orderItems as $oi) {
					$this->em->persist($oi);
				}
				//akce - datum kdy byla objednavka vytvorena
				$this->em->persist($orderActionCreated);
				//pripadny slevovy kod
				if ($discount)
					$this->em->persist($discount);
				//pripadne nastavene priznaky
				if ($orderFlagQuestioning)
					$this->em->persist($orderFlagQuestioning);
				// vytvoreni faktury
				if ($invoice)
					$this->em->persist($invoice);
				//teprve po ulozeni vseho flush (aby byly spravne propojene cizi klice)
				$this->em->flush();
				$this->em->getConnection()->commit();
				//TODO nejake promazavani kosiku z DB, kdyz uz k nim neni aktivni session... nebo po nejakem timeoutu (tyden a pod)
			} catch (Exception $e) {
				$this->em->getConnection()->rollBack();
				throw $e;
			}

			//dale uz pracujeme s objednavkou nactenou z DB
			$orderDetail = $this->ordersService->get($order->getId());
			unset($order);
			$this->cartsService->deleteCurrentCart();
			$this->sessionSection->orderFormData = [];

			// Spuštění události po dokončení objednávky jako odeslání emailu, zboží,...
			$this->eventDispatcher->dispatch(new OrderEvent($orderDetail), 'eshopOrders.orderOnSuccess');
			// Vytáhnutí zpráv z událostí
			$messages = $this->session->getSection('eshopOrders/orderSuccessMessages');

			if ($messages->success)
				foreach ($messages->success as $v)
					$this->getPresenter()->flashMessageSuccess($v);
			if ($messages->error)
				foreach ($messages->error as $v)
					$this->getPresenter()->flashMessageDanger($v);
			$messages->remove();
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		$this->getPresenter()->redirect('Finished:orderFinished', ['orderIdent' => $orderDetail->getIdent()]);

		return true;
	}

	protected function getOrCreateCustomer($email, $firstName = '', $lastName = '', $phone = '')
	{
		$userId = null;
		if ($this->getPresenter()->getUser()->isLoggedIn()) {
			$userId = $this->getPresenter()->getUser()->id;
		}
		$customer = $this->customersService->getOrCreateCustomer($userId, $email, $firstName, $lastName, $phone);

		return $customer;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderCartDetail()
	{
		$control = $this->orderCartDetailFactory->create();

		return $control;
	}

	protected function createComponentPaySpedSummary()
	{
		$control = $this->paySpedSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData['spedition'], $orderData['payment'], $orderData['priceTotal']);

		return $control;
	}

	protected function createComponentOrderSummary()
	{
		$control = $this->orderSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData);

		return $control;
	}

	/**
	 * @return Payment[]
	 */
	public function getSpeditions(): array
	{
		if ($this->speditions === null) {
			/** @var Spedition[] $raw */
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getSpedition()->getId()] = $ps->getSpedition()->getId();
				}

				$raw[$ps->getSpedition()->getId()] = $ps->getSpedition();
			}
			$raw = $this->speditionsService->filterByCart($raw, $this->cart);

			$this->speditions = $this->processPaySpedData($byCountry, $raw);
		}

		return $this->speditions;
	}

	public function getPaymentsSpeditionsCombinations(): array
	{
		if ($this->paymentsSpeditions === null) {
			$filteredSpedition = $this->getSpeditions();
			$filteredPayments  = $this->getPayments();

			$paymentSpeditions        = $this->getPaymentSpeditions();
			$this->paymentsSpeditions = [
				'structured' => [],
				'reversed'   => [],
			];

			foreach ($paymentSpeditions as $key => $item) {
				foreach ($item->getCountries() as $country) {
					/** @var Country $country */
					$spedId    = $item->getSpedition()->getId();
					$payId     = $item->getPayment()->getId();
					$countryId = $country->getId();

					if (!isset($filteredPayments[$countryId][$payId]) || !isset($filteredSpedition[$countryId][$spedId]))
						continue;

					$this->paymentsSpeditions['structured'][$countryId][$spedId][] = $payId;
					$this->paymentsSpeditions['reversed'][$countryId][$payId][]    = $spedId;
				}
			}
		}

		return $this->paymentsSpeditions;
	}

	/**
	 * @return Payment[][]
	 */
	public function getPayments(): array
	{
		if ($this->payments === null) {
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getPayment()->getId()] = $ps->getPayment()->getId();
				}

				$raw[$ps->getPayment()->getId()] = $ps->getPayment();
			}
			$raw = $this->paymentsService->filterByCartValue($raw, $this->cart->getCartItemsPrice());

			$this->payments = $this->processPaySpedData($byCountry, $raw);
		}

		return $this->payments;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getPaymentSpeditions(): array
	{
		if ($this->paymentSpeditions === null)
			$this->paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();

		return $this->paymentSpeditions;
	}

	public function getCountries(): array
	{
		if ($this->cCountries === null) {
			$this->cCountries = [];

			foreach ($this->getPaymentSpeditions() as $row)
				foreach ($row->getCountries() as $id => $country)
					$this->cCountries[$country->getId()] = $country->getName();
		}

		return $this->cCountries;
	}

	protected function processPaySpedData(array $byCountry, array $raw)
	{
		$result = [];

		foreach ($byCountry as $country => $rows) {
			foreach ($rows as $row) {
				if (!isset($raw[$row]))
					continue;

				$raw[$row]->country     = $country;
				$result[$country][$row] = $raw[$row];
			}
		}

		return $result;
	}
}
