<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCardPaymentHistory;
use EshopOrders\Model\Entities\OrderCardPaymentToken;

class CardsPaymentService extends BaseFrontEntityService
{
	protected $entityClass = OrderCardPaymentToken::class;

	public function createToken(Order $order): OrderCardPaymentToken
	{
		$entity = new OrderCardPaymentToken($order);

		$this->em->persist($entity);

		$history = new OrderCardPaymentHistory($entity, OrderCardPaymentHistory::STATUS_CREATED);
		$this->em->persist($history);

		$this->em->flush();

		return $entity;
	}

	public function getLastCreated(int $orderId): ?OrderCardPaymentToken
	{
		return $this->getEr()->createQueryBuilder('t')
			->where('t.order = :order')->setParameter('order', $orderId)
			->orderBy('t.created', 'DESC')
			->setMaxResults(1)
			->getQuery()->getOneOrNullResult() ?: null;
	}

	public function getHistoryOfOrder($orderId): array
	{
		return $this->getEr()->createQueryBuilder('t')
			->addSelect('h')
			->leftJoin('t.history', 'h')
			->where('t.order = :order')->setParameter('order', $orderId)
			->orderBy('t.created')
			->addOrderBy('t.created')
			->getQuery()->getArrayResult();
	}

	public function tokenCheckout($token, ?string $note = null): bool
	{
		return $this->updateHistory($token, OrderCardPaymentHistory::STATUS_CHECKOUT, $note);
	}

	public function tokenPaid($token, ?string $note = null): bool
	{
		return $this->updateHistory($token, OrderCardPaymentHistory::STATUS_PAID, $note);
	}

	public function tokenError($token, ?string $note = null): bool
	{
		return $this->updateHistory($token, OrderCardPaymentHistory::STATUS_ERROR, $note);
	}

	protected function updateHistory($tokenId, string $status, ?string $note = null): bool
	{
		/** @var OrderCardPaymentToken $entity */
		$entity = $this->getEr()->find($tokenId);
		if ($entity) {
			$history       = new OrderCardPaymentHistory($entity, $status);
			$history->note = $note;
			$this->em->persist($history);

			if ($status === OrderCardPaymentHistory::STATUS_PAID) {
				$order         = $history->token->order;
				$order->isPaid = 1;
				$this->em->persist($order);
			}

			$this->em->flush();

			return true;
		}

		return false;
	}
}
