<?php declare(strict_types=1);

namespace EshopOrders\Model\Entities;

use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Utils\Helpers;
use EshopOrders\Model\Utils\Strings;
use Core\Model\Entities\TId;
use Nette\Utils\Arrays;

/**
 * @ORM\Table("eshop_orders__numerical_series")
 * @ORM\Entity
 */
class NumericalSeries
{
	use TId;

	/** @var array */
	public const POSSIBLE_PREFIX_WILDCARDS = [
		'year'  => ['YYYY', 'YY'],
		'month' => ['MM']
	];

	/**
	 * @var int
	 * @ORM\Column(type="string", nullable = false)
	 */
	public $prefix;

	/**
	 * @var int
	 * @ORM\Column(type="integer", nullable = false)
	 */
	public $digitsCount;

	/**
	 * @var int
	 * @ORM\Column(type="integer", nullable = false)
	 */
	public $startNumber;

	/**
	 * NumericalSeries constructor.
	 * @param string $prefix
	 * @param int $digitsCount
	 * @param int $startNumber
	 */
	public function __construct(string $prefix, int $digitsCount, int $startNumber)
	{
		$this->prefix = $prefix;
		$this->digitsCount = $digitsCount;
		$this->startNumber = $startNumber;
	}

	/**
	 * @return string
	 */
	public function getRealPrefix(): string
	{
		return Helpers::replaceTimeWildcards($this->prefix, Arrays::flatten(self::POSSIBLE_PREFIX_WILDCARDS));
	}

	public function increaseStartNumber(): void
	{
		if (((int) \Nette\Utils\Strings::padRight('9', $this->digitsCount, '9')) === $this->startNumber) {
			return;
		}

		$this->startNumber++;
	}

	/**
	 * @return bool
	 */
	public function containsPrefixYearWildcards(): bool
	{
		return Strings::containsMore($this->prefix, self::POSSIBLE_PREFIX_WILDCARDS['year']);
	}

	/**
	 * @return bool
	 */
	public function containsPrefixMonthWildcards(): bool
	{
		return Strings::containsMore($this->prefix, self::POSSIBLE_PREFIX_WILDCARDS['month']);
	}

}
