<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Countries;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Sites;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\PaymentSpedition;
use Doctrine\ORM\Query\Expr\Join;
use Users\Model\Security\User;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Dao;

/**
 * class PaymentSpeditions
 * @package EshopOrders\Model
 *
 * @method PaymentSpedition|object|null getReference($id)
 * @method PaymentSpedition|null get($id)
 */
class PaymentSpeditions extends BaseEntityService
{
	protected $entityClass = PaymentSpedition::class;

	/** @var Sites */
	public Sites $sitesService;

	/** @var Countries */
	protected Countries $countries;

	/** @var User */
	protected User $user;

	/** @var Customers */
	protected Customers $customers;

	/** @var Speditions */
	protected Speditions $speditions;

	/** @var Payments */
	protected Payments $payments;

	protected ?array $cPublished = null;

	public function __construct(Sites $sitesService, User $user, Customers $customers, Countries $countries,
	                            Speditions $speditions, Payments $payments)
	{
		$this->sitesService = $sitesService;
		$this->countries    = $countries;
		$this->user         = $user;
		$this->customers    = $customers;
		$this->speditions   = $speditions;
		$this->payments     = $payments;
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish($id, $state)
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getAllPublished(): array
	{
		if ($this->cPublished === null) {
			$this->cPublished = [];
			$payments         = $this->payments->getAllPublished();
			$speditions       = $this->speditions->getAllPublished();

			$qb = $this->getEr()->createQueryBuilder('ps')
				->select('ps.id as id, p.id as payment, s.id as spedition, GROUP_CONCAT(c.id) as countries')
				->andWhere('ps.isPublished = :isPublished')
				->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
				->innerJoin('ps.countries', 'c')
				->innerJoin('ps.payment', 'p')
				->innerJoin('ps.spedition', 's')
				->addOrderBy('s.position', 'ASC')
				->addOrderBy('p.position', 'ASC')
				->setParameters([
					'isPublished' => 1,
					'site'        => $this->sitesService->getCurrentSite()->getIdent(),
				])
				->groupBy('ps.id');

			foreach ($qb->getQuery()->getArrayResult() as $ps) {
				$payment   = $payments[$ps['payment']];
				$spedition = $speditions[$ps['spedition']];

				if (!$payment || !$spedition)
					continue;

				$dao = new Dao\PaymentSpedition((int) $ps['id'], $payment, $spedition);

				foreach (explode(',', $ps['countries']) as $v) {
					$country = $this->countries->getDao()[$v];
					if ($country)
						$dao->addCountry($country->getId(), $country);
				}

				$this->cPublished[$ps['id']] = $dao;
			}
		}

		return $this->cPublished;
	}

	public function getAllPublishedByCountry(?string $country = null): array
	{
		if (!$country)
			return $this->getAllPublished();

		$result = [];

		foreach ($this->getAllPublished() as $row) {
			if (isset($row->getCountries()[$country]))
				$result[$row->getId()] = $row;
		}

		return $result;
	}
}
