<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\Spedition;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Orders;
use EshopOrders\AdminModule\Model\Speditions;

class OrderSpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public $orderSpeditionId;

	/** @var OrderSpedition */
	public $orderSpedition;

	/** @var Order */
	public $order;

	protected Orders $ordersService;

	protected Speditions $speditionsService;

	protected ?array $cSped = null;

	public function __construct(Orders $orders, Speditions $speditions)
	{
		$this->ordersService     = $orders;
		$this->speditionsService = $speditions;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->getElementPrototype()->setAttribute('data-allow-toggle', 'true');

		$speditions    = $this->getSpeditions();
		$speditionList = [];
		foreach ($speditions as $spedition) {
			$speditionList[$spedition->getId()] = $spedition->getName() . ' (cena ' . $spedition->getPrice() . ' Kč)';
		}

		$form->addSelect('spedition', 'eshopOrders.default.spedition', $speditionList);
		$form->addText('price', 'eshopOrders.default.speditionPrice');

		$event          = new CreateFormEvent(
			$form,
			$this->getPresenter(false) ? $this->template : null);
		$event->control = $this;

		$this->eventDispatcher->dispatch($event, OrderSpeditionForm::class . '::createForm');

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$order          = $this->order;
			$orderSpedition = $this->order->getSpedition();
			$newSpedition   = $this->speditionsService->get($values->spedition);

			$orderSpedition->setPrice((float) Strings::formatEntityDecimal($values->price));
			if (!$orderSpedition->getSpedition() || $orderSpedition->getSpedition()->getId() != $newSpedition->getId()) {
				$orderSpedition->setName($newSpedition->getName());
				$orderSpedition->setSpedition($newSpedition);
			}

			$this->em->persist($orderSpedition);

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null,
			);
			$event->custom['entity'] = $orderSpedition;
			$event->control          = $this;
			$this->eventDispatcher->dispatch($event, OrderSpeditionForm::class . '::formSuccess');

			$this->em->flush();

			$this->getPresenter()->flashMessageSuccess('eshopOrders.orderSpeditionForm.edited');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setOrder($orderId): void
	{
		$this->order          = $this->ordersService->get($orderId);
		$this->orderSpedition = $this->order->getSpedition();

		if ($this->orderSpedition) {
			$i = $this->orderSpedition;
			bdump($i);

			$defaults = ['price' => $i->getPrice()];
			if ($i->getSpedition()) {
				$defaults['spedition'] = $i->getSpedition()->getId();
			}
			$this['form']->setDefaults($defaults);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($this['form'], $this->orderSpedition), OrderSpeditionForm::class . '::setOrder');
		}
	}

	/**
	 * @return Spedition[]
	 */
	public function getSpeditions(): array
	{
		if ($this->cSped === null) {
			$this->cSped = $this->speditionsService->getAll() ?: [];
		}

		return $this->cSped;
	}
}
