<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\IOrderStatus;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\OrderStatusHistory;
use EshopOrders\Model\Orders;
use EshopOrders\Model\OrderStatuses;
use EshopOrders\Model\OrderStatusesHistory;
use Nette\Utils\Html;

class OrderStatusesGrid extends BaseControl
{
	protected Orders $ordersService;
	protected OrderStatuses $orderStatusesService;
	protected OrderStatusesHistory $orderStatusesHistoryService;

	private $orderId;

	public function __construct(Orders $orders, OrderStatuses $orderStatuses, OrderStatusesHistory $orderStatusesHistory)
	{
		$this->ordersService        = $orders;
		$this->orderStatusesService = $orderStatuses;
		$this->orderStatusesHistoryService = $orderStatusesHistory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();

		$orderStatus      = $this->orderStatusesService->get($id);
		$event            = new OrderStatusEvent($orderStatus);
		$event->presenter = $this->getPresenter();
		$this->eventDispatcher->dispatch($event, OrderStatusesGrid::class . '::beforeDelete');

		if ($this->orderStatusesService->remove((int) $id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([]);
		$grid->hideTableFooter();
		$order = $this->ordersService->getReference($this->orderId);

		// OrderStatus
		$qb = $this->orderStatusesService->getEr()->createQueryBuilder('os')
										->andWhere('os.order = :order')
										->setParameter('order', $this->orderId)
										->orderBy('os.id', 'ASC');
		$orderStatuses = $qb->getQuery()->getResult();

		// OrderStatusHistory
		$qb = $this->orderStatusesHistoryService->getEr()->createQueryBuilder('os')
										 ->andWhere('os.order = :order')
										 ->setParameter('order', $this->orderId)
										 ->orderBy('os.id', 'ASC');
		$orderStatusesHistory = $qb->getQuery()->getResult();


		// seskupeni poli a setrideni dle datumu
		$result = array_merge($orderStatuses, $orderStatusesHistory);
		usort($result, fn($d1, $d2) => $d1->getCreated()->getTimestamp() <=> $d2->getCreated()->getTimestamp());

		$grid->setDataSource($result);

		//Columns
		$grid->addColumnText('statusName', 'eshopOrders.statusesGrid.statusName', 'status.name')
			 ->setRenderer(function(IOrderStatus $row) {
				if ($row instanceof OrderStatusHistory)
					return Html::el('strike')->setText($row->getStatus()->getName());

				return $row->getStatus()->getName();
		});
		$grid->addColumnDateTime('statusTime', 'eshopOrders.statusesGrid.statusTime', 'created')->setFormat('d.m.Y H:i:s');

		// Filter

		// Actions
		$grid->addAction('delete', '', 'delete!')
			->setRenderCondition(function(IOrderStatus $row) {
				return $row instanceof OrderStatus && ($row->getStatus()->getId() === OrderStatus::STATUS_CREATED ? false : true);
			})
			->addClass('ajax')
			->setIcon('times')->setBsType('danger')
			->setConfirm('default.reallyDelete');

		// Columns prototype
		$grid->getColumn('statusTime')->getElementPrototype('td')->addAttributes(['class' => 'w1nw']);

		return $grid;
	}

	public function setOrder($id): self
	{
		$this->orderId = $id;

		return $this;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

}
