<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\PaymentSpedition;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Currencies;
use Nette\Forms\Controls\HiddenField;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Payments;
use EshopOrders\Model\PaymentSpeditions;

class PaymentSpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public ?int $id = null;

	protected PaymentSpeditions $paymentSpeditionsService;

	public PaymentSpedition $paymentSpedition;

	protected Speditions $speditionsService;

	protected Payments $paymentsService;

	protected Sites $sitesService;

	protected Countries $countriesService;

	protected Currencies $currencies;

	public function __construct(PaymentSpeditions $paymentSpeditions, Speditions $speditions, Payments $payments, Sites $sites,
	                            Countries $countries, Currencies $currencies)
	{
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->sitesService             = $sites;
		$this->countriesService         = $countries;
		$this->currencies               = $currencies;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$paymentChoices = $speditionChoices = [];
		$speditions     = $this->speditionsService->getAll();
		$payments       = $this->paymentsService->getAll();
		$sites          = $this->sitesService->getAll();

		foreach ($speditions as $s) {
			$speditionChoices[$s->getId()] = $s->getName() . ' - ' . $s->getPrice() . ' Kč';
		}
		foreach ($payments as $p) {
			$paymentChoices[$p->getId()] = $p->getName() . ' - ' . $p->getPrice() . ' Kč';
		}
		$form->addSelect('spedition', 'eshopOrders.orderPage.spedition', $speditionChoices)
			->setTranslator(null)
			->setRequired();
		$form->addSelect('payment', 'eshopOrders.orderPage.payment', $paymentChoices)
			->setTranslator(null)
			->setRequired();
		if (count($sites) > 1)
			$form->addCheckboxList('sites', 'eshopOrders.paymentSpedition.sites', $this->sitesService->getOptionsForSelect())
				->setTranslator(null)
				->setRequired();
		else
			$form->addHidden('site', $sites[0]->getIdent());

		$form->addCheckboxList('countries', 'eshopOrders.paymentSpedition.country', $this->countriesService->getAllNameColumn())
			->setTranslator(null);

		$currencies = ['' => ''];
		foreach ($this->currencies->getActive() as $c)
			$currencies[$c->getCode()] = $c->getCode();
		$form->addSelect('currency', 'eshopOrders.paymentSpedition.currency', $currencies);

		$form->addDateTimePicker('freeFromDate', 'eshopOrders.paymentSpedition.freeFromDate');
		$form->addDateTimePicker('freeToDate', 'eshopOrders.paymentSpedition.freeToDate');

		$form->addBool('isPublished', 'default.isActive')->setDefaultValue(1);
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$paymentSpedition = $this->paymentSpeditionsService->get($this->id);
				$flashMessage     = 'eshopOrders.paymentSpeditionForm.edited';
			} else {
				$paymentSpedition = new PaymentSpedition();
				$flashMessage     = 'eshopOrders.paymentSpeditionForm.added';
			}

			// Sites
			$sites     = $paymentSpedition->sites->toArray();
			$formSites = $values->site ? [$values->site] : $values->sites;

			foreach (array_diff(array_keys($sites), $formSites) as $v)
				$paymentSpedition->sites->remove($v);
			foreach (array_diff($formSites, array_keys($sites)) as $v)
				$paymentSpedition->sites->set($v, $this->sitesService->getReference($v));

			// Countries
			$countries = $paymentSpedition->countries->toArray();
			foreach (array_diff(array_keys($countries), $values->countries) as $v)
				$paymentSpedition->countries->remove($v);
			foreach (array_diff($values->countries, array_keys($countries)) as $v)
				$paymentSpedition->countries->set($v, $this->countriesService->getReference($v));

			$paymentSpedition->setPayment($this->paymentsService->get($values->payment));
			$paymentSpedition->setSpedition($this->speditionsService->get($values->spedition));
			$paymentSpedition->isPublished  = $values->isPublished;
			$paymentSpedition->freeFromDate = $values->freeFromDate;
			$paymentSpedition->freeToDate   = $values->freeToDate;
			$paymentSpedition->currency     = $values->currency ?: null;

			$this->em->persist($paymentSpedition)->flush();
			$form->addCustomData('paymentSpeditionId', $paymentSpedition->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setPaymentSpedition($id): void
	{
		$this->id               = $id;
		$this->paymentSpedition = $this->paymentSpeditionsService->get($id);

		if (!$this->paymentSpedition)
			throw new InvalidArgumentException();

		$m = $this->paymentSpedition;
		$f = $this['form'];
		$d = [
			'payment'      => $m->getPayment()->getId(),
			'spedition'    => $m->getSpedition()->getId(),
			'isPublished'  => $m->isPublished,
			'freeFromDate' => $m->freeFromDate,
			'freeToDate'   => $m->freeToDate,
		];

		if ($f->getComponent('site', false))
			$d['site'] = $m->sites->first() ? $m->sites->first()->getIdent() : null;
		else
			foreach ($m->sites->toArray() as $site)
				if (array_key_exists($site->getIdent(), $f['sites']->getItems()))
					$d['sites'][] = $site->getIdent();

		foreach ($m->countries->toArray() as $country)
			if (array_key_exists($country->getId(), $f['countries']->getItems()))
				$d['countries'][] = $country->getId();

		if ($m->currency && array_key_exists($m->currency, $f['currency']->getItems()))
			$d['currency'] = $m->currency;

		$f->setDefaults($d);
	}
}
