<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Order;

use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\Entities\Site;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\PhoneInput;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\FormValidator;
use EshopOrders\Model\Helpers\Psc;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use EshopOrders\Model\PaymentSpeditions;
use EshopOrders\Model\Statuses;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;
use Nette\Utils\Validators;
use Tracy\Debugger;
use Tracy\ILogger;
use Users\Model\Http\UserStorage;
use Core\Model\Templating\Filters\Price as PriceFilter;
use EshopOrders\FrontModule\Model\Dao;

class OrderForm extends BaseControl
{
	const PERSONAL_COLUMNS = ['company', 'firstName', 'lastName', 'phone', 'email', 'country', 'street', 'city',
		'postal', 'idNumber', 'vatNumber'];

	protected Statuses $statusesService;

	protected CartHelper $cartHelperService;

	protected IOrderCartDetailFactory $orderCartDetailFactory;

	protected IPaySpedSummaryFactory $paySpedSummaryFactory;

	protected IOrderSummaryFactory $orderSummaryFactory;

	protected Orders $ordersService;

	protected CartFacade $cartFacade;

	protected Speditions $speditionsService;

	protected Payments $paymentsService;

	protected PaymentSpeditions $paymentSpeditionsService;

	protected Customers $customersService;

	protected Countries $countriesService;

	protected SessionSection $sessionSection;

	/** @var Spedition[] */
	protected $speditions;

	/** @var Payment[] */
	protected $payments;

	/** @var PaymentSpedition[] */
	private $paymentSpeditions;

	/** @var array|null */
	protected ?array $paymentsSpeditions = null;

	protected ?array $cCountries = null;

	/** @var Cart */
	protected $cart;

	/** @var array */
	protected $orderPageConfig;

	protected UserStorage $userStorage;

	protected Session $session;

	protected ProductsFacade $products;

	protected Invoices $invoices;

	protected InvoiceConfigRepository $invoiceConfigRepository;

	protected Sites $sitesService;

	protected PriceFilter $priceFilter;

	protected Sellers $sellers;

	protected Currencies $currencies;

	protected bool $allowNewsletterFlag = false;

	protected Psc $pscHelper;

	public function __construct($data, IOrderCartDetailFactory $orderCartDetailFactory, IPaySpedSummaryFactory $paySpedSummaryFactory,
	                            IOrderSummaryFactory $orderSummaryFactory, Orders $orders, CartFacade $cartFacade, CartHelper $cartHelper,
	                            Speditions $speditions, Payments $payments, PaymentSpeditions $paymentSpeditions, Customers $customersService,
	                            Statuses $statuses, Countries $countries, Session $session, UserStorage $userStorage, ProductsFacade $products,
	                            Invoices $invoices, InvoiceConfigRepository $invoiceConfigRepository, Sites $sites, PriceFilter $priceFilter,
	                            Sellers $sellers, Currencies $currencies, Psc $pscHelper)
	{
		$this->ordersService            = $orders;
		$this->cartFacade               = $cartFacade;
		$this->cartHelperService        = $cartHelper;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->customersService         = $customersService;
		$this->statusesService          = $statuses;
		$this->countriesService         = $countries;
		$this->userStorage              = $userStorage;
		$this->session                  = $session;
		$this->products                 = $products;
		$this->invoices                 = $invoices;
		$this->invoiceConfigRepository  = $invoiceConfigRepository;
		$this->sitesService             = $sites;
		$this->priceFilter              = $priceFilter;
		$this->sellers                  = $sellers;
		$this->currencies               = $currencies;
		$this->pscHelper                = $pscHelper;

		$this->orderCartDetailFactory = $orderCartDetailFactory;
		$this->paySpedSummaryFactory  = $paySpedSummaryFactory;
		$this->orderSummaryFactory    = $orderSummaryFactory;

		$this->sessionSection  = $session->getSection('eshopOrdersOrderForm');
		$this->orderPageConfig = $data['orderPage'];

		$this->cart = $this->cartFacade->getCart();
	}

	public function render()
	{
		$siteIdent  = $this->sitesService->getCurrentSite()->getIdent();
		$this->cart = $this->cartFacade->getCart();
		$this->template->setFile($this->getTemplateFile());

		$this->template->visibleCountSpedition = $this->orderPageConfig['visibleCountSpedition'];
		$this->template->visibleCountPayment   = $this->orderPageConfig['visibleCountPayment'];

		$this->template->paymentSpeditions = $this->getPaymentSpeditions();
		$this->template->speditions        = $this->getSpeditions();
		$this->template->payments          = $this->getPayments();
		$this->template->countries         = $this->getCountries();
		$this->template->cart              = $this->cart;

		$form                     = $this['form'];
		$this->template->thisForm = $form;

		$this->template->termsAndConditionsNavId = (int) $this->settings->get('eshopCatalog' . ucfirst($siteIdent) . 'TermsAndConditionsNavId', 0);
		$this->template->gdprNavId               = (int) $this->settings->get('eshopCatalog' . ucfirst($siteIdent) . 'GdprNavId', 0);

		$paymentSpeditions                           = $this->getPaymentsSpeditionsCombinations();
		$this->template->paymentSpeditionsStructured = $paymentSpeditions['structured'];
		$this->template->paymentSpeditionsReversed   = $paymentSpeditions['reversed'];

		$this->template->orderData = $this->sessionSection->orderFormData;

		$this->template->render();
	}

	public function attached($presenter): void
	{
		parent::attached($presenter);

		$sessionData = $this->sessionSection->orderFormData;

		$user = $this->getPresenter()->getUser();
		if ($user->isLoggedIn()) {
			$customer = $this->customersService->getByUser($user->getIdentity());

			if ($customer) {
				$deliAddr = $customer->getAddressDelivery();

				if ($customer && $deliAddr && is_null($sessionData['useAddrDeli']) && $deliAddr->getFirstName() && $deliAddr->getLastName()) {
					$sessionData['useAddrDeli']          = true;
					$this->sessionSection->orderFormData = $sessionData;
				}
			}
		}

		if ($sessionData) {
			$this->setOrderData($sessionData);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleSaveStep2()
	{
		$request = $this->getPresenter()->getHttpRequest();
		$data    = $this->sessionSection->orderFormData;

		$data['speditionCountry'] = $request->getPost('speditionCountry');
		foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v)
			$data[$k] = (int) $request->getPost($v)[$data['speditionCountry']];

		$this->eventDispatcher->dispatch(new SaveOrderFormDataEvent($data, $this['form']), 'eshopOrders.saveOrderFormDataStep2');
		$this->sessionSection->orderFormData = $data;

		if ($data['disableDeliveryAddressSpedition'] !== null) {
			$this['form']['country2']->setValue($data['speditionCountry']);
		}

		$currencyChange = null;
		if (isset($data['payment'])) {
			$payments = $this->getPayments();
			$payment  = $payments[$data['speditionCountry']][$data['payment']] ?? null;

			if ($payment && $payment->currency && $this->currencies->getCurrent()->getCode() != $payment->currency) {
				$curr = $this->currencies->getActive()[$payment->currency] ?? null;

				if ($curr)
					$currencyChange = $curr;
			}
		}

		if (!$currencyChange && isset($data['payment'], $data['spedition'])) {
			foreach ($this->getPaymentSpeditions() as $ps) {
				if ($this->currencies->getCurrent()->getCode() != $ps->currency
					&& array_key_exists((string) $data['speditionCountry'], $ps->getCountries())
					&& $ps->getSpedition()->getId() === $data['spedition'] && $ps->getPayment()->getId() === $data['payment']) {
					$curr = $this->currencies->getActive()[$ps->currency] ?? null;

					if ($curr)
						$currencyChange = $curr;
					break;
				}
			}
		}

		if ($currencyChange) {
			$this->currencies->setCurrentCurrency($currencyChange->getId());
			$this->getPresenter()->flashMessageInfo($this->t('eshopOrdersFront.orderPage.currencyPaymentChangeTo', ['curr' => $currencyChange->getCode()]));
			$this->getPresenter()->redirect('this');
		}

		$this->redrawControl('step3');
		$this->redrawControl('orderCartDetail');
		$this->redrawControl('orderSummary');
	}

	public function handleSaveStep3()
	{
		$request    = $this->getPresenter()->getHttpRequest();
		$data       = $this->sessionSection->orderFormData;
		$form       = $this['form'];
		$formValues = $form->getValues();

		$data['companyOrder'] = $formValues->companyOrder;
		foreach (self::PERSONAL_COLUMNS as $k) {
			$data[$k]       = $request->getPost($k);
			$data[$k . '2'] = $request->getPost($k . '2');
		}

		$countries            = $this->countriesService->getAllNameColumn();
		$data['countryText']  = $countries[$request->getPost('country')] ?? null;
		$data['country2Text'] = $countries[$request->getPost('country2')] ?? null;

		foreach (['useAddrDeli', 'speditionCountry'] as $k) {
			$data[$k] = $request->getPost($k);
		}

		$speditionCountry = $data['speditionCountry'];
		if ($data['useAddrDeli'] === 'on' || $data['useAddrDeli'] === true) {
			$data['useAddrDeli'] = true;

			if ($data['disableDeliveryAddress'] !== true) {
				foreach (['firstName2', 'lastName2', 'email2', 'phone2', 'street2', 'city2', 'postal2'] as $col) {
					if (!$data[$col])
						$form[$col]->addError('default.formMessages.filled');
				}

				if (!Validators::isEmail((string) $data['email2']))
					$form['email2']->addError('default.formMessages.email');

				if (strlen($data['phone2']) < 9)
					$form['phone2']->addError('default.formMessages.phone_min');
				else if (!PhoneInput::validatePhone($form['phone2']))
					$form['phone2']->addError(PhoneInput::$phoneMessages);

				$postal2 = str_replace(['-', ' ', '_'], '', $data['postal2']);
				if (strlen($postal2) !== 5)
					$form['postal2']->addError('eshopOrdersFront.orderPage.badPsc');
				elseif (strtolower($data['country2']) == 'cz' && !FormValidator::validateFormInput($form['postal2'], []))
					$form['postal2']->addError('eshopOrdersFront.orderPage.badPsc');

				if ($data['speditionCountry'] !== $data['country2'])
					$form['country2']->addError($this->t('eshopOrdersFront.orderPage.badSpeditionCountry'), false);
			}
		} else {
			$data['useAddrDeli'] = false;
			if ($data['country'] != $speditionCountry)
				$this['form']->addError($this->t('eshopOrdersFront.orderPage.badSpeditionCountry'), false);
		}

		if ($this->isIcoRequired() && $formValues['companyOrder'] === false) {
			$this['form']->addError('eshopOrdersFront.orderPage.companyOrderIsRequired');
		}

		$this->eventDispatcher->dispatch(new SaveOrderFormDataEvent($data, $this['form']), 'eshopOrders.saveOrderFormDataStep3');
		$this['form']['validatePersonal']->validate();
		$this->sessionSection->orderFormData = $data;

		if ($this['form']->hasErrors()) {
			$fieldErrors = [];
			foreach (self::PERSONAL_COLUMNS as $col) {
				foreach (['', '2'] as $num) {
					if (isset($form[$col . $num]) && $form[$col . $num]->hasErrors())
						$fieldErrors[$col . $num] = $form[$col . $num]->getErrors();
				}
			}

			$this->getPresenter()->payload->fieldErrors      = $fieldErrors;
			$this->getPresenter()->payload->orderFormErrors  = $this['form']->getErrors();
			$this->getPresenter()->payload->orderFormIsValid = false;
		}

		$this->redrawControl('step3');
		$this->redrawControl('orderSummary');
	}

	public function isIcoRequired(): bool
	{
		if (EshopOrdersConfig::load('orderForm.icoIsRequired', false))
			return true;

		$customer = $this->userStorage->getIdentity() ? $this->customersService->getByUser($this->userStorage->getIdentity()) : null;

		if ($customer && $customer->getGroupCustomers()) {
			$requireIco = $customer->getGroupCustomers()->getParams()['requireIco'] ?? false;

			return EshopOrdersConfig::load('customerGroup.enableRequireIco') && $requireIco;
		}

		return false;
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */

	/**
	 * @param bool $allow
	 */
	public function allowNewsletterFlag(bool $allow = true): void { $this->allowNewsletterFlag = $allow; }

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{

		$form = $this->createForm();
		$form->setAjax();
		$cart      = $this->cartFacade->getCart();
		$countries = $this->countriesService->getAllNameColumn();

		$currency = $this->currencies->getCurrent();
		$form->addHidden('currency', $currency->getCode());
		$form->addGroup('speditions');

		$defaultCountry = key($this->getCountries());
		if (strtolower((string) $currency->getCode()) === 'eur') {
			// Nekdy neni klic lowecase
			foreach ($this->getCountries() as $k => $v) {
				if (strtolower($k) === 'sk' && isset($this->getCountries()[$k])) {
					$defaultCountry = $k;
					break;
				}
			}
		}

		$form->addSelect('speditionCountry', 'eshopOrders.orderPage.country', $this->getCountries())
			->setTranslator(null)
			->setDefaultValue($defaultCountry)
			->setRequired('eshopOrdersFront.orderPage.requireSpeditionCountry');

		$speditionsContainer = $form->addContainer('speditions');
		$paymentsContainer   = $form->addContainer('payments');
		$paySpedCombinations = $this->getPaymentsSpeditionsCombinations();

		foreach ($this->getSpeditions() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['structured'][$country][$row->getId()]))
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format((float) $row->getPrice());
			}

			$speditionsContainer->addRadioList($country, 'eshopOrders.orderPage.spedition', $list)
				->addConditionOn($form['speditionCountry'], BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED, 'eshopOrdersFront.orderPage.requireSpedition');
		}

		foreach ($this->getPayments() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['reversed'][$country][$row->getId()]))
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format((float) $row->getPrice());
			}

			$paymentsContainer->addRadioList($country, 'eshopOrders.orderPage.payment', $list)
				->addConditionOn($form['speditionCountry'], BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED, 'eshopOrdersFront.orderPage.requirePayment');
		}

		$form->addSubmit('validateSpedition', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form['speditionCountry'],
				$form['payments'],
				$form['speditions'],
			])->onClick[] = [$this, 'handleSaveStep2'];

		$form->addGroup('personal');
		$form->addText('firstName', 'eshopOrders.orderPage.name')->setRequired();
		$form->addText('lastName', 'eshopOrders.orderPage.lastName')->setRequired();
		$form->addPhone('phone', 'eshopOrders.orderPage.phone')
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addEmail('email', 'eshopOrders.orderPage.email')->setRequired();
		$form->addText('street', 'eshopOrders.orderPage.street')->setRequired();
		$form->addText('city', 'eshopOrders.orderPage.city')->setRequired();
		$form->addText('postal', 'eshopOrders.orderPage.postal')
			->setHtmlType('number')
			->setMaxLength(5)
			->addRule(BaseForm::MIN_LENGTH, null, 5)
			->addRule(BaseForm::MAX_LENGTH, null, 5)
			->setRequired();
		$form->addSelect('country', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);
		$form->addCheckbox('companyOrder', 'eshopOrdersForm.orderPage.companyOrder');
		$form->addText('company', 'eshopOrders.orderPage.company')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');

		if ($this->isIcoRequired())
			$form['companyOrder']->setDefaultValue(true)->setRequired();


		if (EshopOrdersConfig::load('orderForm.validatePsc')) {
			$form['postal']
				->setHtmlAttribute('data-autocomplete-url', EshopOrdersConfig::load('api.pscUrl'))
				->setHtmlAttribute('data-autocomplete-target', $form['postal']->getHtmlId())
				->setHtmlAttribute('data-autocomplete-key', 'psc')
				->setHtmlAttribute('data-result-item-data', 'psc');
			$form['postal']
				->addConditionOn($form['country'], BaseForm::EQUAL, 'cz')
				->addRule(FormValidator::class . '::validateFormInput', 'eshopOrdersFront.orderPage.badPsc');
		}

		if (strtolower((string) $currency->getCode()) === 'eur') {
			$form['phone']->setDefaultValue('+421');
			foreach ($this->getCountries() as $k => $v) {
				if (strtolower($k) === 'sk' && isset($countries[$k])) {
					$form['country']->setDefaultValue($k);
					break;
				}
			}
		}

		//fakturacni adresa
		$useAddrDeli = $form->addCheckbox('useAddrDeli', 'eshopOrders.orderPage.useAddrDeli');
		$form->addText('company2', 'eshopOrders.orderPage.company');
		$form->addText('firstName2', 'eshopOrders.orderPage.name');
		$form->addText('lastName2', 'eshopOrders.orderPage.lastName');
		$form->addText('email2', 'eshopOrders.orderPage.email');
		$form->addPhone('phone2', 'eshopOrders.orderPage.phone');
		$form->addText('street2', 'eshopOrders.orderPage.street');
		$form->addText('city2', 'eshopOrders.orderPage.city');
		$form->addText('postal2', 'eshopOrders.orderPage.postal');

		$form->addSelect('country2', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);

		if (EshopOrdersConfig::load('orderForm.validatePsc')) {
			$form['postal2']
				->setHtmlAttribute('data-autocomplete-url', EshopOrdersConfig::load('api.pscUrl'))
				->setHtmlAttribute('data-autocomplete-target', $form['postal2']->getHtmlId())
				->setHtmlAttribute('data-autocomplete-key', 'psc')
				->setHtmlAttribute('data-result-item-data', 'psc');
		}


		$form['country']->addRule(function($item, $args) {
			$data   = $this->sessionSection->orderFormData;
			$values = $args->getValues();

			if ($values->useAddrDeli == false)
				return true;
			else if ($values->useAddrDeli == true)
				return true;
			else
				return $values['speditionCountry'] === $item->getValue();
		}, 'eshopOrdersFront.orderPage.badSpeditionCountry', $form);

		$validateFields = [];
		foreach (self::PERSONAL_COLUMNS as $v) {
			$validateFields[] = $form[$v];

			if (!in_array($v, ['idNumber', 'vatNumber']))
				$validateFields[] = $form[$v . '2'];
		}

		$form->addSubmit('validatePersonal', 'eshopOrders.orderPage.nextStep')
			->setValidationScope($validateFields)
			->onClick[] = [$this, 'handleSaveStep3'];

		$form->addTextArea('messageAlt', 'eshopOrders.orderPage.message')->setAttribute('class', ['messageInput']);
		$form->addTextArea('message', 'eshopOrders.orderPage.message');

		if (EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false)) {
			$form->addCheckbox('agreedOne', 'eshopOrders.orderPage.agreedTerms')
				->setDefaultValue(1);
		} else {
			$form->addCheckbox('agreedTerms', 'eshopOrders.orderPage.agreedTerms')->setRequired('eshopOrders.orderPage.agreedTermsRequired');
			$form->addCheckbox('agreedQuestioning', 'eshopOrders.orderPage.agreedQuestioning')->setDefaultValue(1);
		}

		if ($this->allowNewsletterFlag)
			$form->addCheckbox('agreedNewsletter', 'eshopOrdersFront.orderPage.agreedNewsletter')
				->setDefaultValue(1);

		$form->addSubmit('submit', 'eshopOrders.orderPage.send')
			->setHtmlAttribute('class', ['submitButton', 'eshopOrdersNext4'])
			->setHtmlId('orderFormSubmit');

		if ($this->userStorage->getIdentity()) {
			$customer = $this->customersService->getByUser($this->userStorage->getIdentity());

			if ($customer) {
				$addrDeli = $customer->getAddressDelivery();
				$addrInvo = $customer->getAddressInvoice();

				if ($addrInvo) {
					$form->setDefaults($addrInvo->toArray());

					if ($addrInvo->getIdNumber()) {
						$form->setDefaults([
							'companyOrder' => true,
							'company'      => $addrInvo->getCompany(),
							'idNumber'     => $addrInvo->getIdNumber(),
							'vatNumber'    => $addrInvo->getVatNumber(),
						]);

						$this->template->companyFilled = true;
					}
				}

				if ($addrDeli) {
					$def = [];
					foreach ($addrDeli->toArray() as $k => $v)
						$def[$k . '2'] = $v;
					$form->setDefaults($def);
				}
			}
		}

		if ($this->isIcoRequired()) {
			$form['companyOrder']->setValue(true);
			$form['idNumber']->setRequired();

			$this->template->companyFilled = true;
		}

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		$orderFormEvent           = new OrderFormEvent($form);
		$orderFormEvent->template = $this->template;
		$this->eventDispatcher->dispatch($orderFormEvent, 'eshopOrders.createOrderForm');

		return $form;
	}

	public function setOrderData(array $orderData): void
	{
		$this->eventDispatcher->dispatch(new SetOrderFormDataEvent($orderData, $this['form']), 'eshopOrders.setOrderFormData');

		$form = $this['form'];
		$d    = [
			'messageAlt'       => $orderData['messageAlt'],
			'message'          => $orderData['message'],
			'useAddrDeli'      => $orderData['useAddrDeli'],
			'agreedNewsletter' => $orderData['agreedNewsletter'],
		];

		if (EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false)) {
			$d['agreedOne'] = $orderData['agreedOne'];
		} else {
			$d['agreedTerms']       = $orderData['agreedTerms'];
			$d['agreedQuestioning'] = $orderData['agreedQuestioning'];
		}

		$spedCountry = $orderData['speditionCountry'];
		if ($spedCountry) {
			if (array_key_exists($spedCountry, $form['speditionCountry']->getItems()))
				$d['speditionCountry'] = $spedCountry;

			foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v)
				if ($orderData[$k] && isset($form[$v][$spedCountry]) && array_key_exists($orderData[$k], $form[$v][$spedCountry]->getItems()))
					$d[$k][$spedCountry] = $orderData[$k];
		}

		foreach (self::PERSONAL_COLUMNS as $k) {
			if ($orderData[$k])
				$d[$k] = $orderData[$k];

			if (!in_array($k, ['idNumber', 'vatNumber']) && $orderData[$k . '2'])
				$d[$k . '2'] = $orderData[$k . '2'];
		}

		$form->setDefaults($d);
	}

	public function formOnValidate(BaseForm $form, $a): bool
	{
		$values                              = $form->getValues();
		$cart                                = $this->cartFacade->getCart();
		$cartItems                           = $cart->getCartItems();
		$this->template->formSummaryMessages = [];

		if (empty($cartItems) && EshopOrdersConfig::load('orderForm.enableEmptyCartValidation', true)) {
			$form->addError('eshopOrders.orderPage.emptyCart');
			$this->template->formSummaryMessages = ['eshopOrders.orderPage.emptyCartData'];
		}

		$spedCountry = $values->speditionCountry;
		$spedition   = $values->speditions[$spedCountry] ?? null;
		$payment     = $values->payments[$spedCountry] ?? null;

		$paymentsSpeditions = $this->getPaymentsSpeditionsCombinations();
		$paySped            = $paymentsSpeditions['structured'][$spedCountry][$spedition] ?? null;
		if (!$paySped || !in_array($payment, $paySped)) {
			$form->addError('eshopOrdersFront.orderPage.invalidPaymentSpeditionCombination');
			Debugger::log('payment: ' . $payment . ', spedition: ' . $spedition . ', coutry: ' . $spedCountry, 'paymentSpeditionCombination');
		}

		$remainingValue = $cart->getRemainingValueToCompleteOrder();
		if ($remainingValue > 0) {
			$form->addError(Html::el()
				->setHtml($this->t('eshopOrdersFront.orderPage.youNeedMoreToCompleteOrder', ['price' => $this->priceFilter->format($remainingValue)])),
				false,
			);
		}

		$this->eventDispatcher->dispatch(new OrderFormEvent($form), 'eshopOrders.orderFormValidate');

		if (!empty($form->getErrors()))
			Debugger::log($form->getErrors(), 'orderFormValidate');

		if ($form->hasErrors()) {
			$errs = [];
			foreach ($form->getComponents(true) as $c) {
				if ($c->getErrors())
					$errs[$c->getHtmlId()] = $c->getErrors();
			}

			$this->getPresenter()->payload->orderFormErrors  = $errs;
			$this->getPresenter()->payload->orderFormIsValid = false;

			foreach ($form->getErrors() as $e)
				$this->template->formSummaryMessages[] = $e;
			$this->redrawControl('formMessages');

			$this->getPresenter()->redrawControl('cartMessage');
			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');

			return false;
		} else {
			$this->getPresenter()->payload->orderFormIsValid = true;

			$this->getPresenter()->redrawControl('cartMessage');
			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');
		}

		return true;
		// nebude potreba take validovat jestli doprava a platba jsou aktivni a plati pro cenu kosiku?
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values)
	{
		if (!$form['submit']->isSubmittedBy()) {
			return;
		}

		try {
			/** @var Site $currentSite */
			$currentSite = $this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent());
			$order       = new Order($currentSite);
			$order->lang = $this->translator->getLocale();

			if ($values->speditionCountry && isset(EshopOrdersConfig::load('order.countryLangs')[$values->speditionCountry])) {
				$order->lang = EshopOrdersConfig::load('order.countryLangs')[$values->speditionCountry];
			}

			$order->setMessage($values->message ?: $values->messageAlt);
			$order->setAgreedTerms($values->agreedTerms || $values->agreedOne);

			if ($values->agreedQuestioning || ($values->agreedOne || EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false))) {
				$orderFlagQuestioning = new OrderFlag(OrderFlag::TYPE_QUESTIONING, true, $order);
				$this->em->persist($orderFlagQuestioning);
				$order->addFlag($orderFlagQuestioning);
			}

			if ($values->agreedNewsletter) {
				$orderFlagNewsletter = new OrderFlag(OrderFlag::TYPE_NEWSLETTER, true, $order);
				$this->em->persist($orderFlagNewsletter);
				$order->addFlag($orderFlagNewsletter);
			}

			$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
			$addrInv->setFirstName($values->firstName);
			$addrInv->setLastName($values->lastName);
			$addrInv->setEmail($values->email);
			$addrInv->setPhone($values->phone);
			$addrInv->setStreet($values->street);
			$addrInv->setCity($values->city);
			$addrInv->setPostal($values->postal);
			if ($values->country) {
				$addrInv->setCountry($this->countriesService->getReference($values->country));
			}

			if ($values->companyOrder) {
				$addrInv->setCompany($values->company);
				$addrInv->setIdNumber($values->idNumber);
				$addrInv->setVatNumber($values->vatNumber);
			}

			$this->em->persist($addrInv);

			$order->setAddressInvoice($addrInv);

			$addrDeli = null;
			if ($values->useAddrDeli) {
				$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
				$addrDeli->setFirstName($values->firstName2);
				$addrDeli->setLastName($values->lastName2);
				$addrDeli->setEmail($values->email2);
				$addrDeli->setPhone($values->phone2);
				$addrDeli->setStreet($values->street2);
				$addrDeli->setCity($values->city2);
				$addrDeli->setPostal($values->postal2);
				if ($values->country2)
					$addrDeli->setCountry($this->countriesService->getReference($values->country2));
				$addrDeli->setCompany($values->company2);
				$this->em->persist($addrDeli);

				$order->setAddressDelivery($addrDeli);
			}

			$customer = $this->getOrCreateCustomer(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName(),
				$order->getAddressInvoice()->getLastName(),
				$order->getAddressInvoice()->getPhone());

			if ($customer) {
				$order->setCustomer($customer);
				$customerDeli = $customer->getAddressDelivery();
				$customerInvo = $customer->getAddressInvoice();

				if ($addrInv) {
					if (!$customerInvo)
						$customerInvo = new CustomerAddress($customer);
					$customerInvo->fillFromOrderAddress($addrInv);
					$customer->setAddressInvoice($customerInvo);
					$this->em->persist($customerInvo);
				}

				if ($addrDeli) {
					if (!$customerDeli)
						$customerDeli = new CustomerAddress($customer);
					$customerDeli->fillFromOrderAddress($addrDeli);
					$customer->setAddressDelivery($customerDeli);
					$this->em->persist($customerDeli);
				}

				if (EshopOrdersConfig::load('order.autoValidateIfUserIsValid')) {
					if (EshopOrdersConfig::load('order.allowValidateIdNumber')) {
						if ($addrInv && $addrInv->getIdNumber())
							$addrInv->validatedIdNumber = $customer->getAddressInvoice()->validatedIdNumber;
						if ($addrDeli && $addrDeli->getIdNumber())
							$addrDeli->validatedIdNumber = $customer->getAddressDelivery()->validatedIdNumber;
					}
					if (EshopOrdersConfig::load('order.allowValidateVatNumber')) {
						if ($addrInv && $addrInv->getVatNumber())
							$addrInv->validatedVatNumber = $customer->getAddressInvoice()->validatedVatNumber;
						if ($addrDeli && $addrDeli->getVatNumber())
							$addrDeli->validatedVatNumber = $customer->getAddressDelivery()->validatedVatNumber;
					}

					if ($addrInv)
						$this->em->persist($addrInv);
					if ($addrDeli)
						$this->em->persist($addrDeli);
				}
			}

			$cart       = $this->cartFacade->getCart();
			$cartItems  = $cart->getCartItems();
			$orderItems = $this->ordersService->fillOrderItems($cartItems, $order);
			$order->setOrderItems($orderItems);

			$orderPayment = new OrderPayment($this->paymentsService->getReference((int) $values->payments[$values->speditionCountry]), $order);
			$payment      = $this->paymentsService->get((int) $values->payments[$values->speditionCountry]);
			$orderPayment->setPrice($payment->getPriceActual($cart, true));
			$order->setPayment($orderPayment);

			$orderSpedition = new OrderSpedition($this->speditionsService->getReference((int) $values->speditions[$values->speditionCountry]), $order);
			$spedition      = $this->speditionsService->get((int) $values->speditions[$values->speditionCountry]);
			$orderSpedition->setPrice($spedition->getPriceActual($cart, true));
			$order->setSpedition($orderSpedition);

			$statusCreated      = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);
			$order->getOrderStatuses()->add($orderActionCreated);

			$cartOrderGift = $cart->getOrderGift();
			if ($cartOrderGift) {
				$orderGift        = new OrderGift($order, $this->em->getReference(Product::class, $cartOrderGift->productId), $cartOrderGift->name);
				$orderGift->ean   = $cartOrderGift->ean;
				$orderGift->code1 = $cartOrderGift->code1;

				$this->em->persist($orderGift);
			}

			//			foreach ($cart->getDiscounts() as $k => $v) {
			//				$discount = new OrderDiscount($v->id, $order);
			//				$discount->setValue($v->amount);
			//				$discount->setType($v->type);
			//				$discount->setPrice($cart->calculateDiscount($v));
			//				$discount->setName($this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol);
			//				$this->em->persist($discount);
			//				$order->addOrderDiscount($discount);
			//			}

			$this->em->getConnection()->beginTransaction();
			try {
				$eventOrder                 = new OrderEvent($order);
				$eventOrder->addrDelivery   = $addrDeli;
				$eventOrder->addrInvoice    = $addrInv;
				$eventOrder->orderSpedition = $orderSpedition;
				$eventOrder->orderPayment   = $orderPayment;
				$eventOrder->orderItems     = $orderItems;
				$eventOrder->formData       = $values;
				$this->eventDispatcher->dispatch($eventOrder, 'eshopOrders.orderBeforeSave');
				//ulozit objednavku
				$this->em->persist($order);
				//ulozit k ni adresy
				$this->em->persist($addrInv);
				if ($addrDeli)
					$this->em->persist($addrDeli);
				//polozky doprava a platba
				$this->em->persist($orderSpedition);
				$this->em->persist($orderPayment);
				//ulozit k ni polozky
				foreach ($orderItems as $oi) {
					$this->em->persist($oi);
				}
				//akce - datum kdy byla objednavka vytvorena
				$this->em->persist($orderActionCreated);
				//pripadny slevovy kod
				if (isset($discount) && $discount)
					$this->em->persist($discount);
				//pripadne nastavene priznaky
				if ($orderFlagQuestioning)
					$this->em->persist($orderFlagQuestioning);
				// vytvoreni faktury
				if (isset($invoice) && $invoice)
					$this->em->persist($invoice);
				//teprve po ulozeni vseho flush (aby byly spravne propojene cizi klice)
				$this->em->flush();
				$this->em->getConnection()->commit();
				//TODO nejake promazavani kosiku z DB, kdyz uz k nim neni aktivni session... nebo po nejakem timeoutu (tyden a pod)
			} catch (Exception $e) {
				$this->em->getConnection()->rollBack();
				Debugger::log($e->getMessage(), ILogger::ERROR);
				Debugger::log($e, ILogger::ERROR);
				throw $e;
			}

			//dale uz pracujeme s objednavkou nactenou z DB
			$orderDetail = $this->ordersService->get($order->getId());
			unset($order);
			$this->cartFacade->getCartsService()->deleteCurrentCart();
			$this->sessionSection->orderFormData = [];

			// Spuštění události po dokončení objednávky jako odeslání emailu, zboží,...
			$this->eventDispatcher->dispatch(new OrderEvent($orderDetail), 'eshopOrders.orderOnSuccess');
			// Vytáhnutí zpráv z událostí
			$messages = $this->session->getSection('eshopOrders/orderSuccessMessages');

			if ($messages->success)
				foreach ($messages->success as $v)
					$this->getPresenter()->flashMessageSuccess($v);
			if ($messages->error)
				foreach ($messages->error as $v)
					$this->getPresenter()->flashMessageDanger($v);
			$messages->remove();
		} catch (\Exception $e) {
			Debugger::log($e, 'exception');
			$form->addError($e->getMessage());

			return false;
		}

		$this->getPresenter()->redirect(':EshopOrders:Front:Finished:orderFinished', ['orderIdent' => $orderDetail->getIdent()]);

		return true;
	}

	protected function getOrCreateCustomer($email, $firstName = '', $lastName = '', $phone = '')
	{
		$userId = null;
		if ($this->getPresenter()->getUser()->isLoggedIn()) {
			$userId = $this->getPresenter()->getUser()->id;
		}
		$customer = $this->customersService->getOrCreateCustomer($userId, $email, $firstName, $lastName, $phone);

		return $customer;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderCartDetail()
	{
		$control = $this->orderCartDetailFactory->create();

		return $control;
	}

	protected function createComponentPaySpedSummary()
	{
		$control = $this->paySpedSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData['spedition'], $orderData['payment'], $orderData['priceTotal']);

		return $control;
	}

	protected function createComponentOrderSummary()
	{
		$control = $this->orderSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData);

		return $control;
	}

	/**
	 * @return Payment[]
	 */
	public function getSpeditions(): array
	{
		if ($this->speditions === null) {
			/** @var Spedition[] $raw */
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getSpedition()->getId()] = $ps->getSpedition()->getId();
				}

				$raw[$ps->getSpedition()->getId()] = $ps->getSpedition();
			}
			$raw = $this->speditionsService->filterByCart($raw, $this->cart);

			$this->speditions = $this->processPaySpedData($byCountry, $raw);
		}

		return $this->speditions;
	}

	public function getPaymentsSpeditionsCombinations(): array
	{
		if ($this->paymentsSpeditions === null) {
			$filteredSpedition = $this->getSpeditions();
			$filteredPayments  = $this->getPayments();

			$paymentSpeditions        = $this->getPaymentSpeditions();
			$this->paymentsSpeditions = [
				'structured' => [],
				'reversed'   => [],
			];

			foreach ($paymentSpeditions as $key => $item) {
				foreach ($item->getCountries() as $country) {
					/** @var Country $country */
					$spedId    = $item->getSpedition()->getId();
					$payId     = $item->getPayment()->getId();
					$countryId = $country->getId();

					if (!isset($filteredPayments[$countryId][$payId]) || !isset($filteredSpedition[$countryId][$spedId]))
						continue;

					$this->paymentsSpeditions['structured'][$countryId][$spedId][] = $payId;
					$this->paymentsSpeditions['reversed'][$countryId][$payId][]    = $spedId;
				}
			}
		}

		return $this->paymentsSpeditions;
	}

	/**
	 * @return Payment[][]
	 */
	public function getPayments(): array
	{
		if ($this->payments === null) {
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getPayment()->getId()] = $ps->getPayment()->getId();
				}

				$raw[$ps->getPayment()->getId()] = $ps->getPayment();
			}
			$raw = $this->paymentsService->filterByCartValue($raw, $this->cart->getCartItemsPrice());

			$this->payments = $this->processPaySpedData($byCountry, $raw);
		}

		return $this->payments;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getPaymentSpeditions(): array
	{
		if ($this->paymentSpeditions === null)
			$this->paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();

		return $this->paymentSpeditions;
	}

	public function getCountries(): array
	{
		if ($this->cCountries === null) {
			$this->cCountries = [];

			foreach ($this->getPaymentSpeditions() as $row)
				foreach ($row->getCountries() as $id => $country)
					$this->cCountries[$country->getId()] = $country->getName();
		}

		return $this->cCountries;
	}

	protected function processPaySpedData(array $byCountry, array $raw)
	{
		$result = [];

		foreach ($byCountry as $country => $rows) {
			foreach ($rows as $row) {
				if (!isset($raw[$row]))
					continue;

				$raw[$row]->country     = $country;
				$result[$country][$row] = $raw[$row];
			}
		}

		return $result;
	}
}
