<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\BotDetect;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Module;
use Currency\Model\Exchange;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\PersistentCollection;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\Event\FillDaoEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\Model\Entities\Cart;
use EshopOrders\Model\Entities\CartItem;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopCatalog\FrontModule\Model\Tags;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\SmartObject;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use Nette\Utils\DateTime;
use Tracy\Debugger;
use Users\Model\Security\User;

/**
 * class Carts
 * @package EshopOrders\Model
 *
 */
class Carts extends BaseFrontEntityService
{
	use SmartObject;

	/** @var SessionSection */
	private $sessionSection;

	protected $entityClass = Cart::class;

	/** @var Dao\Cart[]|null */
	protected $cartsCached;

	/** @var ProductsFacade */
	private $productsFacadeService;

	/** @var Tags */
	protected $tagsService;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	protected Customers $customers;

	protected Exchange $exchange;

	protected User $user;

	/** @var Dao\CartItem[] */
	public $cDaoItems;

	public function __construct(Session $session, ProductsFacade $productsFacade, Tags $tags, EventDispatcher $eventDispatcher,
	                            Customers $customers, Exchange $exchange, User $user)
	{
		$this->sessionSection        = $session->getSection('eshopOrdersCart');
		$this->productsFacadeService = $productsFacade;
		$this->tagsService           = $tags;
		$this->eventDispatcher       = $eventDispatcher;
		$this->customers             = $customers;
		$this->exchange              = $exchange;
		$this->user                  = $user;

		$this->sessionSection->setExpiration('1 month');
	}

	/**
	 *
	 * @param int $itemId
	 *
	 * @return Dao\CartItem|null
	 */
	public function getCartItem($itemId): ?Dao\CartItem
	{
		return $this->getCurrentCart()->getCartItems()[$itemId] ?? null;
	}

	public function getCurrentCartId(): int
	{
		if (BotDetect::isBot())
			return 0;

		$cartId = $this->sessionSection->cartId;

		return $cartId ?: 0;
	}

	/** Vrati soucasny kosik podle ID ze session. Kdyz kosik neexistuje, vytvori novy
	 * @return Dao\Cart
	 */
	public function getCurrentCart()
	{
		if (BotDetect::isBot())
			return new Dao\Cart();

		$cartId = $this->getCurrentCartId();

		if (is_null($cartId) || $cartId === 0) {
			return new Dao\Cart();
		} else {
			if (isset($this->cartsCached[$cartId]))
				return $this->cartsCached[$cartId];

			$cartRaw = $this->loadRawCart($cartId);

			$dateTime = (new DateTime())->modify('-1 hour');
			if ($cartRaw->lastActivity instanceof \DateTimeInterface && $cartRaw->lastActivity->getTimestamp() <= $dateTime->getTimestamp()) {
				$this->em->getConnection()->executeStatement("UPDATE eshop_orders__cart SET `last_activity` = ? WHERE `id` = ?", [
					(new DateTime())->format('Y-m-d H:i:s'),
					$cartId,
				]);
			}
		}

		$cartId                     = $cartRaw->getId();
		$this->cartsCached[$cartId] = $this->fillDao($cartRaw);

		$cart = $this->cartsCached[$cartId] ?? null;

		if ($cart) {
			foreach ($cart->getCartItems() as &$item) {
				$q = $this->checkProductQuantity($item->getProductId(), $item->getQuantity());

				if ($q <= 0)
					$this->removeItem($item->getId());
				else
					$item->setQuantity($q);

				foreach ($item->getChilds() as $k => $child) {
					if (!$child->getProductId())
						continue;
					$q = $this->checkProductQuantity($child->getProductId(), $child->getQuantity());

					if ($q <= 0)
						$item->removeChild($k);
				}
			}

			return $cart;
		}

		return null;
	}

	/** smaze kosik pro daneho uzivatele - po dokonceni objednavky
	 * pak, pri volani getCurrentCart se vygeneruje novy
	 */
	public function deleteCurrentCart()
	{
		$oldCartId = $this->sessionSection->cartId;
		$this->removeRawCart($oldCartId);
		$this->sessionSection->cartId = null;
		unset($this->cartsCached[$oldCartId]);
	}

	public function getCurrentCartItemsPrice(): float
	{
		$id = $this->getCurrentCartId();
		if (!$id)
			return 0;

		$rawCart = $this->loadRawCart($id);

		return $this->getDaoCart($rawCart)->getCartItemsPrice();
	}

	private function getDaoCart(Cart $rawCart): Dao\Cart
	{
		$cart = new Dao\Cart();
		$cart
			->setId($rawCart->getId())
			->setIdent($rawCart->ident)
			->setCartItems($this->fillDaoItems($rawCart->getCartItems()));

		$cart->orderGiftId = $rawCart->orderGiftId;

		if ($this->user->isLoggedIn() && Strings::startsWith(Module::$currentPresenterName, 'EshopOrders')) {
			$customer = $this->customers->getByUser($this->user->getIdentity());

			if ($customer && $customer->hasMinimalOrderPrice() > 0) {
				$cart->minimalOrderPrice = $this->exchange->change($customer->hasMinimalOrderPrice());
			}
		}

		return $cart;
	}

	/** nacte kosik z databaze do vlastni property
	 *
	 * @param int $cartId
	 *
	 * @return Cart
	 */
	private function loadRawCart($cartId): Cart
	{
		$cartRawQuery = $this->getEr()->createQueryBuilder('c', 'c.id')
			->addSelect('ci')
			->leftJoin('c.cartItems', 'ci')
			->andWhere('c.id = :id')
			->setParameter('id', $cartId);

		$cartRaw = $cartRawQuery->getQuery()->getOneOrNullResult();

		if (is_null($cartRaw)) {
			$cartRaw = $this->createRawCart();

			$this->sessionSection->cartId = $cartRaw->getId();
		};

		return $cartRaw;
	}

	/**
	 * @return Cart|null
	 */
	private function createRawCart()
	{
		if (BotDetect::isBot()) {
			return null;
		} else {
			$ident = substr(md5((string) rand()), 0, 7);
			$cart  = new Cart($ident);
			$this->em->persist($cart);
			$this->em->flush($cart);
			Debugger::log($ident . ' --- ' . $_SERVER['HTTP_USER_AGENT'], '_create-cart');
			Debugger::log($ident . ' --- ' . $_SERVER['REMOTE_ADDR'], '_create-cart');
			Debugger::log($_SERVER, '_create-cart');
		}

		return $cart;
	}

	/**
	 * @param $cartId id kosiku ke smazani
	 */
	private function removeRawCart($cartId)
	{
		$cartRaw = $this->getEr()->getReference($cartId);
		$this->em->remove($cartRaw);
		$this->em->flush();
	}

	/** nacte jednu polozku kosiku z databaze
	 *
	 * @param int $itemId
	 *
	 * @return CartItem
	 */
	private function loadRawCartItem($itemId)
	{
		$cartRawQuery = $this->em->getRepository(CartItem::class)->createQueryBuilder('ci', 'ci.id');
		$cartRawQuery->andWhere('ci.id = :item_id')->setParameter('item_id', $itemId);

		$cartItemRaw = $cartRawQuery->getQuery()->setMaxResults(1)->getOneOrNullResult();

		return $cartItemRaw;
	}

	public function addItem(AddedCartItem &$item)
	{
		if (BotDetect::isBot())
			return;

		$productId = (int) $item->productId;
		$quantity  = !$item->ignoreValidation ? $this->checkProductQuantity($productId, (int) $item->quantity) : (int) $item->quantity;
		$optionId  = $item->optionId;
		$itemIdent = $productId . '|' . $optionId; //pripadne dalsi vlastnosti s vlivem na cenu

		if ($item->parentId)
			$itemIdent = 'P' . $item->parentId . '|' . $itemIdent;

		if (isset($item->moreData['disableStacking']) && $item->moreData['disableStacking'] === true)
			$itemIdent .= '|tm' . time();

		$cartId = $this->getCurrentCartId();

		$cartRaw = $this->loadRawCart($cartId);

		$itemRaw = $cartRaw->getCartItems()->get($itemIdent);

		if ($itemRaw) {
			//kdyz polozka uz je v kosiku, jen zvysime pocet
			$itemRaw->quantity = $itemRaw->quantity + $quantity;
		} else {
			//kdyz zatim neexistuje, pridame ji do kosiku
			$itemRaw = new CartItem($itemIdent, $productId, $cartRaw, $quantity);
		}

		$itemRaw->quantity = $this->checkProductQuantity((int) $itemRaw->getProductId(), (int) $itemRaw->quantity);

		$itemRaw->setMoreData($item->moreData);
		if ($item->parentId)
			$itemRaw->setParent($this->em->getReference(CartItem::class, $item->parentId));

		$this->em->persist($itemRaw);
		$this->em->flush();

		$item->itemId = $itemRaw->getId();

		//vymazeme kosik z cache, aby se priste obnovil z DB.
		//Alternativni moznost - vzit soucasne DAO a pridat mu polozku
		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);
		//TODO vratit nejaky priznak, ze to probehlo v poradku?
	}

	public function updateItemQuantity($itemId, $quantity, array $moreData = [])
	{
		if ($quantity <= 0 || !$itemId) {
			return $this->removeItem($itemId);
		}
		$cartId = $this->getCurrentCartId();

		$itemRaw = $this->loadRawCartItem($itemId);

		if (!$itemRaw)
			return $this->removeItem($itemId);

		$itemRaw->quantity = $this->checkProductQuantity((int) $itemRaw->getProductId(), (int) $quantity);

		if (isset($moreData['note']))
			$itemRaw->setMoreDataValue('note', $moreData['note']);

		$this->em->persist($itemRaw);
		$this->em->flush();

		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]); //obnoveni cache
	}

	public function removeItem($itemId)
	{
		$cartId = $this->getCurrentCartId();

		$itemRaw = $this->loadRawCartItem($itemId);
		if ($itemRaw) {
			//			unset($this->cDaoItems[$itemId]);
			$this->em->remove($itemRaw);
			$this->em->flush();
		}

		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]); //obnoveni cache
	}

	public function setOrderGift(int $giftId)
	{
		$cartId = $this->getCurrentCartId();
		if (!$cartId)
			return;

		$cartRaw = $this->loadRawCart($cartId);

		$cartRaw->orderGiftId = $giftId;
		$this->em->persist($cartRaw);
		$this->em->flush();

		$cart              = $this->getCurrentCart();
		$cart->orderGiftId = $giftId;
	}

	public function clearCartItems()
	{
		$cartId = $this->getCurrentCartId();
		if (!$cartId)
			return;

		$cart = $this->loadRawCart($cartId);

		foreach ($cart->cartItems->toArray() as $item) {
			$this->em->remove($item);
		}

		$this->em->flush();
		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);
	}

	protected function checkProductQuantity(int $productId, int $quantity): int
	{
		if (EshopCatalogConfig::load('pseudoWarehouse')) {
			$product = $this->productsFacadeService->getProduct($productId);

			if ((int) $product->unlimitedQuantity === 0 && $quantity > $product->getQuantity() + $product->getQuantityExternal()) {
				$quantity = $product->getQuantity() + $product->getQuantityExternal();
			}
		} else {
			$product = $this->productsFacadeService->getProduct($productId);

			if (!$product->canAddToCart)
				$quantity = 0;
		}

		return $quantity;
	}

	/**
	 * @param Cart $cartRaw
	 *
	 * @return Dao\Cart
	 */
	protected function fillDao($cartRaw)
	{
		$cart = $this->getDaoCart($cartRaw);

		if ($cart->getCartItems())
			$this->eventDispatcher->dispatch(new FillDaoEvent($cart, $cartRaw), 'eshopOrders.cartFillDao');

		return $cart;
	}

	/**
	 * @param Entity\CartItem[] $cartItemsRaw
	 *
	 * @return ArrayCollection(Dao\CartItem) polozky
	 */
	protected function fillDaoItems($cartItemsRaw)
	{
		if ($cartItemsRaw instanceof ArrayCollection || $cartItemsRaw instanceof PersistentCollection)
			$cartItemsRaw = $cartItemsRaw->toArray();

		$this->eventDispatcher->dispatch(new FillDaoItemsEvent($cartItemsRaw), 'eshopOrders.cartFillDaoItems');

		return $this->cDaoItems;
	}

}

