<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\FrontModule\Model\Event\PaymentsSpeditionsEvent;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\EshopOrdersConfig;
use Users\Model\Security\User;

/**
 * Class Payments
 * @package EshopOrders\FrontModule\Model
 */
class Payments extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Payment::class;

	/** @var EventDispatcher */
	public EventDispatcher $eventDispatcher;

	/** @var Sites */
	protected Sites $sitesService;

	/** @var User */
	protected User $user;

	/** @var Customers */
	protected Customers $customers;

	/** @var Dao\Payment[] */
	protected $cPublished;

	public function __construct(EventDispatcher $eventDispatcher, Sites $sites, User $user, Customers $customers)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->sitesService    = $sites;
		$this->user            = $user;
		$this->customers       = $customers;
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Payment
	 */
	public function get($id)
	{
		return $this->getAllPublished()[$id];
	}

	/**
	 * @return Dao\Payment[]
	 */
	public function getAllPublished(): array
	{
		if ($this->cPublished === null) {
			$this->cPublished = [];
			$qb               = $this->getEr()->createQueryBuilder('p', 'p.id')
				->andWhere('p.isPublished = :isPublished')
				->innerJoin(PaymentSpedition::class, 'ps', Join::WITH, 'ps.payment = p.id')
				->setParameters([
					'isPublished' => 1,
				])
				->orderBy('p.position')
				->groupBy('p.id');

			if (count($this->sitesService->getSites()) > 1) {
				$qb->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());
			}

			$customer = $this->user->isLoggedIn() ? $this->customers->getByUserId($this->user->getId()) : null;
			if ($customer) {
				if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer', false) && $customer->disabledPayments->getKeys()) {
					$qb->andWhere('p.id NOT IN (:notIds)')
						->setParameter('notIds', $customer->disabledPayments->getKeys());
				}
			}

			$this->cPublished = $this->fillDao($qb->getQuery()->getArrayResult());
		}

		return $this->cPublished;
	}

	/**
	 * @param Dao\Payment[] $payments
	 * @param float         $cartValue
	 *
	 * @return Dao\Payment[]
	 */
	public function filterByCartValue(array $payments, float $cartValue): array
	{
		foreach ($payments as $k => $payment) {
			if ($payment->getAvailableFrom() > $cartValue || $payment->getAvailableTo() < $cartValue)
				unset($payments[$k]);
		}

		return $payments;
	}

	/**
	 * @param int $id
	 *
	 * @return \Doctrine\ORM\Proxy\Proxy|Payment|null
	 */
	public function getReference(int $id) { return $this->getEr()->getReference($id); }

	/**
	 * @param array[] $paymentsRaw
	 *
	 * @return Dao\Payment[]
	 */
	public function fillDao($paymentsRaw)
	{
		$payments = [];
		foreach ($paymentsRaw as $v) {
			$payment = new Dao\Payment();

			$payment
				->setId((int) $v['id'])
				->setIdent($v['ident'])
				->setName($v['name'])
				->setText($v['text'])
				->setPublished((int) $v['isPublished'])
				->setPosition((int) $v['position'])
				->setFreeFrom($v['freeFrom'] !== null ? (float) $v['freeFrom'] : null)
				->setAvailableFrom($v['availableFrom'])
				->setAvailableTo($v['availableTo'])
				->setImage($v['image']);
			$payment->vat = (int) EshopOrdersConfig::load('payments.vat', 21);

			$price = (float) $v['price'];
			if (EshopOrdersConfig::load('payments.priceIsWithoutVat'))
				$price = round($price * (($payment->vat / 100) + 1), 2);
			$payment->setPrice($price);

			$payment->currency               = $v['currency'] ?: null;
			$payment->priceInBaseCurrency    = $payment->getPrice();
			$payment->freeFromInBaseCurrency = $payment->getFreeFrom();
			$payment->disableInGoogle        = (bool) $v['disableInGoogle'];

			$payments[$payment->getId()] = $payment;
		}

		$this->eventDispatcher->dispatch(new PaymentsSpeditionsEvent($payments, null), Payments::class . '::afterFillDao');

		return $payments;
	}
}

