<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use EshopOrders\FrontModule\Components\Customer\IOrdersListFactory;
use EshopOrders\FrontModule\Components\Customer\IProfileFormFactory;
use EshopOrders\FrontModule\Model\Orders;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Nette\Application\BadRequestException;
use Users\Model\Users;

class CustomerPresenter extends BasePresenter
{
	/** @var Orders */
	protected $ordersService;

	/** @var Users */
	protected $usersService;

	/** @var Customers */
	protected $customersService;

	public Invoices $invoices;

	public IInvoicePdfBuilderFactory $invoicePdfBuilderFactory;

	public function __construct(Orders $orders, Users $users, Customers $customers, Invoices $invoices, IInvoicePdfBuilderFactory $invoicePdfBuilder)
	{
		$this->ordersService            = $orders;
		$this->usersService             = $users;
		$this->customersService         = $customers;
		$this->invoices                 = $invoices;
		$this->invoicePdfBuilderFactory = $invoicePdfBuilder;
	}

	public function checkRequirements($element): void
	{
		parent::checkRequirements($element);
		if (!$this->getUser()->isLoggedIn())
			$this->redirect(':Users:Front:Login:');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault()
	{
		$this->template->messages     = $this->getTemplate()->flashes;
		$this->getTemplate()->flashes = [];
	}

	public function actionOrderDetail(int $orderId)
	{
		$userId   = (int) $this->getUser()->id;
		$customer = $this->customersService->getByUserId($userId);
		$order    = $this->ordersService->getDetail($orderId);

		if ($order->getCustomer() == null || $order->getCustomer()->getId() != $customer->getId())
			throw new BadRequestException();

		$this->template->order = $order;
	}

	public function actionEditProfile()
	{
	}

	public function actionInvoice(int $id)
	{
		if (!EshopOrdersConfig::load('invoice.enable', false))
			throw new BadRequestException();

		/** @var Order $order */
		$order = $this->ordersService->getDetail($id);

		if (!$order || !$order->getCustomer() || $order->getCustomer()->getId() != $this->getUser()->getId())
			throw new BadRequestException();

		$invoice = $this->invoices->getOneByOrder($order, false);

		if (!$invoice)
			throw new BadRequestException();

		$this->invoicePdfBuilderFactory->create($invoice)->render();
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersList(IOrdersListFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentProfileForm(IProfileFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getUser()->id) {
			$userId              = (int) $this->getUser()->id;
			$control->onAnchor[] = function() use ($userId, $control) {
				$user = $this->usersService->get($userId);

				$customer = $this->customersService->getOrCreateCustomer($user->getId(), $user->getEmail(), $user->getName(), $user->getLastname());
				$control->setUser($user, $customer);
			};
		}

		$control['form']->onSuccess['redirect'] = function() {
			$this->redirect('this');
		};

		return $control;
	}
}
