<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Event\Event;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImageHelper;
use Currency\Model\Config as CurrenciesConfig;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\Dao\BankAccount;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\SellerInSite;
use EshopOrders\AdminModule\Model\Event\InvoiceRegenerateEvent;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use Exception;
use Doctrine\ORM\Query\Expr\Join;
use Core\Model\Event\EventDispatcher;
use Nette\Localization\ITranslator;
use Nette\Utils\DateTime;
use Nette\Utils\Image;
use Tracy\Debugger;
use function Symfony\Component\String\b;

/**
 * Class Invoices
 * @package EshopOrders\Model
 *
 * @method Invoice|null get($id)
 */
class Invoices extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Invoice::class;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var ITranslator */
	protected $translator;

	/** @var Sellers */
	protected $sellers;

	protected Products $products;

	protected ProductsFacade $productsFacade;

	protected InvoiceConfigRepository $invoiceConfigRepository;

	public function __construct(EventDispatcher $eventDispatcher, ProductsFacade $productsFacade, ITranslator $translator, Sellers $sellers,
	                            InvoiceConfigRepository $invoiceConfigRepository, Products $products)
	{
		$this->eventDispatcher         = $eventDispatcher;
		$this->productsFacade          = $productsFacade;
		$this->translator              = $translator;
		$this->sellers                 = $sellers;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->products                = $products;
	}

	/**
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('i')->orderBy('i.ident', 'desc');
	}

	/**
	 * @param Order  $order
	 * @param bool   $generateIfNotExist
	 * @param string $type
	 *
	 * @return Invoice|null
	 * @throws NonUniqueResultException
	 */
	public function getOneByOrder(Order $order, bool $generateIfNotExist = false, string $type = InvoiceConfig::TYPE_INVOICE): ?Invoice
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('i, o, s')
			->from(Invoice::class, 'i')
			->innerJoin('i.order', 'o', Join::WITH, 'o.invoice = i.id')
			->innerJoin('i.seller', 's')
			->where('o.id = :orderId')
			->setParameter('orderId', $order->getId());

		$invoice = $qb->getQuery()->getOneOrNullResult();

		if (!$invoice && $generateIfNotExist) {
			$invoice = $this->buildInvoice($order, null, $type);

			$this->em->persist($invoice);
			$this->em->persist($order);
			$this->em->flush($invoice);
			$this->em->flush($order);
		}

		return $invoice;
	}

	public function getByDateAndSeller(?int $sellerId = null, \DateTimeInterface $from, \DateTimeInterface $to): array
	{
		$result = [];

		$qb = $this->em->createQueryBuilder()
			->select('i, o, op, sp, oCur, corrTaxDoc, iData, customer, supplier, customerAddr, oAddrDeli, oAddrInv')
			->from(Invoice::class, 'i')
			->innerJoin('i.order', 'o', Join::WITH, 'o.invoice = i.id')
			->innerJoin('i.invoiceData', 'iData')
			->leftJoin('o.addressDelivery', 'oAddrDeli')
			->leftJoin('o.addressInvoice', 'oAddrInv')
			->leftJoin('iData.customer', 'customer')
			->leftJoin('customer.address', 'customerAddr')
			->leftJoin('iData.supplier', 'supplier')
			->leftJoin('iData.payment', 'op')
			->leftJoin('iData.spedition', 'sp')
			->leftJoin('o.currency', 'oCur')
			->leftJoin('o.correctiveTaxDocumentOf', 'corrTaxDoc')
			->andWhere('i.createdAt >= :fromDate')
			->andWhere('i.createdAt <= :toDate')
			->setParameters([
				'fromDate' => $from,
				'toDate'   => $to,
			])
			->groupBy('i.id');

		if ($sellerId !== null) {
			$qb->andWhere('i.seller = :sellerId')
				->setParameter('sellerId', $sellerId);
		}

		foreach ($qb->getQuery()->getResult() as $row) {
			/** @var Invoice $row */
			$result[$row->seller->getId()][] = $row;
		}

		return $result;
	}

	public function getCheckList(?int $sellerId = null, \DateTimeInterface $from, \DateTimeInterface $to): array
	{
		$list    = [];
		$allVats = [];
		$items   = [];
		foreach ($this->getByDateAndSeller($sellerId, $from, $to) as $seller => $rows) {
			foreach ($rows as $row) {
				/** @var Invoice $row */
				$date          = $row->createdAt->format('m/Y');
				$curr          = $row->invoiceData->currency;
				$type          = $row->order->isCorrectiveTaxDocument ? 'corr' : 'inv';
				$key           = $date . '-' . $type . '-' . $curr;
				$otherCurrency = $row->invoiceData->currency !== CurrenciesConfig::load('default', 'CZK');
				$isCanceled    = $row->order->getNewestOrderStatus() == OrderStatus::STATUS_CANCELED;
				$item          = [
					'ident'    => $row->ident,
					'date'     => $row->createdAt->format('d/m/Y'),
					'currency' => $curr,
					'type'     => $type,
					'vats'     => [],
				];

				if (!isset($list[$key])) {
					$list[$key] = [
						'date'     => $date,
						'currency' => $curr,
						'type'     => $type,
						'count'    => 0,
						'sumBase'  => 0,
						'sumCurr'  => 0,
						'vats'     => [
						],
					];
				}

				if (!$isCanceled) {
					$price = $row->invoiceData->getPriceWithoutVat();

					$list[$key]['count']++;

					$list[$key]['sumCurr'] += $price;
					$item['sumCurr']       = $price;


					if ($otherCurrency) {
						$price                 = round($price * $row->order->currency->rate);
						$list[$key]['sumBase'] += $price;
						$item['sumBase']       = $price;
					} else {
						$list[$key]['sumBase'] += $price;
						$item['sumBase']       = $price;
					}
				}

				foreach ($row->invoiceData->getVatRates() as $k => $v) {
					if (!isset($list[$key]['vats'][$k])) {
						$list[$key]['vats'][$k] = 0;
						$allVats[]              = $k;
					}

					if (!$isCanceled) {
						$price = round($v['total'] - $v['withoutVat'], 2);
						if ($otherCurrency)
							$price = round($price * $row->order->currency->rate);

						$list[$key]['vats'][$k] += $price;
						$item['vats'][$k]       = $price;
					}
				}

				$items[] = $item;
			}
		}

		$total = [
			'sumBase' => 0,
			'count'   => 0,
			'vats'    => [],
		];
		foreach ($list as $v) {
			$total['sumBase'] += $v['sumBase'];
			$total['count']   += $v['count'];

			foreach ($v['vats'] as $k => $v) {
				if (!isset($total['vats'][$k]))
					$total['vats'][$k] = 0;

				$total['vats'][$k] += $v;
			}
		}

		ksort($list);

		return [
			'list'    => $list,
			'total'   => $total,
			'allVats' => $allVats,
			'items'   => $items,
		];
	}

	public function getOrdersByDateAndSeller(?int $sellerId = null, \DateTimeInterface $from, \DateTimeInterface $to): array
	{
		$ids   = [];
		$sites = [];

		$qb = $this->em->createQueryBuilder()->select('IDENTITY(sin.site) as site')
			->from(SellerInSite::class, 'sin');

		if ($sellerId !== null) {
			$qb->where('sin.seller = :seller')
				->setParameter('seller', $sellerId);
		}

		foreach ($qb->getQuery()->getArrayResult() as $row)
			$sites[] = $row['site'];

		foreach ($this->em->createQueryBuilder()
			         ->select('o.id')
			         ->from(Order::class, 'o')
			         ->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :status AND os.created >= :fromDate AND os.created<= :toDate')
			         ->andWhere('o.site IN (:sites)')
			         ->setParameters([
				         'sites'    => $sites,
				         'status'   => OrderStatus::STATUS_CREATED,
				         'fromDate' => $from,
				         'toDate'   => $to,
			         ])->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	/**
	 * @param int $invoiceId
	 *
	 * @return bool
	 */
	public function regenerateInvoice(int $invoiceId): bool
	{
		try {
			$this->em->beginTransaction();

			$qb = $this->em->createQueryBuilder();
			$qb->select('o')
				->from(Order::class, 'o')
				->join('o.invoice', 'i')
				->where('i.id = :invoiceId')
				->setParameter('invoiceId', $invoiceId);

			/** @var Order|null $order */
			$order = $qb->getQuery()->getOneOrNullResult();

			if ($order === null) {
				return false;
			}

			$invoice = $order->getInvoice();

			if ($invoice === null) {
				return false;
			}

			$invoice = $this->buildInvoice($order, $invoice);

			$this->eventDispatcher->dispatch(new InvoiceRegenerateEvent($invoice), 'eshopOrders.beforeSaveRegeneratedInvoice');

			$this->em->persist($invoice->invoiceData);
			$this->em->flush();

			$this->em->commit();

			return true;
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();

			return false;
		}
	}

	/**
	 * @param array $sellersIds
	 *
	 * @return int
	 * @throws NoResultException
	 * @throws NonUniqueResultException
	 */
	public function getIvoicesCount(array $sellersIds): int
	{
		if (empty($sellersIds))
			return 0;

		$qb = $this->em->createQueryBuilder();
		$qb->select('COUNT(i.id)')
			->from(Invoice::class, 'i')
			->join('i.seller', 's')
			->where($qb->expr()->in('s.id', $sellersIds));

		return (int) $qb->getQuery()->getSingleScalarResult();
	}

	protected function buildInvoice(Order $order, ?Invoice $invoice = null, string $type = InvoiceConfig::TYPE_INVOICE): ?Invoice
	{
		$isNew = false;
		/** @var Seller $seller */
		$seller = $this->sellers->getSellerForSite($order->site->getIdent());

		if (!$invoice) {
			$sellerId = $seller->id;
			$event    = new Event(['order' => $order]);
			$this->eventDispatcher->dispatch($event, self::class . '::beforeGenerateInvoiceIdent');

			if (isset($event->data['seller']) && $event->data['seller'] !== null) {
				$sellerId = $event->data['seller']->getId();
			}

			$isNew        = true;
			$invoiceIdent = $this->invoiceConfigRepository->generateIdent($sellerId, $type);

			if (!$invoiceIdent)
				return null;

			$invoice        = new Invoice($this->invoiceConfigRepository->getDueDate($sellerId), $invoiceIdent,
				$this->em->getReference(\EshopCatalog\Model\Entities\Seller::class, $seller->id));
			$invoice->order = $order;
			if ($type !== InvoiceConfig::TYPE_CORRECTIVE) {
				$order->setInvoice($invoice);
			}
		}

		$invoiceData = $isNew ? null : $invoice->invoiceData;

		// invoice payment
		$invoicePayment = $isNew ? new Invoice\Payment() : $invoiceData->getPayment();
		$orderPayment   = $order->getPayment();

		$invoicePayment->paymentId = $orderPayment->getId();
		$invoicePayment->name      = $orderPayment->getName();
		$invoicePayment->setPrice(Strings::formatEntityDecimal($orderPayment->getPrice(true)));

		// invoice spedition
		$invoiceSpedition = $isNew ? new Invoice\Spedition() : $invoiceData->getSpedition();
		$orderSpedition   = $order->getSpedition();

		$invoiceSpedition->speditionId = $orderSpedition->getId();
		$invoiceSpedition->name        = $orderSpedition->getName();
		$invoiceSpedition->setPrice(Strings::formatEntityDecimal($orderSpedition->getPrice(true)));

		// customer address
		$customerAddress = $isNew ? new Invoice\Address() : $invoiceData->customer->address;
		$addrInv         = $order->getAddressInvoice();

		$customerAddress->city    = $addrInv->getCity();
		$customerAddress->country = $addrInv->getCountry() ? $addrInv->getCountry()->getName() : null;
		$customerAddress->postal  = $addrInv->getPostal();
		$customerAddress->street  = $addrInv->getStreet();

		// customer
		$invoiceCustomer = $isNew ? new Invoice\Customer($customerAddress) : $invoiceData->customer;
		$addrInv         = $order->getAddressInvoice();

		$invoiceCustomer->phone     = $addrInv->getPhone();
		$invoiceCustomer->email     = $addrInv->getEmail();
		$invoiceCustomer->company   = $addrInv->getCompany();
		$invoiceCustomer->firstName = $addrInv->getFirstName();
		$invoiceCustomer->lastName  = $addrInv->getLastName();
		$invoiceCustomer->idNumber  = $addrInv->getIdNumber();
		$invoiceCustomer->vatNumber = $addrInv->getVatNumber();

		// bank
		$bank = $isNew ? new Invoice\Bank() : $invoiceData->getSupplier()->getBank();
		// supplier address
		$supplierAddress = $isNew ? new Invoice\Address() : $invoiceData->getSupplier()->getAddress();
		if ($seller && $seller->getBankAccount($invoiceData->currency)) {
			/** @var BankAccount $bankAccount */
			$bankAccount       = $seller->getBankAccount($invoiceData->currency);
			$bank->bankAccount = $bankAccount->numberPart1;
			$bank->bankCode    = $bankAccount->numberPart2;
			$bank->iban        = $bankAccount->iban;
			$bank->swift       = $bankAccount->swift;
			$bank->note        = $bankAccount->note;

			$supplierAddress->street = $seller->street;
			$supplierAddress->postal = $seller->postal;
			$supplierAddress->city   = $seller->city;

			// supplier
			$supplier             = $isNew ? new Invoice\Supplier($bank, $supplierAddress) : $invoiceData->getSupplier();
			$supplier->email      = $seller->email;
			$supplier->vatNumber  = $seller->dic;
			$supplier->idNumber   = $seller->ic;
			$supplier->name       = $seller->name;
			$supplier->isPayerVat = $seller->dic ? true : false;
		} else {
			$bank->bankAccount = EshopOrdersConfig::load('invoice.bank.bankAccount');
			$bank->bankCode    = EshopOrdersConfig::load('invoice.bank.bankCode');
			$bank->iban        = EshopOrdersConfig::load('invoice.bank.iban');
			$bank->swift       = EshopOrdersConfig::load('invoice.bank.swift');

			$supplierAddress->street = EshopOrdersConfig::load('invoice.supplier.street');
			$supplierAddress->postal = EshopOrdersConfig::load('invoice.supplier.postCode');
			$supplierAddress->city   = EshopOrdersConfig::load('invoice.supplier.city');

			// supplier
			$supplier             = $isNew ? new Invoice\Supplier($bank, $supplierAddress) : $invoiceData->getSupplier();
			$supplier->email      = EshopOrdersConfig::load('invoice.supplier.email');
			$supplier->vatNumber  = EshopOrdersConfig::load('invoice.supplier.taxIdentificationNumber');
			$supplier->idNumber   = EshopOrdersConfig::load('invoice.supplier.companyIdentificationNumber');
			$supplier->name       = EshopOrdersConfig::load('invoice.supplier.name');
			$supplier->isPayerVat = EshopOrdersConfig::load('invoice.isPayerVat');
		}

		if ($isNew) {
			$bank->variableSymbol = $invoiceIdent ?? '0';

			$invoiceData           = new Invoice\InvoiceData($invoiceCustomer, $supplier, $invoiceSpedition, $invoicePayment, $invoice);
			$invoiceData->lang     = $this->translator->getLocale();
			$invoiceData->currency = $order->getCurrencyCode();
		}

		$invoiceData->zeroVat = (int) $order->isZeroVat();

		// products
		$invoiceProducts = [];

		if (!$isNew) {
			foreach ($invoiceData->products as $product) {
				$this->em->remove($product);
			}
		}

		foreach ($order->getOrderItems() as $p) {
			$product = $p->getProductId() ? $this->products->get($p->getProductId()) : null;

			$invoiceProduct            = new Invoice\Product($invoiceData);
			$invoiceProduct->productId = $p->getProductId();
			$invoiceProduct->code1     = $p->getCode1();
			$invoiceProduct->name      = $p->getOrderItemText($this->translator->getLocale())->getName();
			$invoiceProduct->price     = Strings::formatEntityDecimal($p->getPrice(true));
			$invoiceProduct->quantity  = $p->getQuantity();
			$invoiceProduct->setVatRate((int) $p->getVatRate());

			if ($product && $product->getGallery()) {
				$cover = $product->getGallery()->getCoverImage();

				// create base64 image miniature
				if ($cover && file_exists(WWW_DIR . $cover->getFilePath())) {
					$invoiceProduct->imageBase64 = ImageHelper::createBase64Miniature(
						WWW_DIR . $cover->getFilePath(),
						EshopOrdersConfig::load('invoice.productMiniature.width'),
						null,
						Image::SHRINK_ONLY);
				}
			}

			$invoiceProducts[] = $invoiceProduct;
		}
		$invoiceData->products = $invoiceProducts;

		// discounts
		$invoiceDiscounts = [];

		// remove discounts
		foreach ($invoiceData->discounts as $discount) {
			$this->em->remove($discount);
		}

		$currentPrice = $invoiceData->getPriceItems();
		foreach ($order->getOrderDiscounts() as $orderDiscount) {
			$amount                 = $orderDiscount->calculateDiscount($currentPrice);
			$currentPrice           += $amount;
			$invoiceDiscount        = new Invoice\Discount($invoiceData);
			$invoiceDiscount->price = $amount;
			$invoiceDiscount->name  = $orderDiscount->getName();
			$invoiceDiscount->type  = $orderDiscount->getType();
			$invoiceDiscount->value = $orderDiscount->getValue();
			$invoiceDiscount->code  = $orderDiscount->getCode();

			$invoiceDiscounts[] = $invoiceDiscount;
		}

		$invoiceData->discounts = $invoiceDiscounts;

		if ($isNew)
			$invoice->invoiceData = $invoiceData;

		return $invoice;
	}

	/**
	 * @return Invoice[]
	 */
	public function getInvoicesOverdue(): array
	{
		$now = (new DateTime)->setTime(23, 59, 59, 59);
		$qb  = $this->getEr()->createQueryBuilder('i');
		$qb->join('i.order', 'o', 'WITH', 'o.invoice = i.id')
			->where('i.dueDate < :date AND o.isPaid = 0')
			->setParameter('date', $now);

		return $qb->getQuery()->getResult();
	}

}
