<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\UI\Form;
use Nette\Application\UI\ITemplateFactory;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Statuses;
use Users\Model\Users;

class OrderStatusForm extends BaseControl
{
	/** @var OrderStatus */
	public $orderStatus;
	
	/** @var Order */
	public $order;
	
	/** @var Orders */
	protected $ordersService;
	
	/** @var Users */
	protected $usersService;
	
	/** @var Statuses */
	protected $statusesService;
	
	/** @var ITemplateFactory */
	protected $templateFactory;
	
	/** @var IMailer */
	protected $mailer;
	
	/** @var Message */
	protected $message;
	
	/** @var string */
	protected $siteName;
	
	public function __construct($data, string $siteName, IMailer $mailer, ITemplateFactory $templateFactory, Orders $orders, Users $users, Statuses $statuses)
	{
		$this->ordersService   = $orders;
		$this->usersService    = $users;
		$this->statusesService = $statuses;
		
		$this->siteName			= $siteName;
		$this->message			= new Message();
		$this->mailer			= $mailer;
		$this->templateFactory	= $templateFactory;
		
		$this->setFrom($data['fromEmail'] ?? null, $data['fromName'] ?? null);
		$this->addTo($data['toEmail'] ?? null, $data['toName'] ?? null);
		$this->setSubject('');
		
		if (isset($data['bcc']) && $data['bcc']) {
			foreach (explode(',', $data['bcc']) as $v) {
				$this->message->addBcc(trim($v));
			}
		}
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}
	
	/*******************************************************************************************************************
	 * ======================== Setters
	 */
	public function setFrom($fromEmail, $fromName = null)
	{
		$this->message->setFrom($fromEmail, $fromName);
		return $this;
	}
	
	public function addTo($toEmail, $toName = null)
	{
		$this->message->addTo($toEmail, $toName);
		return $this;
	}
	
	public function setSubject($subject)
	{
		$this->message->setSubject($subject);
		return $this;
	}
	
	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		
		$statuses = $this->statusesService->getAll();
		$statusList = [];
		foreach ($statuses as $status) {
			$id = $status->getId();
			if($id != 'created' && $id != 'paid') {
				$statusList[$id] = $status->getName();
			}
		}

		//TODO defaultne nevybrany zadny status, a validace na to ze musi byt vybrany
		$form->addSelect('status', 'eshopOrders.default.status', $statusList);
		$form->addTextArea('message', 'eshopOrders.default.statusMessage');

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$order = $this->order;
			$status = $this->statusesService->getReference($values->status);
			$userRef = $this->usersService->getReference($this->getPresenter()->getUser()->id);
			
			$orderStatus = new OrderStatus($order, $status, $userRef);
			$orderStatus->setMessage($values->message);
			$this->em->persist($orderStatus)->flush();
			
			$this->getPresenter()->flashMessageSuccess('eshopOrders.statusForm.added');
			
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
			
			return false;
		}
		
		
		try {
			$this->sendOrderStatusEmail($order, $orderStatus);
			$this->getPresenter()->flashMessageSuccess('eshopOrders.statusForm.emailSent');
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageDanger('eshopOrders.statusForm.emailError');
			$this->redrawControl('form');

			return false;
		}
	}
	
	/**
	 * @param Order $order
	 */
	private function sendOrderStatusEmail($order, $orderStatus)
	{
		$template = $this->templateFactory->createTemplate();
		$template->setFile(__DIR__ . '/emailOrderStatus.latte');
		$template->setParameters([
			'subject' => $this->message->getSubject(),
			'orderStatus' => $orderStatus,
			'orderId' => $this->order->getId(),
		]);
		
		$this->message->setHtmlBody($template->renderToString());
		
		$this->addTo($order->getAddressDelivery()->getEmail(), $order->getAddressDelivery()->getFirstName() .' '. $order->getAddressDelivery()->getLastName());
		
		$this->mailer->send($this->message);
	}
	
	public function setOrder($orderId)
	{
		$this->order   = $this->ordersService->get($orderId);
		$subject = $this->translator->translate('eshopOrders.emails.subjectStatus', ['orderId' => $orderId, 'siteName' => $this->siteName]);
		$this->setSubject($subject);
	}
}
