<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Order;

use EshopOrders\Model\Entities\OrderFlag;
use Soukicz\Zbozicz;
use Nette\Utils\ArrayHash;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Nette\Forms\Controls\Button;
use Nette\Application\UI\ITemplateFactory;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Core\Model\UI\BaseControl;
use Core\Model\Countries;
use Users\Model\Users;
use Users\Model\Entities\User;
use Users\Model\Roles;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Payments;
use EshopOrders\Model\PaymentSpeditions;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Statuses;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\CartHelper;

class OrderForm extends BaseControl
{
	/** @var Statuses */
	protected $statusesService;
	
	/** @var CartHelper */
	protected $cartHelperService;
	
	/** @var IOrderCartDetailFactory */
	protected $orderCartDetailFactory;
	
	/** @var IPaySpedSummaryFactory */
	protected $paySpedSummaryFactory;
	
	/** @var IOrderSummaryFactory */
	protected $orderSummaryFactory;
	
	/** @var Orders */
	private $ordersService;
	
	/** @var Carts */
	private $cartsService;
	
	/** @var Speditions */
	private $speditionsService;
	
	/** @var Payments */
	private $paymentsService;
	
	/** @var PaymentSpeditions */
	protected $paymentSpeditionsService;
	
	/** @var Customers */
	private $customersService;
	
	/** @var Users */
	protected $usersService;
	
	/** @var Roles */
	protected $rolesService;
	
	/** @var Countries */
	protected $countriesService;
	
	/** @var ITemplateFactory */
	protected $templateFactory;
	
	/** @var IMailer */
	protected $mailer;
	
	/** @var Message */
	protected $message;
	
	/** @var [] */
	protected $zboziCzConfig;
	
	/** @var SessionSection */
	protected $sessionSection;
	
	/** @var Spedition[] */
	protected $speditions;
	
	/** @var Payment[] */
	protected $payments;
	
	/** @var PaymentSpedition[] */
	private $paymentSpeditions;
	
	/** @var Cart  */
	protected $cart;
	
	/** @var string */
	protected $siteName;
	
	/** @var array */
	protected $orderPageConfig;
	
	public function __construct($data, string $siteName, IMailer $mailer, ITemplateFactory $templateFactory, IOrderCartDetailFactory $orderCartDetailFactory, IPaySpedSummaryFactory $paySpedSummaryFactory, IOrderSummaryFactory $orderSummaryFactory, Orders $orders, Carts $carts, CartHelper $cartHelper, Speditions $speditions, Payments $payments, PaymentSpeditions $paymentSpeditions, Customers $customersService, Users $users, Roles $roles, Statuses $statuses, Countries $countries, Session $session)
	{
		$this->ordersService            = $orders;
		$this->cartsService             = $carts;
		$this->cartHelperService        = $cartHelper;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->customersService         = $customersService;
		$this->usersService             = $users;
		$this->rolesService             = $roles;
		$this->statusesService          = $statuses;
		$this->countriesService         = $countries;
		
		$this->orderCartDetailFactory   = $orderCartDetailFactory;
		$this->paySpedSummaryFactory    = $paySpedSummaryFactory;
		$this->orderSummaryFactory      = $orderSummaryFactory;
		
		$this->siteName					= $siteName;
		$this->message                  = new Message();
		$this->mailer                   = $mailer;
		$this->templateFactory          = $templateFactory;
		
		$this->sessionSection = $session->getSection('eshopOrdersOrderForm');
		
		$this->setFrom($data['fromEmail'] ?? null, $data['fromName'] ?? null);
		$this->addTo($data['toEmail'] ?? null, $data['toName'] ?? null);
		$this->setSubject('');

		if (isset($data['bcc']) && $data['bcc']) {
			foreach (explode(',', $data['bcc']) as $v) {
				$this->message->addBcc(trim($v));
			}
		}
		
		$this->zboziCzConfig = $data['zboziCzConfig'];
		$this->orderPageConfig = $data['orderPage'];
		
		$this->speditions        = $this->speditionsService->getAllPublished();
		$this->payments          = $this->paymentsService->getAllPublished();
		$this->paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();
		
		$this->cart             = $this->cartsService->getCurrentCart();
	}
	
	public function render()
	{
		$this->template->setFile($this->getTemplateFile());
		
		$this->template->visibleCountSpedition = $this->orderPageConfig['visibleCountSpedition'];
		$this->template->visibleCountPayment = $this->orderPageConfig['visibleCountPayment'];
		
		$this->template->speditions        = $this->speditions;
		$this->template->payments          = $this->payments;
		$this->template->paymentSpeditions = $this->paymentSpeditions;
		$this->template->cart              = $this->cart;
		$this->template->cartValue         = $this->cart->getPriceTotal();
		
		$paySpedStructured = [];
		$paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();
		foreach ($paymentSpeditions as $key => $item) {
			$paySpedStructured[$item->getSpedition()->getId()][] = $item->getPayment()->getId();
			$paySpedReversed[$item->getPayment()->getId()][] = $item->getSpedition()->getId();
		}
		$this->template->paymentSpeditionsStructured = $paySpedStructured;
		$this->template->paymentSpeditionsReversed = $paySpedReversed;
		
		$this->template->orderData  = $this->sessionSection->orderFormData;
		
		$discountCoupon = $this->cartHelperService->getDiscountCoupon($this->cart->getPriceTotal());
		if($discountCoupon) {
			$this->template->discountCoupon = $discountCoupon;
		}
		
		$this->template->render();
	}
	
	public function attached($presenter)
	{
		parent::attached($presenter);
		
		if($this->sessionSection->orderFormData) {
			$this->setOrderData($this->sessionSection->orderFormData);
		}
	}
	
	/*******************************************************************************************************************
	 * ======================== Setters
	 */
	public function setFrom($fromEmail, $fromName = null)
	{
		$this->message->setFrom($fromEmail, $fromName);
		return $this;
	}

	public function addTo($toEmail, $toName = null)
	{
		$this->message->addTo($toEmail, $toName);
		return $this;
	}

	public function setSubject($subject)
	{
		$this->message->setSubject($subject);
		return $this;
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();
		$cart = $this->cartsService->getCurrentCart();
		$cartValue = $cart->getPriceTotal();
		
		$form->addGroup('speditions');
		$paymentChoices = $speditionChoices = [];
		$speditions = $this->speditionsService->getAllByCartValue($cartValue);
		$payments = $this->paymentsService->getAllByCartValue($cartValue);
		foreach($speditions as $s) {
			$speditionChoices[$s->getId()] = $s->getName() .' - '. $s->getPriceActual($cart) . ' Kč'; //TODO cuurency
		}
		foreach($payments as $p) {
			$paymentChoices[$p->getId()] = $p->getName() .' - '. $p->getPrice() . ' Kč'; //TODO cuurency
		}
		$form->addRadioList('spedition', 'eshopOrders.orderPage.spedition', $speditionChoices)->setRequired();
		$form->addRadioList('payment', 'eshopOrders.orderPage.payment', $paymentChoices)->setRequired();
		
		$form->addSubmit('validateSpedition', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form['payment'],
				$form['spedition'],
			]);
		
		$form->addGroup('personal');
		$form->addText('firstName', 'eshopOrders.orderPage.name')->setRequired();
		$form->addText('lastName', 'eshopOrders.orderPage.lastName')->setRequired();
		$form->addEmail('email', 'eshopOrders.orderPage.email')->setRequired();
		$form->addText('phone', 'eshopOrders.orderPage.phone')->setRequired();
		$form->addText('street', 'eshopOrders.orderPage.street')->setRequired();
		$form->addText('city', 'eshopOrders.orderPage.city')->setRequired();
		$form->addText('postal', 'eshopOrders.orderPage.postal')->setRequired();
		$countries = $this->countriesService->getAllNameColumn();
		$form->addSelect('country', 'eshopOrders.orderPage.country', $countries);
		$form->addText('company', 'eshopOrders.orderPage.company');
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber');
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');
		
		//fakturacni adresa
		$form->addCheckbox('useAddressInvoice', 'eshopOrders.orderPage.useAddressInvoice'); //TODO na tohle lze pouzit nette funkci ->addCondition($form::EQUAL, true) ->toggle('addressInvoice');
		$form->addText('firstName2', 'eshopOrders.orderPage.name');
		$form->addText('lastName2', 'eshopOrders.orderPage.lastName');
		$form->addEmail('email2', 'eshopOrders.orderPage.email');
		$form->addText('phone2', 'eshopOrders.orderPage.phone');
		$form->addText('street2', 'eshopOrders.orderPage.street');
		$form->addText('city2', 'eshopOrders.orderPage.city');
		$form->addText('postal2', 'eshopOrders.orderPage.postal');
		$form->addSelect('country2', 'eshopOrders.orderPage.country', $countries);
		$form->addText('company2', 'eshopOrders.orderPage.company');
		$form->addText('idNumber2', 'eshopOrders.orderPage.idNumber');
		$form->addText('vatNumber2', 'eshopOrders.orderPage.vatNumber');
		
		$form->addSubmit('validatePersonal', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form['firstName'],
				$form['lastName'],
				$form['phone'],
				$form['email'],
				$form['street'],
				$form['city'],
				$form['postal'],
				$form['company'],
				$form['idNumber'],
				$form['vatNumber'],
				$form['firstName2'],
				$form['lastName2'],
				$form['street2'],
				$form['city2'],
				$form['postal2'],
			]);
		
		$form->addTextArea('messageAlt', 'eshopOrders.orderPage.message')->setAttribute('class', ['messageInput']);
		$form->addTextArea('message', 'eshopOrders.orderPage.message');
		
		$form->addCheckbox('agreedTerms', 'eshopOrders.orderPage.agreedTerms')->setRequired();
		$form->addCheckbox('agreedQuestioning', 'eshopOrders.orderPage.agreedQuestioning')->setDefaultValue(1);
		
		//tlacitko pro ulozeni dat do session mezi kroky objednavky
		$saveButtonCallback = [$this, 'saveFormData'];
		$form->addSubmit('saveData', 'save')
			->setValidationScope(false)
			->setHtmlAttribute('class', ['saveFormData'])
			->onClick[] = function(Button $button) use ($saveButtonCallback) {
				$values = $button->getForm()->getValues(true);
				$saveButtonCallback($values);
			};
		
		$form->addSubmit('submit', 'eshopOrders.orderPage.send')->setAttribute('class', ['submitButton', 'eshopOrdersNext4']);
		
		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}
	
	protected function saveFormData($data)
	{
		$cart = $this->cartsService->getCurrentCart();
		$cartValue = $cart->getPriceTotal();
		$priceTotal = $cartValue;
		if($data['spedition']) $priceTotal += $this->speditions[$data['spedition']]->getPriceActual($cart);
		if($data['payment']) $priceTotal += $this->payments[$data['payment']]->getPrice();
		if($this->sessionSection->discountCode) {
			$type = OrderDiscount::TYPE_PERCENT;
			$priceTotal += $this->cartHelperService->countDiscountPrice($cartValue, $type);
		}
		$data['priceTotal'] = $priceTotal;
		
		if($data['message']) {
			$data['messageAlt'] = $data['message'];
		} else {
			$data['message'] = $data['messageAlt'];
		}
		
		$this->sessionSection->orderFormData = $data;
		
		$this->redrawControl('orderCartDetailSummary');
		$this->redrawControl('orderSummary');
		$this->redrawControl('paySpedSummary');
	}
	
	public function setOrderData($orderData)
	{
		$this['form']->setDefaults([
			'firstName'         => $orderData['firstName'],
			'lastName'          => $orderData['lastName'],
			'phone'             => $orderData['phone'],
			'email'             => $orderData['email'],
			'street'            => $orderData['street'],
			'city'              => $orderData['city'],
			'postal'            => $orderData['postal'],
			'company'           => $orderData['company'],
			'country'           => $orderData['country'],
			'idNumber'          => $orderData['idNumber'],
			'vatNumber'         => $orderData['vatNumber'],
			'firstName2'        => $orderData['firstName2'],
			'lastName2'         => $orderData['lastName2'],
			'phone2'            => $orderData['phone2'],
			'email2'            => $orderData['email2'],
			'street2'           => $orderData['street2'],
			'city2'             => $orderData['city2'],
			'postal2'           => $orderData['postal2'],
			'country2'          => $orderData['country2'],
			'company2'          => $orderData['company2'],
			'idNumber2'         => $orderData['idNumber2'],
			'vatNumber2'        => $orderData['vatNumber2'],
			'spedition'         => $orderData['spedition'],
			'payment'           => $orderData['payment'],
			'messageAlt'        => $orderData['message'],
			'message'           => $orderData['message'],
			'useAddressInvoice' => $orderData['useAddressInvoice'],
			'agreedTerms'       => $orderData['agreedTerms'],
			'agreedQuestioning' => $orderData['agreedQuestioning'],
			
		]);
	}

	public function formOnValidate($form)
	{
		$cart = $this->cartsService->getCurrentCart();
		$cartItems = $cart->getCartItems();
		if(empty($cartItems)) {
			$form->addError('eshopOrders.orderPage.emptyCart');
			//$this->getPresenter()->redrawControl('orderPage');
			$this->template->formSummaryMessages = ['eshopOrders.orderPage.emptyCart', 'eshopOrders.orderPage.emptyCartData'];
			$this->redrawControl('summaryMessages');
		}
		// nebude potreba take validovat jestli doprava a platba jsou aktivni a plati pro cenu kosiku?
	}

	public function formOnSuccess($form, ArrayHash $values)
	{
		if(!$form['submit']->isSubmittedBy()) {
			return;
		}
		
		try {
			$ident = $this->genUuid();
			$order = new Order($ident);
			
			$order->setMessage($values->message);
			$order->setAgreedTerms($values->agreedTerms);
			
			if($values->agreedQuestioning) {
				$orderFlagQuestioning = new OrderFlag(OrderFlag::TYPE_QUESTIONING, true, $order);
				$order->addFlag($orderFlagQuestioning);
			}
			
			$addrDelivery = new OrderAddress($this->ordersService::ADDRESS_DELIVERY);
			$addrDelivery->setFirstName($values->firstName);
			$addrDelivery->setLastName($values->lastName);
			$addrDelivery->setEmail($values->email);
			$addrDelivery->setPhone($values->phone);
			$addrDelivery->setStreet($values->street);
			$addrDelivery->setCity($values->city);
			$addrDelivery->setPostal($values->postal);
			if($values->country) {
				$addrDelivery->setCountry($this->countriesService->getReference($values->country));
			}
			$addrDelivery->setCompany($values->company);
			$addrDelivery->setIdNumber($values->idNumber);
			$addrDelivery->setVatNumber($values->vatNumber);
			
			$order->setAddressDelivery($addrDelivery);
			
			if($values->useAddressInvoice) {
				$addrInvoice = new OrderAddress($this->ordersService::ADDRESS_INVOICE);
				$addrInvoice->setFirstName($values->firstName2);
				$addrInvoice->setLastName($values->lastName2);
				$addrInvoice->setEmail($values->email2);
				$addrInvoice->setPhone($values->phone2);
				$addrInvoice->setStreet($values->street2);
				$addrInvoice->setCity($values->city2);
				$addrInvoice->setPostal($values->postal2);
				$addrInvoice->setCountry($this->countriesService->getReference($values->country2));
				$addrInvoice->setCompany($values->company2);
				$addrInvoice->setIdNumber($values->idNumber2);
				$addrInvoice->setVatNumber($values->vatNumber2);
				
				$order->setAddressInvoice($addrInvoice);
			}
			
			$customer = $this->getOrCreateCustomer(
				$order->getAddressDelivery()->getEmail(),
				$order->getAddressDelivery()->getFirstName(),
				$order->getAddressDelivery()->getLastName(),
				$order->getAddressDelivery()->getPhone());
			if($customer) {
				$order->setCustomer($customer);
			}
			
			$cart = $this->cartsService->getCurrentCart();
			$cartItems = $cart->getCartItems();
			$orderItems = $this->ordersService->fillOrderItems($cartItems, $order);
			$order->setOrderItems($orderItems);
			
			$orderPayment = new OrderPayment($this->paymentsService->getReference($values->payment), $order);
			$order->setPayment($orderPayment);
			
			$orderSpedition = new OrderSpedition($this->speditionsService->getRawReference($values->spedition), $order);
			$spedition = $this->speditionsService->get($values->spedition);
			$orderSpedition->setPrice($spedition->getPriceActual($cart)); //pripadne doprava zdarma podle ceny objednavky
			$order->setSpedition($orderSpedition);
			
			$statusCreated = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);
			
			//ale mozna ze to nebude $values->discountCode, ale session[discountCode]
			if($this->sessionSection->discountCode && $this->cartHelperService->checkValidDiscountCode($this->sessionSection->discountCode)) {
				$type = OrderDiscount::TYPE_PERCENT;
				$typeText = $this->cartHelperService->getDiscountTypeText($type);
				$discount = new OrderDiscount($this->sessionSection->discountCode, $order);
				$discount->setValue($this->cartHelperService->discountValue);
				$discount->setType($type);
				$discount->setPrice($this->cartHelperService->countDiscountPrice($cart->getPriceTotal(), $type));
				$discount->setName($this->translator->translate('eshopOrders.orderPage.discount') . $discount->getValue() . $typeText);
				$order->addOrderDiscount($discount); //nutne, jinak se objednavka zacachuje bez slevovych kodu a v emailu budou chybet
				unset($this->sessionSection->discountCode); //smazeme vlozeny kod ze session -> v nove objednavce vlozit znova
				
				//TODO promítnout do celkové ceny objednávky
			}

			$this->em->getConnection()->beginTransaction();
			try {
				//ulozit objednavku
				$this->em->persist($order);
				//ulozit k ni adresy
				$this->em->persist($addrDelivery);
				if($addrInvoice) $this->em->persist($addrInvoice);
				//polozky doprava a platba
				$this->em->persist($orderSpedition);
				$this->em->persist($orderPayment);
				//ulozit k ni polozky
				foreach($orderItems as $oi) {
					$this->em->persist($oi);
				}
				//akce - datum kdy byla objednavka vytvorena
				$this->em->persist($orderActionCreated);
				//pripadny slevovy kod
				if($discount) $this->em->persist($discount);
				//pripadne nastavene priznaky
				if($orderFlagQuestioning) $this->em->persist($orderFlagQuestioning);
				//teprve po ulozeni vseho flush (aby byly spravne propojene cizi klice)
				$this->em->flush();
				$this->em->getConnection()->commit();
				$this->getPresenter()->flashMessageSuccess('eshopOrders.orderPage.saved');
				
				//TODO nejake promazavani kosiku z DB, kdyz uz k nim neni aktivni session... nebo po nejakem timeoutu (tyden a pod)
			} catch (Exception $e) {
				$this->em->getConnection()->rollBack();
				throw $e;
			}
			
			//dale uz pracujeme s objednavkou nactenou z DB
			$orderDetail = $this->ordersService->get($order->getId());
			unset($order);
			$this->cartsService->deleteCurrentCart();
			$this->sessionSection->orderFormData = [];
			
			$this->sendOrderEmail($orderDetail);
			
			if($orderDetail->hasFlag(OrderFlag::TYPE_QUESTIONING)) {
				$this->sendZboziOrder($orderDetail);
			}
			
			$this->sessionSection->trackConversion = true;
			
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
		
		$this->getPresenter()->redirect('Finished:orderFinished', array('orderIdent' => $orderDetail->getIdent()));
		
	}
	
	/** odesle konverzi na Zbozi.cz
	 * @param Order $order
	 * @return bool
	 */
	private function sendZboziOrder($order)
	{
		if( ! $this->zboziCzConfig['shopId'] || ! $this->zboziCzConfig['privateKey'])
			return false;
		
		$client = new Zbozicz\Client($this->zboziCzConfig['shopId'], $this->zboziCzConfig['privateKey'], $this->zboziCzConfig['isSandbox']);
		
		//radeji pouzijeme kod, ale pokud neni, tak alespon ID
		$spedition = $order->getSpedition();
		$spedId = $spedition->getSpedition()->getZboziId();
		if(!$spedId) $spedId = $spedition->getSpedition()->getId();
		
		$payment = $order->getPayment();
		$payId = $payment->getPayment()->getIdent();
		if(!$payId) $payId = $payment->getPayment()->getId();
		
		$zboziOrder = new Zbozicz\Order($order->getId());
		$zboziOrder
			->setEmail($order->getAddressDelivery()->getEmail())
			->setDeliveryType($spedId)
			->setDeliveryPrice($spedition->getPrice())
			->setPaymentType($payId);
		
		$otherCosts = 0;
		$otherCosts += $payment->getPrice();
		if($order->getOrderDiscounts()) {
			foreach ($order->getOrderDiscounts() as $orderDiscount) {
				$otherCosts += $orderDiscount->getPrice();
			}
		}
		$zboziOrder->setOtherCosts($otherCosts);
		
		foreach ($order->getOrderItems() as $orderItem) {
			$zboziOrder->addCartItem((new Zbozicz\CartItem)
				->setId($orderItem->getProductId())
				->setName($orderItem->getOrderItemText('cs')->getName()) //TODO lang
				->setUnitPrice($orderItem->getPrice())
				->setQuantity($orderItem->getQuantity())
			);
		}
		
		$client->sendOrder($zboziOrder);
		return true;
	}
	
	/**
	 * @param Order $order
	 */
	private function sendOrderEmail($order)
	{
		$subject = $this->translator->translate('eshopOrders.emails.subject', ['orderId' => $order->getId(), 'siteName' => $this->siteName]);
		$this->setSubject($subject);
		
		$template = $this->templateFactory->createTemplate();
		$template->setFile(__DIR__ . '/email.latte');
		$template->setParameters([
			'subject' => $this->message->getSubject(), 
			'order' => $order,
		]);
		
		$this->message->setHtmlBody($template->renderToString());
		
		$this->addTo($order->getAddressDelivery()->getEmail(), $order->getAddressDelivery()->getFirstName() .' '. $order->getAddressDelivery()->getLastName());

		$this->mailer->send($this->message);
		$this->getPresenter()->flashMessageSuccess('eshopOrders.orderPage.sent');
	}
	
	protected function getOrCreateCustomer($email, $firstName = '', $lastName = '', $phone = '')
	{
		$userId = null;
		if($this->getPresenter()->getUser()->isLoggedIn()) {
			$userId = $this->getPresenter()->getUser()->id;
		}
		$customer = $this->customersService->getOrCreateCustomer($userId, $email, $firstName, $lastName, $phone);
		return $customer;
	}
	
	/** http://stackoverflow.com/a/2040279
	 * @return string UUID - unikatni ID, tezke na uhodnuti
	 */
	protected function genUuid() {
		return sprintf( '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ),
			
			// 16 bits for "time_mid"
			mt_rand( 0, 0xffff ),
			
			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand( 0, 0x0fff ) | 0x4000,
			
			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand( 0, 0x3fff ) | 0x8000,
			
			// 48 bits for "node"
			mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff )
		);
	}
	
	/*******************************************************************************************************************
	 * ============================== Components
	 */
	
	protected function createComponentOrderCartDetail()
	{
		$control = $this->orderCartDetailFactory->create();
		return $control;
	}
	
	protected function createComponentPaySpedSummary()
	{
		$control = $this->paySpedSummaryFactory->create();
		
		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData['spedition'], $orderData['payment'], $orderData['priceTotal']);
		
		return $control;
	}
	
	protected function createComponentOrderSummary()
	{
		$control = $this->orderSummaryFactory->create();
		
		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData);
		
		return $control;
	}
}
