<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopOrders\Model\Entities\Cart;
use EshopOrders\Model\Entities\CartItem;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\FrontModule\Model\Dao\Product;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\SmartObject;

/**
 * class Carts
 * @package EshopOrders\Model
 *
 */
class Carts extends BaseFrontEntityService
{
	use SmartObject;
	
	/** @var SessionSection */
	private $sessionSection;

	protected $entityClass = Cart::class;

	/** @var Dao\Cart[]|null */
	protected $cartsCached;

	/** @var Products */
	private $productsService;
	
	/** @var Tags */
	protected $tagsService;

	/** udalost pridani produktu do kosiku, na kterou je navazany listener */
	public $onAddItem = [];

	public function __construct(Session $session, Products $products, Tags $tags)
	{
		$this->sessionSection  = $session->getSection('eshopOrdersCart');
		$this->productsService = $products;
		$this->tagsService     = $tags;
		$this->sessionSection->setExpiration('1 month');
	}

	/**
	 *
	 * @param int $itemId
	 *
	 * @return Dao\Cart
	 */
	public function getCartItem($itemId)
	{
		foreach ($this->cartsCached as $cart) {
			if (isset($cart->cartItems[$itemId])) {
				return $cart->cartItems[$itemId];
			}
		}

		$itemRaw = $this->loadRawCartItem($itemId);
		if($itemRaw) {
			$items   = $this->fillDaoItems([$itemRaw]);
			$item    = $items[$itemId];
		}

		return $item ?: null;
	}

	/** Vrati soucasny kosik podle ID ze session. Kdyz kosik neexistuje, vytvori novy
	 * @return Dao\Cart
	 */
	public function getCurrentCart()
	{
		$cartId = $this->sessionSection->cartId;

		if (!is_null($cartId)) {
			if (isset($this->cartsCached[$cartId]))
				return $this->cartsCached[$cartId];

			$cartRaw = $this->loadRawCart($cartId);
		}
		if (is_null($cartId) || is_null($cartRaw)) {
			//kdyz neni ID v session, nebo k danemu ID neni kosik v databazi, tak vygenerovat novy
			$cartRaw = $this->createRawCart();
		}

		$cartId                       = $cartRaw->getId();
		$this->sessionSection->cartId = $cartId;
		$this->cartsCached[$cartId]   = $this->fillDao($cartRaw);

		return isset($this->cartsCached[$cartId]) ? ($this->cartsCached[$cartId] ?: null) : null;
	}

	/** smaze kosik pro daneho uzivatele - po dokonceni objednavky
	 * pak, pri volani getCurrentCart se vygeneruje novy
	 */
	public function deleteCurrentCart()
	{
		$oldCartId = $this->sessionSection->cartId;
		$this->removeRawCart($oldCartId);
		$this->sessionSection->cartId = null;
		unset($this->cartsCached[$oldCartId]);
	}

	/** nacte kosik z databaze do vlastni property
	 *
	 * @param int $cartId
	 *
	 * @return Cart
	 */
	private function loadRawCart($cartId)
	{
		$cartRawQuery = $this->getEr()->createQueryBuilder('c', 'c.id');
		$cartRawQuery->andWhere('c.id = :id')->setParameter('id', $cartId);

		$cartRaw = $cartRawQuery->getQuery()->getOneOrNullResult();

		return $cartRaw;
	}

	/**
	 * @return Cart
	 */
	private function createRawCart()
	{
		$ident = substr(md5((string) rand()), 0, 7);
		$cart  = new Cart($ident);
		$this->em->persist($cart)->flush($cart);

		return $cart;
	}

	/**
	 * @param $cartId id kosiku ke smazani
	 */
	private function removeRawCart($cartId)
	{
		$cartRaw = $this->getEr()->getReference($cartId);
		$this->em->remove($cartRaw)->flush();
	}

	/** nacte jednu polozku kosiku z databaze
	 *
	 * @param int $itemId
	 *
	 * @return CartItem[]
	 */
	private function loadRawCartItem($itemId)
	{
		$cartRawQuery = $this->em->getRepository(CartItem::class)->createQueryBuilder('ci', 'ci.id');
		$cartRawQuery->andWhere('ci.id = :item_id')->setParameter('item_id', $itemId);

		$cartItemRaw = $cartRawQuery->getQuery()->getOneOrNullResult();

		return $cartItemRaw;
	}

	public function addItem(AddedCartItem $item)
	{
		$quantity  = $item->quantity;
		$productId = $item->productId;
		$optionId  = $item->optionId;
		$itemIdent = $productId .'|'. $optionId; //pripadne dalsi vlastnosti s vlivem na cenu

		$cartId = $this->getCurrentCart()->getId();

		$cartRaw = $this->loadRawCart($cartId);

		$itemRaw = $cartRaw->getCartItems()->get($itemIdent);

		if ($itemRaw) {
			//kdyz polozka uz je v kosiku, jen zvysime pocet
			$itemRaw->quantity = $itemRaw->quantity + $quantity;
		} else {
			//kdyz zatim neexistuje, pridame ji do kosiku
			$itemRaw = new CartItem($itemIdent, $productId, $cartRaw);
			$itemRaw->setVariantId($optionId);
			$itemRaw->setQuantity($quantity);
		}

		$this->em->persist($itemRaw)->flush();

		//vymazeme kosik z cache, aby se priste obnovil z DB.
		//Alternativni moznost - vzit soucasne DAO a pridat mu polozku
		unset($this->cartsCached[$cartId]);
		//TODO vratit nejaky priznak, ze to probehlo v poradku?
	}

	public function updateItemQuantity($itemId, $quantity)
	{
		if ($quantity <= 0) {
			return $this->removeItem($itemId);
		}
		$cartId = $this->getCurrentCart()->getId();

		$itemRaw = $this->loadRawCartItem($itemId);
		$itemRaw->setQuantity($quantity);
		$this->em->persist($itemRaw)->flush();

		unset($this->cartsCached[$cartId]); //obnoveni cache
	}

	public function removeItem($itemId)
	{
		$cartId = $this->getCurrentCart()->getId();

		$itemRaw = $this->loadRawCartItem($itemId);
		if($itemRaw) {
			$this->em->remove($itemRaw)->flush();
		}

		unset($this->cartsCached[$cartId]); //obnoveni cache
	}

	/**
	 * @param Cart $cartRaw
	 *
	 * @return Dao\Cart
	 */
	protected function fillDao($cartRaw)
	{
		$cart = new Dao\Cart();
		$cart
			->setId($cartRaw->getId())
			->setIdent($cartRaw->ident)
			->setCartItems($this->fillDaoItems($cartRaw->getCartItems()));

		return $cart;
	}

	/**
	 * @param Entity\CartItem[] $cartItemsRaw
	 *
	 * @return ArrayCollection(Dao\CartItem) polozky
	 */
	protected function fillDaoItems($cartItemsRaw)
	{
		$items = [];
		foreach ($cartItemsRaw as $ci) {
			$product = $this->productsService->get($ci->getProductId());

			if ($product) {
				$this->productsService->loadVariants($product);
				$this->tagsService->loadTagsToProduct($product);
			}
			$variant = null;
			if($ci->getVariantId()) {
				$variant = $this->productsService->getVariant((int) $ci->getVariantId()); //varianta
			}
			
			//TODO kdyz produkt neexistuje, odstranit z kosiku - to zaridi to napojeni na entitu, nejen pres ID
			if ($product) {
				$cartItem = new Dao\CartItem();

				$cartItem
					->setId($ci->getId())
					->setCartId($ci->cart->getId())
					->setProductId($ci->getProductId())
					->setProduct($product)
					->setVariantId($ci->getVariantId())
					->setProductVariant($variant)
					->setQuantity($ci->getQuantity());
				$items[$ci->getId()] = $cartItem;
			}
		}

		return $items;
	}

}

