<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\WebColors;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;

/**
 * @method Order|object|null getReference($id)
 * @method Order[]|null getAll()
 * @method Order|null get($id)
 */
class Orders extends BaseEntityService
{
	protected $entityClass = Order::class;

	/** @var EventDispatcher */
	protected EventDispatcher $eventDispatcher;

	/** @var WebColors */
	protected WebColors $webColors;

	public function __construct(EventDispatcher $eventDispatcher, WebColors $webColors)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->webColors       = $webColors;
	}

	/**
	 * @return OrderExpedition[]
	 */
	public function getOrdersForExpedition(): array
	{
		$data  = [];
		$stats = [];

		$qb = $this->em->createQueryBuilder()->select('IDENTITY(os.status) as ident, IDENTITY(os.order) as order')
			->from(OrderStatus::class, 'os')
			->orderBy('os.created', 'desc');

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$stats[$row['order']][] = $row['ident'];
		}

		$processStats = EshopOrdersConfig::load('expeditionOrdersGrid.status');
		$ids          = [];
		foreach ($stats as $orderId => $v) {
			if (in_array($v[0], $processStats)) {
				$ids[] = $orderId;
			}
		}

		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(os.order) as id, oss.ident')
			         ->from(OrderSpedition::class, 'os')
			         ->leftJoin('os.spedition', 'oss')
			         ->andWhere('os.order IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->groupBy('os.id')
			         ->orderBy('os.id', 'asc')->getQuery()->getArrayResult() as $row) {
			$data[] = [
				'id'      => $row['id'],
				'service' => $row['ident'],
			];
		}

		$event = new Event([
			'data'       => $data,
			'expedition' => [],
		]);
		$this->eventDispatcher->dispatch($event, Orders::class . '::processOrdersForExpedition');

		foreach ($event->data['expedition'] as &$expedition) {
			/** @var OrderExpedition $expedition */
			$expedition->siteColors = $this->webColors->getColors($expedition->order->site->getIdent());
		}

		ksort($event->data['expedition']);

		return $event->data['expedition'];
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	public function sendExpeditionData(array $ids, int $quantity = 1, array $advancedOptions = []): array
	{
		$event = new Event([
			'data'            => $this->getOrderIdsWithSpeditionCode($ids),
			'quantity'        => $quantity,
			'advancedOptions' => $advancedOptions,
			'ok'              => [],
			'okCount'         => 0,
			'errors'          => [],
			'errorCount'      => 0,
		]);

		$this->eventDispatcher->dispatch($event, Orders::class . '::sendExpeditionData');

		$arr = [
			'okCount'    => $event->data['okCount'],
			'errorCount' => $event->data['errorCount'],
		];

		if (isset($event->data['message'])) {
			$arr['message'] = $event->data['message'];
		}

		return $arr;
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	public function generateLabel(array $ids)
	{
		$event = new Event([
			'data'       => $this->getOrderIdsWithSpeditionCode($ids),
			'ok'         => [],
			'okCount'    => 0,
			'errors'     => [],
			'errorCount' => 0,
			'files'      => [],
		]);

		$this->eventDispatcher->dispatch($event, Orders::class . '::generateLabel');
		$this->eventDispatcher->dispatch($event, Orders::class . '::afterGenerateLabel');

		return [
			'okCount'    => $event->data['okCount'],
			'errorCount' => $event->data['errorCount'],
			'files'      => $event->data['files'],
		];
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	protected function getOrderIdsWithSpeditionCode(array $ids): array
	{
		$result = [];
		$qb     = $this->getEr()->createQueryBuilder('o')
			->select('o.id, oss.ident')
			->innerJoin('o.spedition', 'os')
			->leftJoin('os.spedition', 'oss')
			->andWhere('o.id IN (:ids)')
			->setParameter('ids', $ids);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$result[] = [
				'id'      => $row['id'],
				'service' => $row['ident'],
			];
		}

		return $result;
	}
}
