<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Country;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__customer_address")
 * @ORM\Entity
 */
class CustomerAddress
{
	use TId;

	/**
	 * @ORM\Column(name="address_name", type="string", length=255, nullable=true)
	 */
	public ?string $addressName = null;

	/**
	 * @var string
	 * @ORM\Column(name="first_name", type="string", nullable=true)
	 */
	public $firstName;

	/**
	 * @var string
	 * @ORM\Column(name="last_name", type="string", nullable=true)
	 */
	public $lastName;

	/**
	 * @var string
	 * @ORM\Column(name="email", type="string", length=255, nullable=true)
	 */
	public $email;

	/**
	 * @var string
	 * @ORM\Column(name="phone", type="string", length=255, nullable=true)
	 */
	public $phone;

	/**
	 * @var string
	 * @ORM\Column(name="company", type="string", nullable=true)
	 */
	public $company;

	/**
	 * @var string
	 * @ORM\Column(name="street", type="string", nullable=true)
	 */
	public $street;

	/**
	 * @var string
	 * @ORM\Column(name="city", type="string", nullable=true)
	 */
	public $city;

	/**
	 * @var string
	 * @ORM\Column(name="postal", type="string", nullable=true)
	 */
	public $postal;

	/**
	 * IČO
	 * @var string
	 * @ORM\Column(name="id_number", type="string", nullable=true)
	 */
	public $idNumber;

	/**
	 * DIČ
	 * @var string
	 * @ORM\Column(name="vat_number", type="string", nullable=true)
	 */
	public $vatNumber;

	/**
	 * @var Customer
	 * @ORM\ManyToOne(targetEntity="Customer", inversedBy="addressesAvailable")
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $customer;

	/**
	 * @var Country
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public $country;

	/**
	 * @ORM\Column(name="validated_id_number", type="smallint", length=1, options={"default": 0})
	 */
	public int $validatedIdNumber = 0;

	/**
	 * @ORM\Column(name="validated_vat_number", type="smallint", length=1, options={"default": 0})
	 */
	public int $validatedVatNumber = 0;

	/**
	 * CustomerAddress constructor.
	 *
	 * @param $customer
	 */
	public function __construct($customer)
	{
		$this->customer = $customer;
	}

	/*******
	 * === Customer
	 */

	public function getCustomer()
	{
		return $this->customer;
	}

	/*******
	 * === FirstName
	 */

	public function getFirstName()
	{
		return $this->firstName;
	}

	public function setFirstName($firstName)
	{
		$this->firstName = $firstName;

		return $this;
	}

	/*******
	 * === LastName
	 */

	public function getLastName()
	{
		return $this->lastName;
	}

	public function setLastName($lastName)
	{
		$this->lastName = $lastName;

		return $this;
	}

	public function getName(): string { return trim($this->getFirstName() . ' ' . $this->getLastName()); }

	/*******
	 * === Company
	 */

	public function getCompany()
	{
		return $this->company;
	}

	public function setCompany($company)
	{
		$this->company = $company;

		return $this;
	}

	/*******
	 * === Street
	 */

	public function getStreet()
	{
		return $this->street;
	}

	public function setStreet($street)
	{
		$this->street = $street;

		return $this;
	}

	/*******
	 * === City
	 */

	public function getCity()
	{
		return $this->city;
	}

	public function setCity($city)
	{
		$this->city = $city;

		return $this;
	}

	/*******
	 * === Postal
	 */

	public function getPostal()
	{
		return $this->postal;
	}

	public function setPostal($postal)
	{
		$this->postal = $postal;

		return $this;
	}

	/*******
	 * === IdNumber - ICO
	 */

	public function getIdNumber()
	{
		return $this->idNumber;
	}

	public function setIdNumber($idNumber): self
	{
		$this->idNumber = $idNumber;

		return $this;
	}

	/*******
	 * === VatNumber - DIC
	 */

	public function getVatNumber()
	{
		return $this->vatNumber;
	}

	public function setVatNumber($vatNumber): self
	{
		$this->vatNumber = $vatNumber;

		return $this;
	}

	public function getCountry(): ?Country { return $this->country; }

	public function setCountry(Country $country): self
	{
		$this->country = $country;

		return $this;
	}

	public function toArray(): array
	{
		return [
			'firstName' => $this->getFirstName(),
			'lastName'  => $this->getLastName(),
			'company'   => $this->company,
			'email'     => $this->email,
			'phone'     => $this->phone,
			'street'    => $this->getStreet(),
			'city'      => $this->getCity(),
			'postal'    => $this->getPostal(),
			'country'   => $this->country ? $this->country->getId() : null,
			'idNumber'  => $this->idNumber,
			'vatNumber' => $this->vatNumber,
		];
	}

	public function fillFromOrderAddress(OrderAddress $address): self
	{
		$this->setFirstName($address->getFirstName());
		$this->setLastName($address->getLastName());
		$this->setCompany($address->getCompany());
		$this->email     = $address->getEmail();
		$this->phone     = $address->getPhone();
		$this->street    = $address->getStreet();
		$this->city      = $address->getCity();
		$this->postal    = $address->getPostal();
		$this->country   = $address->getCountry();
		$this->idNumber  = $address->getIdNumber();
		$this->vatNumber = $address->getVatNumber();

		return $this;
	}
}
