<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities\Invoice;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Invoice;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__invoice_data")
 * @ORM\Entity
 */
class InvoiceData
{
	use TId;

	/**
	 * @var ArrayCollection|Product[]
	 *
	 * @ORM\OneToMany(targetEntity="Product", mappedBy="invoiceData", cascade={"persist"})
	 * @ORM\OrderBy({"name" = "ASC"})
	 */
	public $products;

	/**
	 * @var Customer
	 * @ORM\OneToOne(targetEntity="Customer", cascade={"persist"})
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $customer;

	/**
	 * @var ArrayCollection|Discount[]
	 *
	 * @ORM\OneToMany(targetEntity="Discount", mappedBy="invoiceData", cascade={"persist"})
	 */
	public $discounts;

	/**
	 * @var string
	 * @ORM\Column(type="string")
	 */
	public $lang;

	/**
	 * @var string
	 * @ORM\Column(type="string")
	 */
	public $currency;

	/**
	 * @ORM\Column(name="zero_vat", type="smallint", length=1, options={"default": 0})
	 */
	public int $zeroVat = 0;

	/**
	 * @var Supplier
	 * @ORM\OneToOne(targetEntity="Supplier", cascade={"persist"})
	 * @ORM\JoinColumn(name="supplier_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $supplier;

	/**
	 * @var Spedition
	 * @ORM\OneToOne(targetEntity="Spedition", inversedBy="invoiceData", cascade={"persist"})
	 * @ORM\JoinColumn(name="spedition_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $spedition;

	/**
	 * @var Payment
	 * @ORM\OneToOne(targetEntity="Payment", inversedBy="invoiceData", cascade={"persist"})
	 * @ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $payment;

	/**
	 * @var Invoice
	 * @ORM\OneToOne(targetEntity="EshopOrders\Model\Entities\Invoice", cascade={"persist"})
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $invoice;

	/**
	 * InvoiceData constructor.
	 *
	 * @param Customer  $customer
	 * @param Supplier  $supplier
	 * @param Spedition $spedition
	 * @param Payment   $payment
	 * @param Invoice   $invoice
	 */
	public function __construct(Customer $customer, Supplier $supplier, Spedition $spedition, Payment $payment, Invoice $invoice)
	{
		$this->customer  = $customer;
		$this->supplier  = $supplier;
		$this->spedition = $spedition;
		$this->payment   = $payment;
		$this->invoice   = $invoice;
		$this->discounts = new ArrayCollection();
	}

	/**
	 * @return Customer
	 */
	public function getCustomer(): Customer
	{
		return $this->customer;
	}

	/**
	 * @return Supplier
	 */
	public function getSupplier(): Supplier
	{
		return $this->supplier;
	}

	/**
	 * @return Spedition
	 */
	public function getSpedition(): Spedition
	{
		return $this->spedition;
	}

	/**
	 * @return Payment
	 */
	public function getPayment(): Payment
	{
		return $this->payment;
	}

	public function getPrice()
	{
		$priceTotal = $this->getPriceItems();
		$priceTotal += $this->getSpedition()->getPrice();
		$priceTotal += $this->getPayment()->getPrice();
		if ($this->discounts) {
			foreach ($this->discounts as $discount) {
				$priceTotal += $discount->calculateDiscount($priceTotal);
			}
		}

		return $priceTotal;
	}

	public function getPriceItems()
	{
		$priceTotal = 0;
		foreach ($this->products as $item) {
			$priceTotal += $item->getPrice() * $item->getQuantity();
		}

		return $priceTotal;
	}

	public function getPriceItemsDiscount()
	{
		$priceTotal = $this->getPriceItems();
		if ($this->discounts) {
			foreach ($this->discounts as $discount) {
				$priceTotal += $discount->getPrice();
			}
		}

		return $priceTotal;
	}

	public function getPriceWithoutVat()
	{
		$priceTotal = $this->calculateDiscounts($this->getItemsPriceWithoutVat());
		$priceTotal += $this->getPaySpedPriceWithoutVat();

		return $priceTotal;
	}

	public function getItemsPriceWithoutVat(): float
	{
		$priceTotal = 0;
		foreach ($this->products as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getPaySpedPrice()
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition()->getPrice();
		$priceTotal += $this->getPayment()->getPrice();

		return $priceTotal;
	}

	public function getPaySpedPriceWithoutVat(bool $useCurrency = false)
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition()->getPriceWithoutVat();
		$priceTotal += $this->getPayment()->getPriceWithoutVat();

		return $priceTotal;
	}

	public function getPaySpedId(): string { return $this->getSpedition()->speditionId . '-' . $this->getPayment()->paymentId; }

	public function getPaySpedName(): string { return $this->getSpedition()->name . ' - ' . $this->getPayment()->name; }

	public function calculateDiscounts(float $price): float
	{
		foreach ($this->discounts as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	public function getPaySpedVatRate(): int { return $this->zeroVat ? 0 : ($this->getSpedition()->getVatRate() ?: 21); }

	public function getVatRates(): array
	{
		$list = [];

		foreach ($this->products as $item) {
			$vat = $item->getVatRate();

			if (!isset($list[$vat]))
				$list[$vat] = [
					'withoutVat' => 0,
					'total'      => 0,
				];

			$list[$vat]['withoutVat'] += $item->getTotalPriceWithoutVat();
			$list[$vat]['total']      += $item->getPriceTotal();
		}

		if (!isset($list[$this->getPaySpedVatRate()])) {
			$list[$this->getPaySpedVatRate()] = [
				'withoutVat' => 0,
				'total'      => 0,
			];
		}

		$list[$this->getPaySpedVatRate()]['withoutVat'] += $this->getPaySpedPriceWithoutVat();
		$list[$this->getPaySpedVatRate()]['total']      += $this->getPaySpedPrice();

		foreach ($list as $vat => $prices) {
			$list[$vat]['rate'] = round($prices['total'] - $prices['withoutVat'], 2);
		}

		return $list;
	}
}
