<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Parameters;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Currency\DI\CurrencyExtension;
use Currency\Model\Config as CurrencyConfig;
use Core\Model\Entities\Site;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_orders__order")
 * @ORM\Entity
 * @ORM\EntityListeners({"OrderListener"})
 */
class Order
{
	use TId;

	/**
	 * @var string
	 * @ORM\Column(name="ident", type="string", length=255, nullable = false)
	 */
	protected $ident;

	/**
	 * @var Customer
	 * @ORM\ManyToOne(targetEntity="Customer")
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $customer;

	/**
	 * @ORM\Column(name="message", type="text", nullable=true)
	 */
	protected ?string $message = null;

	/**
	 * @var OrderPayment
	 * @ORM\OneToOne(targetEntity="OrderPayment", cascade={"persist"})
	 * @ORM\JoinColumn(name="payment", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $payment;

	/**
	 * @var OrderSpedition
	 * @ORM\OneToOne(targetEntity="OrderSpedition", cascade={"persist"})
	 * @ORM\JoinColumn(name="spedition", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $spedition;

	/**
	 * @var OrderItem[]
	 * @ORM\OneToMany(targetEntity="OrderItem", mappedBy="order", indexBy="id")
	 */
	protected $orderItems;

	/**
	 * @var OrderDiscount[]
	 * @ORM\OneToMany(targetEntity="OrderDiscount", mappedBy="order", indexBy="id")
	 */
	protected $orderDiscounts;

	/**
	 * @var OrderAddress
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_delivery_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $addressDelivery;

	/**
	 * @var OrderAddress
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $addressInvoice;

	/**
	 * @var ArrayCollection|OrderStatus[]
	 * @ORM\OneToMany(targetEntity="OrderStatus", mappedBy="order", cascade={"persist"})
	 */
	protected $orderStatuses;

	/**
	 * @var boolean
	 * @ORM\Column(name="agreed_terms", type="boolean", nullable=true)
	 */
	protected $agreedTerms;

	/**
	 * @var OrderFlag[]
	 * @ORM\OneToMany(targetEntity="OrderFlag", mappedBy="order", indexBy="type")
	 */
	protected $orderFlags;

	/**
	 * @var Invoice|null
	 * @ORM\OneToOne(targetEntity="Invoice", inversedBy="order")
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $invoice;

	/**
	 * @var Site
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Site")
	 * @ORM\JoinColumn(name="site_id", referencedColumnName="ident", onDelete="SET NULL", nullable=true)
	 */
	public $site;

	/**
	 * @var OrderCurrency|null
	 * @ORM\OneToOne(targetEntity="OrderCurrency", mappedBy="order", cascade={"persist"})
	 */
	public $currency;

	/**
	 * @var OrderGift[]
	 * @ORM\OneToMany(targetEntity="OrderGift", mappedBy="order")
	 */
	protected $gifts;

	/**
	 * @ORM\Column(name="is_paid", type="smallint", options={"default": 0})
	 */
	public int $isPaid = 0;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public $paid;

	/**
	 * @var self|null
	 * @ORM\OneToOne(targetEntity="Order", inversedBy="orderForCorrectiveTaxDocument")
	 * @ORM\JoinColumns({
	 *  @ORM\JoinColumn(referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 * })
	 */
	public $correctiveTaxDocumentOf;

	/**
	 * @var self|null
	 * @ORM\OneToMany(targetEntity="Order", mappedBy="correctiveTaxDocumentOf")
	 */
	public $orderForCorrectiveTaxDocument;

	/**
	 * @ORM\Column(type="smallint", options={"default": 0})
	 */
	public int $isCorrectiveTaxDocument = 0;

	/**
	 * @ORM\Column(name="lang", type="string", nullable=false, options={"default": "cs"})
	 */
	public string $lang;

	/** @var PriceFilter */
	public $priceFilter;

	public function __construct(Site $site)
	{
		$this->ident                         = $this->genUuid();
		$this->site                          = $site;
		$this->orderItems                    = new ArrayCollection();
		$this->orderDiscounts                = new ArrayCollection();
		$this->orderFlags                    = new ArrayCollection();
		$this->gifts                         = new ArrayCollection();
		$this->orderStatuses                 = new ArrayCollection();
		$this->addressDelivery               = null;
		$this->addressInvoice                = null;
		$this->orderForCorrectiveTaxDocument = new ArrayCollection();
		$this->lang                          = (string) Parameters::load('translation.default', 'cs');
	}

	public function getIdent()
	{
		return $this->ident;
	}

	public function getOrderForCorrectiveTaxDocument()
	{
		if ($this->orderForCorrectiveTaxDocument) {
			if ($this->orderForCorrectiveTaxDocument instanceof Order)
				return $this->orderForCorrectiveTaxDocument;

			$first = $this->orderForCorrectiveTaxDocument->first();
			if ($first !== false)
				return $first;
		}

		return null;
	}

	/**
	 * @return Customer
	 */
	public function getCustomer()
	{
		return $this->customer;
	}

	public function getMessage()
	{
		return $this->message;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		$priceTotal = $this->getPriceItems($useCurrency);

		foreach ($this->orderDiscounts as $discount) {
			$priceTotal -= abs($discount->getPrice($useCurrency));
		}

		$priceTotal += ($this->spedition ? $this->getSpedition()->getPrice($useCurrency) : 0)
			+ ($this->payment ? $this->getPayment()->getPrice($useCurrency) : 0);

		return $priceTotal;
	}

	public function getPriceItems(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->orderItems as $item) {
			if (!$item->getParent())
				$priceTotal += $item->getPriceTotalWithChilds($useCurrency);
		}

		return $priceTotal;
	}

	public function getPriceItemsDiscount(bool $useCurrency = false): float
	{
		$priceTotal = $this->getPriceItems($useCurrency);
		if ($this->orderDiscounts) {
			foreach ($this->orderDiscounts as $discount) {
				$priceTotal -= abs($discount->getPrice($useCurrency));
			}
		}

		return $priceTotal;
	}

	public function getPriceWithoutVat(bool $useCurrency = false)
	{
		$itemsPrice          = $this->getPriceItems($useCurrency) + $this->getPaySpedPrice($useCurrency);
		$itemsPriceAfterSale = $itemsPrice;

		foreach ($this->orderDiscounts as $discount)
			$itemsPriceAfterSale -= abs($discount->getPrice($useCurrency));

		$dphRates = [];
		foreach ($this->getVatRates($useCurrency) as $dph => $values)
			$dphRates[$dph] = $values['total'];

		return OrderHelper::calculatePriceWithoutVatWithSales($itemsPrice, $itemsPriceAfterSale, $dphRates);
	}

	public function getItemsPriceWithoutVat(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat($useCurrency);
		}

		return $priceTotal;
	}

	public function getItemsVatPrice(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getPriceTotal($useCurrency) - $item->getTotalPriceWithoutVat($useCurrency);
		}

		return $priceTotal;
	}

	public function getPaySpedPrice(bool $useCurrency = false)
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition() ? $this->getSpedition()->getPrice($useCurrency) : 0;
		$priceTotal += $this->getPayment() ? $this->getPayment()->getPrice($useCurrency) : 0;

		return $priceTotal;
	}

	public function getPaySpedPriceWithoutVat(bool $useCurrency = false)
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition() ? $this->getSpedition()->getPriceWithoutVat($useCurrency) : 0;
		$priceTotal += $this->getPayment() ? $this->getPayment()->getPriceWithoutVat($useCurrency) : 0;

		return $priceTotal;
	}

	public function getPaySpedId(): string { return $this->getSpedition()->getId() . '-' . $this->getPayment()->getId(); }

	public function getPaySpedName(): string { return $this->getSpedition()->getName() . ' - ' . $this->getPayment()->getName(); }

	/**
	 * @return OrderPayment
	 */
	public function getPayment()
	{
		if ($this->payment && !$this->payment->order)
			$this->payment->order = $this;

		return $this->payment;
	}

	public function getPaymentIdent(): ?string
	{
		return $this->getPayment() && $this->getPayment()->getPayment() ? $this->getPayment()->getPayment()->getIdent() : null;
	}

	public function getPaySpedVatRate(): int { return $this->isZeroVat() ? 0 : $this->spedition->getVatRate(); }

	/**
	 * @return OrderSpedition
	 */
	public function getSpedition()
	{
		if ($this->spedition && !$this->spedition->order)
			$this->spedition->order = $this;

		return $this->spedition;
	}

	public function getSpeditionIdent()
	{
		return $this->getSpedition() && $this->getSpedition()->getSpedition() ? $this->getSpedition()->getSpedition()->getIdent() : null;
	}

	/**
	 * @return OrderItem[]
	 */
	public function getOrderItems()
	{
		return $this->orderItems;
	}

	public function getOrderItemByProductId(int $productId): ?OrderItem
	{
		foreach ($this->getOrderItems() as $item) {
			if ($item->getProductId() === $productId) {
				return $item;
			}
		}

		return null;
	}

	/**
	 * @return OrderAddress|null
	 */
	public function getAddressDelivery()
	{
		return $this->addressDelivery ?: $this->addressInvoice;
	}

	public function getAddressDeliveryRaw()
	{
		return $this->addressDelivery;
	}

	/**
	 * @return OrderAddress|null
	 */
	public function getAddressInvoice()
	{
		return $this->addressInvoice ?: ($this->addressDelivery ?: null);
	}

	public function getAddressInvoiceRaw()
	{
		return $this->addressInvoice;
	}

	/**
	 * @param Customer $customer
	 *
	 * @return Order
	 */
	public function setCustomer(Customer $customer): Order
	{
		$this->customer = $customer;

		return $this;
	}

	public function setMessage($message)
	{
		$this->message = $message;

		return $this;
	}

	/**
	 * @param OrderPayment $payment
	 *
	 * @return Order
	 */
	public function setPayment($payment)
	{
		$this->payment = $payment;

		return $this;
	}

	/**
	 * @param OrderSpedition $spedition
	 *
	 * @return Order
	 */
	public function setSpedition($spedition)
	{
		$this->spedition = $spedition;

		return $this;
	}

	public function setOrderItems(array $orderItems)
	{
		$this->orderItems = new ArrayCollection($orderItems);

		return $this;
	}

	public function setAddressDelivery(OrderAddress $addressDelivery)
	{
		$this->addressDelivery = $addressDelivery;

		return $this;
	}

	public function setAddressInvoice(OrderAddress $addressInvoice)
	{
		$this->addressInvoice = $addressInvoice;

		return $this;
	}

	/*******
	 * === OrderDiscounts
	 */

	public function getOrderDiscounts()
	{
		return $this->orderDiscounts;
	}

	public function setOrderDiscounts($orderDiscounts): Order
	{
		$this->orderDiscounts = $orderDiscounts;

		return $this;
	}

	public function addOrderDiscount($orderDiscount): Order
	{
		$this->orderDiscounts->add($orderDiscount);

		return $this;
	}

	protected function calculateDiscounts(float $price): float
	{
		foreach ($this->getOrderDiscounts() as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	/*******
	 * === AgreedTerms
	 */

	public function getAgreedTerms()
	{
		return $this->agreedTerms;
	}

	public function setAgreedTerms($agreedTerms): Order
	{
		$this->agreedTerms = $agreedTerms;

		return $this;
	}

	/*******
	 * === OrderFlags
	 */

	public function getOrderFlags()
	{
		return $this->orderFlags;
	}

	public function setOrderFlags($orderFlags): Order
	{
		if (is_array($orderFlags))
			$orderFlags = new ArrayCollection($orderFlags);
		$this->orderFlags = $orderFlags;

		return $this;
	}

	public function addFlag(OrderFlag $flag)
	{
		if (!$this->orderFlags->containsKey($flag->getType())) {
			$this->orderFlags->set($flag->getType(), $flag);
		}
	}

	public function hasFlag($type)
	{
		return ($this->orderFlags->containsKey($type) && $this->orderFlags[$type] == true);
	}

	/*******
	 * === OrderStatuses
	 */

	/**
	 * @return OrderStatus[]|ArrayCollection
	 */
	public function getOrderStatuses()
	{
		return $this->orderStatuses;
	}

	/**
	 * @return OrderStatus
	 */
	public function getNewestOrderStatus()
	{
		return $this->orderStatuses->last() ?: null;
	}

	public function setOrderStatuses($orderStatuses): Order
	{
		$this->orderStatuses = $orderStatuses;

		return $this;
	}

	public function getCreatedTime()
	{
		foreach ($this->orderStatuses as $orderStatus) {
			if ($orderStatus->getStatus()->getId() == 'created') {
				$createdStatus = $orderStatus;
				break;
			}
		}
		if ($createdStatus) {
			return $createdStatus->getCreated();
		}

		return false;
	}

	public function getInvoice(): ?Invoice
	{
		return $this->invoice;
	}

	public function setInvoice(?Invoice $invoice)
	{
		$this->invoice = $invoice;
	}

	public function getCurrencyCode(): string
	{
		if (class_exists(CurrencyExtension::class) && $this->currency)
			return $this->currency->code;

		return $this->getDefaultCurrency();
	}

	public function getDefaultCurrency() { return class_exists(CurrencyConfig::class) ? CurrencyConfig::load('default') : 'CZK'; }

	public function calculateCurrencyPrice(float $price): float
	{
		return $this->currency
			? OrderHelper::calculateCurrencyPrice($price, (float) $this->currency->rate, (int) $this->currency->decimals)
			: $price;
	}

	public ?int $renderPriceDecimals = null;

	public function renderPrice(float $price, bool $useCurrency = false, ?string $method = null, ?bool $currencyFromLeft = null, ?string $separator = null, ?string $decSep = null, ?int $decimals = null)
	{
		if (!$method)
			$method = 'format';
		if (!$decimals && $this->renderPriceDecimals)
			$decimals = $this->renderPriceDecimals;

		return $this->priceFilter->$method($price, $useCurrency ? $this->getCurrencyCode() : $this->getDefaultCurrency(), $currencyFromLeft, $separator, $decSep, $decimals);
	}

	/*************************************
	 * == Gifts
	 */

	/** @return ArrayCollection|OrderGift[] */
	public function getGifts() { return $this->gifts; }

	public function addGift(OrderGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}

	/** http://stackoverflow.com/a/2040279
	 * @return string UUID - unikatni ID, tezke na uhodnuti
	 */
	protected function genUuid()
	{
		return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff),

			// 16 bits for "time_mid"
			mt_rand(0, 0xffff),

			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand(0, 0x0fff) | 0x4000,

			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand(0, 0x3fff) | 0x8000,

			// 48 bits for "node"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
		);
	}

	public function getVatRates(bool $useCurrency = false): array
	{
		$list = [];

		foreach ($this->getOrderItems() as $item) {
			$vat = $item->getVatRate();

			if (!isset($list[$vat]))
				$list[$vat] = [
					'withoutVat' => 0,
					'total'      => 0,
				];

			$list[$vat]['withoutVat'] += $item->getTotalPriceWithoutVat($useCurrency);
			$list[$vat]['total']      += $item->getPriceTotal($useCurrency);
		}

		$paySpedVatRate = $this->getPaySpedVatRate();

		if (!isset($list[$paySpedVatRate])) {
			$list[$paySpedVatRate] = [
				'withoutVat' => 0,
				'total'      => 0,
			];
		}

		$list[$paySpedVatRate]['withoutVat'] += $this->getPaySpedPriceWithoutVat($useCurrency);
		$list[$paySpedVatRate]['total']      += $this->getPaySpedPrice($useCurrency);

		foreach ($list as $vat => $prices) {
			$list[$vat]['rate'] = round($prices['total'] - $prices['withoutVat'], 2);
		}

		return $list;
	}

	public function isZeroVat(): bool
	{
		return $this->getAddressInvoice() && $this->getAddressInvoice()->getCountry()
			&& in_array(strtolower($this->getAddressInvoice()->getCountry()->getId()), EshopOrdersConfig::load('invoice.zeroVatIfVatNumberValidatedFor'))
			&& $this->getAddressInvoice()->validatedVatNumber;
	}

	public function getPaidDate(): ?\DateTimeInterface { return $this->isPaid ? $this->paid : null; }

	public function __clone()
	{
		$this->id    = null;
		$this->ident = $this->genUuid();
	}

	public function isCanceled(): bool
	{
		$isCanceled = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$isCanceled = true;
			}
		}

		return $isCanceled;
	}

	public function getItemsWeight(bool $inKg = true): float
	{
		$weight = 0;

		foreach ($this->getOrderItems() as $orderItem) {
			if ($orderItem->getProduct() && $orderItem->getProduct()->weight)
				$weight += $orderItem->getProduct()->weight;
		}

		return round($inKg ? $weight / 1000 : $weight, 2);
	}
}
