<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use Core\Model\Entities\TId;
use EshopCatalog\Model\Entities\Supplier;
use EshopOrders\Model\Helpers\OrderHelper;
use Nette\Utils\Html;

/**
 * @ORM\Table("eshop_orders__order_item")
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener"})
 */
class OrderItem
{
	use TId;
	use TTranslateListener;

	/**
	 * @var Product
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\Product")
	 * @ORM\JoinColumn(name="product_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $product;

	/**
	 * @var string
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	protected $code1;

	/**
	 * @var int
	 * @ORM\Column(name="quantity", type="smallint")
	 */
	protected $quantity;

	/**
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @var int
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	public $vatRate;

	/**
	 * @var double
	 * @ORM\Column(name="recycling_fee", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $recyclingFee;

	/**
	 * @var Order
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderItems")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $order;

	/**
	 * @var OrderItemTexts[]
	 * @ORM\OneToMany(targetEntity="OrderItemTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 * TODO moznost optimalizace - nekaskadovat
	 */
	protected $orderItemTexts;

	/**
	 * @var OrderItemGift[]
	 * @ORM\OneToMany(targetEntity="OrderItemGift", mappedBy="orderItem")
	 */
	protected $gifts;

	/**
	 * @var OrderItemSale[]
	 * @ORM\OneToMany(targetEntity="OrderItemSale", mappedBy="orderItem")
	 */
	public $sales;

	/**
	 * @var array
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected $moreData = [];

	/**
	 * @var self[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="OrderItem", mappedBy="parent")
	 */
	protected $children;

	/**
	 * @ORM\ManyToOne(targetEntity="OrderItem", inversedBy="children")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?OrderItem $parent = null;

	public function __construct(?Product $product, Order $order)
	{
		$this->product        = $product;
		$this->order          = $order;
		$this->quantity       = 1;
		$this->orderItemTexts = new ArrayCollection();
		$this->moreData       = [];
		$this->gifts          = new ArrayCollection();
		$this->children       = new ArrayCollection();
	}

	/*****
	 * === Quantity
	 */

	public function setQuantity($quantity)
	{
		$this->quantity = $quantity && $quantity > 0 ? $quantity : null;
	}

	public function getQuantity() { return $this->quantity; }

	public function getQuantityByParent() { return $this->getParent() ? $this->getParent()->getQuantity() * $this->getQuantity() : $this->getQuantity(); }

	/*******
	 * === Product
	 */

	public function getProduct()
	{
		return $this->product;
	}

	public function getProductId()
	{
		return $this->product ? $this->product->getId() : null;
	}

	public function setProductId($productId)
	{
		$this->productId = $productId;

		return $this;
	}

	public function getManufacturer(): ?Manufacturer { return $this->getProduct() ? $this->getProduct()->getManufacturer() : null; }

	public function getProductNameWithoutManufacturer(): string
	{
		$manu = $this->getManufacturer();
		$text = $this->getOrderItemText();

		if (!$text || !$manu)
			return $text->getName();

		return Strings::removeFromStart($text->getName(), $manu->name);
	}

	/*******
	 * === Code1
	 */

	public function getCode1()
	{
		return $this->code1;
	}

	public function setCode1($code1): OrderItem
	{
		$this->code1 = $code1;

		return $this;
	}

	/*******
	 * === Order
	 */

	public function getOrder(): Order
	{
		return $this->order;
	}

	public function setOrder(Order $order)
	{
		$this->order = $order;

		return $this;
	}

	/*******
	 * === Price
	 */

	public function getPrice(bool $useCurrency = false): float
	{
		if ($this->order->isZeroVat())
			return $this->getPriceWithoutVat($useCurrency);

		$price = (float) $this->price;

		return $useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice($price)
			: $price;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$price = (float) ($useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
			: $this->price);

		return OrderHelper::getItemPriceWithoutVat($price, (int) $this->vatRate);
	}

	public function getPriceWithChilds(bool $useCurrency = false): float
	{
		$price = $this->getPrice($useCurrency);

		foreach ($this->getChildren()->toArray() as $child) {
			/** @var self $child */
			$price += $child->getPriceTotal($useCurrency);
		}

		return $price;
	}

	public function getPriceWithChildsWithoutVat(bool $useCurrency = false): float
	{
		$price = $this->getPriceWithoutVat($useCurrency);

		foreach ($this->getChildren()->toArray() as $child) {
			/** @var self $child */
			$price += $child->getPriceTotalWithoutVat($useCurrency);
		}

		return $price;
	}

	public function getPriceWithoutSales(bool $useCurrency = false): float
	{
		return (float) ($this->getPrice($useCurrency) * $this->quantity);
	}

	public function getPriceTotal(bool $useCurrency = false): float
	{
		$price = $this->getPrice($useCurrency);
		$sales = [];

		foreach ($this->sales as $s)
			$sales[] = $s->getSaleValue($useCurrency);

		return OrderHelper::getItemPriceTotal($price, $this->getQuantity(), $sales);
	}

	public function getPriceTotalWithoutVat(bool $useCurrency = false): float
	{
		$price = $this->getPriceWithoutVat($useCurrency);
		$sales = [];

		foreach ($this->sales as $s)
			$sales[] = $s->getSaleValue($useCurrency);

		return OrderHelper::getItemPriceTotal($price, $this->getQuantity(), $sales);
	}

	public function getPriceTotalWithChilds(bool $useCurrency = false): float
	{
		$price = $this->getPrice($useCurrency);

		foreach ($this->getChildren()->toArray() as $child) {
			/** @var self $child */
			$price += $child->getPriceTotal($useCurrency);
		}

		$price *= $this->getQuantity();

		foreach ($this->sales as $s)
			$price -= $s->getSaleValue($useCurrency);

		return $price;
	}

	public function getPriceTotalWithChildsWithoutVat(bool $useCurrency = false): float
	{
		$price = $this->getPriceWithoutVat($useCurrency);

		foreach ($this->getChildren()->toArray() as $child) {
			/** @var self $child */
			$price += $child->getPriceTotalWithoutVat($useCurrency);
		}

		$price *= $this->getQuantity();

		foreach ($this->sales as $s)
			$price -= $s->getSaleValue($useCurrency);

		return $price;
	}

	public function getPriceWithoutVatWithoutChilds(bool $useCurrency = false): float
	{
		$price = $useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
			: $this->price;

		return OrderHelper::removeVat($price, $this->vatRate);
	}

	public function getTotalPriceWithoutVat(bool $useCurrency = false): float
	{
		$sales = [];
		foreach ($this->sales as $s)
			$sales[] = $s->getSaleValue($useCurrency);

		return OrderHelper::getItemPriceTotalWithoutVat($this->getPrice($useCurrency), $this->getQuantity(), $this->vatRate, $sales);
	}

	public function setPrice(float $price)
	{
		// Asi bug, float neprojde
		$price = str_replace(',', '.', $price);

		$this->price = $price;

		return $this;
	}

	/*******
	 * === VatRate
	 */

	public function getVatRate()
	{
		return $this->getOrder()->isZeroVat() ? 0 : $this->vatRate;
	}

	public function setVatRate($vatRate): OrderItem
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	/*******
	 * === OrderItemText
	 */

	public function addOrderItemText($lang)
	{
		$this->orderItemTexts->set($lang, new OrderItemTexts($this, $lang));
	}

	public function setOrderItemText(OrderItemTexts $orderItemTexts)
	{
		$this->orderItemTexts->set($orderItemTexts->getLang(), $orderItemTexts);
	}

	/**
	 * @param string|null $lang
	 *
	 * @return OrderItemTexts|null
	 */
	public function getOrderItemText(?string $lang = null): ?OrderItemTexts
	{
		return $this->orderItemTexts->get($lang ?: $this->locale) ?? $this->orderItemTexts->first();
	}


	/*************************************
	 * == MoreData
	 */

	/**
	 * @param string $key
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue($key)
	{
		return $this->moreData[$key] ?? null;
	}

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function setMoreDataValue($key, $value)
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param array $data
	 *
	 * @return $this
	 */
	public function setMoreData($data)
	{
		$this->moreData = $data;

		return $this;
	}

	/*************************************
	 * == Gifts
	 */

	/** @return ArrayCollection|OrderItemGift[] */
	public function getGifts()
	{
		return $this->gifts;
	}

	public function addGift(OrderItemGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}

	/*************************************
	 * == Children
	 */

	public function getChildren() { return $this->children ?: new ArrayCollection(); }

	public function getParent(): ?OrderItem { return $this->parent; }

	public function setParent(OrderItem $parent)
	{
		$this->parent = $parent;
		$parent->getChildren()->add($this);
	}

	/*************************************
	 * == Recycling Fee
	 */
	public function getRecyclingFeeWithoutVat(): float
	{
		$price = (float) $this->recyclingFee;

		return $useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice($price)
			: $price;
	}

	public function getTotalRecyclingFeeWithoutVat(): float
	{
		return $this->getQuantity() * $this->getRecyclingFeeWithoutVat();
	}

	public function getSuppliersLinks(): ?Html
	{
		$product = $this->getProduct();
		if (!$product)
			return null;

		$wrap = Html::el();
		foreach ($product->getSuppliers()->toArray() as $supp) {
			$supp = $supp->getSupplier();
			/** @var Supplier $supp */
			$div = Html::el('div');
			if ($supp->website)
				$div->addHtml(Html::el('a', [
					'href'   => $supp->website,
					'target' => '_blank',
				])->setText($supp->name));
			else
				$div->setText($supp->name);
			$wrap->addHtml($div);
		}

		return $wrap;
	}
}

