<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Event\OrderStatusEvent;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Nette\SmartObject;
use Core\Model\Templating\Filters\Price as PriceFilter;

class OrderListener implements EventSubscriberInterface
{
	use SmartObject;

	protected PriceFilter $priceFilter;

	protected EventDispatcher $eventDispatcher;

	public function __construct(PriceFilter $priceFilter, EventDispatcher $eventDispatcher)
	{
		$this->priceFilter     = $priceFilter;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param Order      Order
	 * @param LifecycleEventArgs $event
	 */
	public function postUpdateHandler(Order $order, LifecycleEventArgs $event): void
	{
		$em        = $event->getEntityManager();
		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($order);

		if ($changeSet && isset($changeSet['isPaid']) && $changeSet['isPaid'][1] === 1) {
			$order->paid = new DateTime();
			$this->eventDispatcher->dispatch(new OrderStatusEvent($order, OrderStatus::STATUS_IS_PAID), Order::class . '::changeStatus');
		}
	}

	/**
	 * @ORM\PostLoad
	 *
	 * @param Order              $order
	 * @param LifecycleEventArgs $args
	 */
	public function postLoad(Order $order, LifecycleEventArgs $args): void
	{
		$order->priceFilter = clone $this->priceFilter;

		$this->eventDispatcher->dispatch(new OrderEvent($order), Order::class . '::postLoad');
	}
}
