<?php declare(strict_types = 1);

namespace EshopOrders\Model\Helpers;

use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItemSale;
use EshopOrders\Model\Entities\OrderStatus;
use EshopSales\Model\Entities\OrderSale;

class OrderHelper
{
	public static function canSendTrackingUrl(string $statusId, ?Order $order = null): bool
	{
		return in_array($statusId, [OrderStatus::STATUS_SPEDITION]);
	}

	public static function calculateCurrencyPrice(float $price, float $rate, int $decimals): float
	{
		return round($price * (1 / $rate), $decimals);
	}

	public static function removeVat(float $price, int $vat): float
	{
		return round($price / (1 + ($vat / 100)), 2);
	}

	public static function getItemPriceWithoutVat(float $price, int $vatRate): float
	{
		return self::removeVat($price, $vatRate);
	}

	public static function getItemPriceTotalWithoutVat(float $price, int $quantity, int $vatRate, array $sales = []): float
	{
		$price = self::removeVat($price, $vatRate) * $quantity;

		foreach ($sales as $sale)
			$price -= $sale;

		return $price;
	}

	public static function getItemPriceTotal(float $price, int $quantity, array $sales = []): float
	{
		$price = $price * $quantity;

		foreach ($sales as $sale)
			$price -= $sale;

		return $price;
	}

	public static function calculateSaleValue(string $type, $value, int $quantity, float $basePrice)
	{
		switch ($type) {
			case OrderItemSale::TYPE_FIX_ALL:
				return $value;
			case OrderItemSale::TYPE_FIX_ONE:
				return $value * $quantity;
			case OrderItemSale::TYPE_PERCENT_ALL:
				return (($basePrice * $quantity) / 100) * $value;
			case OrderItemSale::TYPE_PERCENT_ONE:
				return (($basePrice / 100) * $value) * $quantity;
		}

		return $value;
	}

	public static function calculatePriceWithoutVatWithSales(float $totalPrice, float $totalPriceAfterSale, array $dphValues): float
	{
		$result = 0;

		foreach ($dphValues as $dph => $value)
			$result += (($totalPriceAfterSale * ($value / $totalPrice)) / (100 + $dph)) * 100;

		return $result;
	}
}
