<?php declare(strict_types = 1);

namespace EshopOrders\Model\Pdf;

use Core\Model\Pdf\Pdf;
use Core\Model\Sites;
use Currency\Model\Currencies;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use Mpdf\Mpdf;
use Mpdf\MpdfException;
use Mpdf\Output\Destination;
use Nette\Localization\ITranslator;
use Nette\Utils\Strings;

class InvoicePdfBuilder
{
	/** @var Invoice */
	protected $invoice;

	/** @var InvoiceConfigRepository */
	protected $invoiceConfigRepository;

	/** @var Invoices */
	protected $invoices;

	/** @var ITranslator */
	protected $translator;

	/** @var Pdf */
	protected $pdfService;

	protected Currencies $currencies;

	protected Sites $sites;

	public function __construct(Invoice $invoice, InvoiceConfigRepository $invoiceConfigRepository, Invoices $invoices,
	                            ITranslator $translator, Pdf $pdfService, Currencies $currencies, Sites $sites)
	{
		$this->invoice                 = $invoice;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->invoices                = $invoices;
		$this->translator              = $translator;
		$this->pdfService              = $pdfService;
		$this->currencies              = $currencies;
		$this->sites                   = $sites;
	}

	/**
	 * @param string $destination
	 *
	 * @return string
	 * @throws MpdfException
	 */
	public function render(string $destination = Destination::INLINE): string
	{
		$pdf = $this->build();

		return $pdf->Output($this->getFileName(), $destination);
	}

	/**
	 * @return Mpdf
	 * @throws MpdfException
	 */
	private function build(): Mpdf
	{
		// css
		$customCssFile  = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/invoiceStyle.css';
		$defaultCssFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/invoiceStyle.css';

		// latte
		$customTemplateFile  = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/invoice.latte';
		$defaultTemplateFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/invoice.latte';

		// build template
		$template = $this->pdfService->getTemplateFactory();
		$template->setFile(file_exists($customTemplateFile) ? $customTemplateFile : $defaultTemplateFile);
		$template->originTemplate = $defaultTemplateFile;

		$template->invoice = $this->invoice;

		$order                                      = $this->invoice->order;
		$order->renderPriceDecimals                 = 2;
		\Core\Model\Helpers\Strings::$priceKeepTail = true;

		$template->order    = $order;
		$template->supplier = $this->invoice->invoiceData->getSupplier();
		$template->bank     = $this->invoice->invoiceData->getSupplier()->getBank();

		$invoiceData = $this->invoice->invoiceData;

		$invoiceData->spedition->invoiceData = &$invoiceData;
		$invoiceData->payment->invoiceData   = &$invoiceData;

		$template->subscriber      = $invoiceData->getCustomer();
		$template->invoiceData     = $invoiceData;
		$template->defaultCurrency = $this->currencies->getDefaultCode();

		// service config
		$pdf = $this->pdfService->newPdf();
		$pdf->WriteHTML(file_get_contents(file_exists($customCssFile) ? $customCssFile : $defaultCssFile), 1);
		$pdf->WriteHTML($template->renderToString(), 0);
		$pdf->SetTitle($this->getTitle());

		$site = $this->sites->getSites()[$this->invoice->order->site->getIdent()];

		if ($site) {
			$pdf->SetHTMLFooter('<table style="border-top: 1px solid black"><tr><td style="padding-top: 1mm">E-shop ' . $site->getSiteName() . ' - ' . $site->getCurrentDomain()->getDomain() . '</td></tr></table>');
		}

		return $pdf;
	}

	/**
	 * @return string
	 */
	private function getTitle(): string
	{
		return $this->translator->translate('eshopOrders.title.invoice', null, ['id' => $this->invoice->ident]);
	}

	/**
	 * @return string
	 */
	public function getFileName(): string
	{
		return sprintf('%s.pdf', Strings::webalize($this->getTitle()));
	}

}
