<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Mailing\TemplateFactory;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\Dao\DaoOrderStatusEmail;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use EshopOrders\Model\Event\EmailEvent;
use EshopOrders\Model\Utils\Helpers;
use Nette\Localization\ITranslator;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Tracy\Debugger;
use Users\Model\Entities\User;
use Users\Model\Http\UserStorage;

/**
 * class Statuses
 * @package EshopOrders\Model
 *
 * @method Status|object|null getReference($id)
 * @method Status|null get($id)
 */
class Statuses extends BaseEntityService
{
	protected $entityClass = Status::class;

	/** @var ITranslator */
	protected ITranslator $translator;

	/** @var IMailer */
	protected IMailer $mailer;

	/** @var TemplateFactory */
	protected TemplateFactory $mailTemplateFactory;

	/** @var UserStorage */
	protected UserStorage $userStorage;

	/** @var EventDispatcher */
	protected EventDispatcher $eventDispatcher;

	protected Sellers $sellers;

	protected Sites $sites;

	protected Payments $paymentsService;

	public function __construct(ITranslator $translator, TemplateFactory $mailTemplateFactory, IMailer $mailer, UserStorage $userStorage,
	                            EventDispatcher $eventDispatcher, Sellers $sellers, Sites $sites, Payments $payments)
	{
		$this->translator          = $translator;
		$this->mailer              = $mailer;
		$this->mailTemplateFactory = $mailTemplateFactory;
		$this->userStorage         = $userStorage;
		$this->eventDispatcher     = $eventDispatcher;
		$this->sellers             = $sellers;
		$this->sites               = $sites;
		$this->paymentsService     = $payments;
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function getAll()
	{
		return $this->getEr()->findBy([], ['position' => 'ASC']);
	}

	public function changeStatus(array $ids, string $status, bool $sendEmail = true): bool
	{
		try {
			/** @var OrderStatus[] $statuses */
			$statuses = [];
			/** @var Order[] $orders */
			$orders = [];
			foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
				         ->where('o.id IN (:ids)')->setParameter('ids', $ids)
				         ->getQuery()->getResult() as $row)
				$orders[$row->getId()] = $row;

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->select('IDENTITY(os.order) as order, IDENTITY(os.status) as stat')
				         ->where('os.order IN (:ids)')
				         ->andWhere('os.deleted IS NULL')
				         ->setParameter('ids', $ids)
				         ->orderBy('os.created', 'asc')
				         ->getQuery()->getArrayResult() as $row) {
				if ($row['stat'] === $status)
					unset($orders[$row['order']]);
			}

			foreach ($ids as $id) {
				if (!$orders[$id])
					continue;

				$texts         = new DaoOrderStatusEmail();
				$texts->status = $status;

				if ($status === OrderStatus::STATUS_IS_PAID) {
					$entity         = $orders[$id];
					$entity->isPaid = 1;

					$texts->statusText = $this->translator->translate('eshopOrdersFront.emails.orderIsPaid');

					$this->em->persist($entity);
					$statuses[] = [
						'order'       => $entity,
						'orderStatus' => null,
						'statusId'    => $status,
						'texts'       => $texts,
					];
				} else {
					$entity         = new OrderStatus(
						$orders[$id],
						$this->getReference($status),
						$this->userStorage->getIdentity()
							? $this->em->getReference(User::class, $this->userStorage->getIdentity()->getId()) : null,
					);
					$texts->message = $entity->getMessage();

					$this->em->persist($entity);
					$statuses[] = [
						'order'       => $orders[$id],
						'orderStatus' => $entity,
						'statusId'    => $status,
						'texts'       => $texts,
					];
				}
			}
			$this->em->flush();

			if ($sendEmail) {
				foreach ($statuses as $v) {
					$this->sendOrderStatusEmail($v['order'], $v['orderStatus'], $v['texts'], $v['statusId']);
				}
			}
		} catch (\Exception $e) {
			Debugger::log($e, 'status-change');

			return false;
		}

		return true;
	}

	/**
	 * @param Order                    $order
	 * @param OrderStatus|null         $orderStatus
	 * @param DaoOrderStatusEmail|null $texts
	 */
	public function sendOrderStatusEmail(Order $order, ?OrderStatus $orderStatus = null, ?DaoOrderStatusEmail $texts = null, ?string $statusId = null): bool
	{
		try {
			if (!$orderStatus && !$texts || !EshopOrdersConfig::load('orderForm.send', false) || !$order->getAddressInvoice()->getEmail())
				return false;

			if (!$statusId && $orderStatus)
				$statusId = $orderStatus->getStatus()->getId();

			if (in_array($statusId, EshopOrdersConfig::load('orderStatus.excludeStatusEmail', []))) {
				return false;
			}

			// Vypnutí odeslání emailu po odeslání faktury pro určité stavyOrder
			if (EshopOrdersConfig::load('invoice.invoiceInEmailIsEndStatusEmail') && $statusId
				&& in_array($statusId, [OrderStatus::STATUS_IS_PAID, OrderStatus::STATUS_FINISHED])
				&& $order->hasFlag(OrderFlag::TYPE_INVOICE_SENT))
				return false;

			// Vypnutí odeslání emailu se stavem zaplaceno u dobírek
			if ($statusId === OrderStatus::STATUS_IS_PAID
				&& in_array((string) $order->getSpeditionIdent(), EshopOrdersConfig::load('disableIsPaidEmailForSpedition', [])))
				return false;

			if ($orderStatus && !$texts) {
				$texts          = new DaoOrderStatusEmail();
				$texts->status  = $orderStatus->getStatus()->getId();
				$texts->message = $orderStatus->getMessage();
			}

			$this->translator->setLocale($order->lang);
			Sites::$currentIdentOverride = $order->lang;

			$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/status.latte';
			$originFile = __DIR__ . '/../FrontModule/_emails/status.latte';
			$template   = $this->mailTemplateFactory->create($order->site->getIdent());
			$seller     = $this->sellers->getSellerForSite($order->site->getIdent());

			if (!file_exists($file))
				$file = $originFile;

			$subject = $this->translator->translate('eshopOrders.emails.subjectStatus',
				['orderId' => $order->getId(), 'siteName' => $template->siteName]);

			$template->setFile($file);
			$template->setParameters([
				'templateParts'  => __DIR__ . '/../FrontModule/_emails/_parts.latte',
				'subject'        => $subject,
				'order'          => $order,
				'orderStatus'    => $orderStatus,
				'texts'          => $texts,
				'orderId'        => $order->getId(),
				'originTemplate' => $originFile,
				'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
					&& $order->getPaymentIdent() === 'card'
					&& $this->paymentsService->getByIdent('card', true),
			]);

			$siteIdent   = $order->site->getIdent();
			$sellerEmail = Helpers::getSellerEmail($seller, $order->site);
			$sellerName  = Helpers::getSellerName($seller, $siteIdent);

			$message = new Message();
			$message->setFrom($sellerEmail, $sellerName ?? null);

			if ($seller->sendStatusToSeller)
				$message->addTo($sellerEmail, $sellerName ?? null);

			$bcc = EshopOrdersConfig::load('orderForm.bcc');
			if ($bcc) {
				foreach (explode(',', $bcc) as $v) {
					$message->addBcc(trim($v));
				}
			}

			$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

			$statusString = $orderStatus ? $orderStatus->getStatus()->getId() : $texts->status;
			if ($statusString)
				$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, $statusString, $orderStatus), 'eshopOrders.emailChangeOrderStatus');

			$message->setHtmlBody($template->renderToString());
			$this->mailer->send($message);
		} catch (\Exception $e) {
			Debugger::log('Order email failed - ' . $order->getId(), 'eshopOrderEmail');
			Debugger::log($e, 'eshopOrderEmail');
		}

		return true;
	}
}
