<?php declare(strict_types = 1);

namespace EshopOrders\Model\Utils;

use Core\Model\Dao\Site;
use Core\Model\Templating\Filters\Price;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;
use EshopSales\Model\Entities\OrderSale;

class Helpers
{
	protected Price $priceFilter;

	protected Exchange $exchange;

	public function __construct(Price $priceFilter, Exchange $exchange)
	{
		$this->priceFilter = $priceFilter;
		$this->exchange    = $exchange;
	}

	/**
	 * @param string $input
	 * @param array  $timeWildcards
	 *
	 * @return string
	 */
	public static function replaceTimeWildcards(string $input, array $timeWildcards): string
	{
		foreach ($timeWildcards as $item) {
			switch ($item) {
				case 'YYYY':
					$input = str_replace('YYYY', date('Y'), $input);
					break;
				case 'YY':
					$input = str_replace('YY', date('y'), $input);
					break;
				case 'MM':
					$input = str_replace('MM', date('m'), $input);
					break;
			}
		}

		return $input;
	}

	/**
	 * @param IDiscount $discount
	 *
	 * @return string
	 */
	public function formatDiscount(IDiscount $discount): string
	{
		if ($discount instanceof OrderSale) {
			if ($discount->getValue()
				&& in_array($discount->getType(), [OrderDiscount::TYPE_FIX, OrderDiscount::TYPE_DELIVERY_PRICE]))
				$value = $this->exchange->change((float) $discount->getValue());
			else
				$value = $discount->getValue();
		} else
			$value = $discount->getValue();

		if ($discount->getType() === OrderDiscount::TYPE_FIX)
			return $this->priceFilter->format($value);

		// type percent
		if ($discount->getType() === OrderDiscount::TYPE_PERCENT)
			return sprintf('%s %s', $value, '%');

		return (string) $value;
	}

	/**
	 * @param OrderItem[] $orderItems
	 *
	 * @return bool
	 */
	public static function isCorrectiveByOrderItems(array $orderItems): bool
	{
		foreach ($orderItems as $orderItem) {
			if ($orderItem->getPrice() < 0) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param Seller                         $seller
	 * @param Site|\Core\Model\Entities\Site $site
	 *
	 * @return string
	 */
	public static function getSellerEmail(Seller $seller, $site): string
	{
		if ($seller->sendSellerEmailType === \EshopCatalog\Model\Entities\Seller::EMAIL_TYPE_SELLER)
			return EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$site->getIdent()}.email", null) ?: $seller->email;

		return $site->email;
	}

	public static function getSellerName(Seller $seller, string $siteIdent): string
	{
		return EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$siteIdent}.name", null) ?: $seller->name;
	}
}
