<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Affiliate;

use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use EshopOrders\AdminModule\Model\AffiliateCampaigns;
use EshopOrders\AdminModule\Model\AffiliatePartners;
use EshopOrders\Model\Entities\Affiliate\AffiliateCampaign;
use EshopOrders\Model\Helpers\AffiliateHelper;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Html;

class CampaignsGrid extends BaseControl
{
	protected AffiliateCampaigns         $affiliateCampaigns;
	protected AffiliatePartners          $affiliatePartners;
	protected ICampaignFormFactory       $campaignFormFactory;
	protected ICampaignsStatsGridFactory $campaignsStatsGridFactory;
	protected PriceFilter                $priceFilter;

	public function __construct(
		AffiliateCampaigns         $affiliateCampaigns,
		AffiliatePartners          $affiliatePartners,
		ICampaignFormFactory       $campaignFormFactory,
		ICampaignsStatsGridFactory $campaignsStatsGridFactory,
		PriceFilter                $priceFilter
	)
	{
		$this->affiliateCampaigns        = $affiliateCampaigns;
		$this->affiliatePartners         = $affiliatePartners;
		$this->campaignFormFactory       = $campaignFormFactory;
		$this->campaignsStatsGridFactory = $campaignsStatsGridFactory;
		$this->priceFilter               = $priceFilter;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->affiliateCampaigns->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.affiliate.addCampaign');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->setCampaign($id);

		$campaign = $this['form']->affiliateCampaign;

		$this->template->modalTitle = $this->t('eshopOrders.affiliate.editCampaign') . ' ' . ($campaign->name);

		$this->template->modal = 'form';
		$this->redrawControl('modal');
	}

	public function handleStats(int $id): void
	{
		$campaign = $this->affiliateCampaigns->get($id);

		$this->template->campaignId = $id;
		$this->template->modalTitle = $this->t('eshopOrders.affiliate.statistics') . ' - ' . ($campaign->name ?? '');
		$this->template->modal      = 'statsGrid';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->affiliateCampaigns->getEr()->createQueryBuilder('ac')
			->addSelect('ap')
			->innerJoin('ac.partner', 'ap')
			->orderBy('ac.created', 'DESC');

		$dataSource = new DoctrineDataSource($qb, 'ap.id');

		$grid->setDataSource($dataSource);

		$campaignsValue       = [];
		$campaignsOrdersValue = [];

		$dataSource->onDataLoaded[] = function(array $items) use (&$campaignsValue, &$campaignsOrdersValue) {
			/** @var AffiliateCampaign[] $items */
			$ids  = [];
			$conn = $this->em->getConnection();

			foreach ($items as $item) {
				$ids[] = $item->getId();
			}

			if (!empty($ids)) {
				foreach ($conn->fetchAllAssociative("SELECT id, campaign_id, `price`, order_value 
						FROM eshop_orders__affiliate_order 
						WHERE campaign_id IN (" . implode(',', $ids) . ")
						AND cancelled IS NULL") as $row) {
					$campaignsValue[$row['campaign_id']][$row['id']]       = (float) $row['price'];
					$campaignsOrdersValue[$row['campaign_id']][$row['id']] = (float) $row['order_value'];
				}
			}
		};

		//Columns
		$grid->addColumnText('ident', 'eshopOrders.affiliate.ident')
			->setFilterText();
		$grid->addColumnText('name', 'eshopOrders.affiliate.name')
			->setFilterText();
		$grid->addColumnNumber('status', 'eshopOrders.affiliate.status')
			->setRenderer(function(AffiliateCampaign $row) use (&$campaignsValue) {
				return count($campaignsValue[$row->getId()] ?? []) . ' - ' . $this->priceFilter->format((float) array_sum($campaignsValue[$row->getId()] ?? []), null, null, null, null, 2);
			});
		$grid->addColumnNumber('ordersValue', 'eshopOrders.affiliate.ordersValue')
			->setRenderer(function(AffiliateCampaign $row) use (&$campaignsOrdersValue) {
				return $this->priceFilter->format((float) array_sum($campaignsOrdersValue[$row->getId()] ?? []), null, null, null, null, 2);
			});
		$grid->addColumnText('partner', 'eshopOrders.affiliate.partner', 'partner.name')
			->setFilterSelect(['' => ''] + $this->affiliatePartners->getForSelectOption(), 'partner');

		// Actions
		$grid->addAction('copy', '', 'javascript:;')
			->setDataAttribute('copy-utm', '1')
			->setTitle('eshopOrders.affiliate.copyUtm')
			->setIcon('copy')
			->setBsType('default');
		$grid->addAction('stats', '', 'stats!')
			->setIcon('chart-line')
			->setBsType('default')
			->addClass('ajax');
		$grid->addAction('edit', '', 'edit!')
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setConfirm('default.reallyDelete')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax');

		// Columns prototype
		$grid->getColumn('ident')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('ident')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('status')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('ordersValue')->getElementPrototype('td')->addClass('w1nw');

		$grid->setRowCallback(function(AffiliateCampaign $item, Html $tr) {
			$tr->setAttribute('data-utm', AffiliateHelper::getUtm($item->ident, $item->partner->ident));
		});

		return $grid;
	}

	protected function createComponentForm(): CampaignForm
	{
		$control = $this->campaignFormFactory->create();

		if ($this->getParameter('id')) {
			$control->setCampaign((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[] = function() {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentStatsGrid(): Multiplier
	{
		return new Multiplier(function($id): CampaignsStatsGrid {
			return $this->campaignsStatsGridFactory->create((int) $id);
		});
	}
}
