<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Affiliate;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\AdminModule\Model\AffiliateCampaigns;
use EshopOrders\AdminModule\Model\AffiliateOrders;
use EshopOrders\AdminModule\Model\AffiliatePartners;
use EshopOrders\Model\Entities\Affiliate\AffiliateOrder;
use Nette\Utils\Html;

class OrdersGrid extends BaseControl
{
	protected AffiliatePartners  $affiliatePartners;
	protected AffiliateCampaigns $affiliateCampaigns;
	protected AffiliateOrders    $affiliateOrders;
	protected PriceFilter        $priceFilter;

	public function __construct(
		AffiliatePartners  $affiliatePartners,
		AffiliateCampaigns $affiliateCampaigns,
		AffiliateOrders    $affiliateOrders,
		PriceFilter        $priceFilter
	)
	{
		$this->affiliatePartners  = $affiliatePartners;
		$this->affiliateCampaigns = $affiliateCampaigns;
		$this->affiliateOrders    = $affiliateOrders;
		$this->priceFilter        = $priceFilter;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->affiliateOrders->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->affiliateOrders->getEr()->createQueryBuilder('ao')
			->orderBy('ao.id', 'DESC');
		$grid->setDataSource($qb);

		$campaigns = $this->affiliateCampaigns->getForSelectOption();

		//Columns
		$grid->addColumnText('order', 'eshopOrders.affiliate.order')->setRenderer(function(AffiliateOrder $row) {
			return Html::el('a', [
				'href' => $this->presenter->link('Default:editOrder', $row->order->getId()),
			])
				->setText($row->order->getId());
		})
			->setFilterText()->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('IDENTITY(ao.order) LIKE :order')
					->setParameter('order', "%$value%");
			});
		$grid->addColumnText('partner', 'eshopOrders.affiliate.partner')->setRenderer(function(AffiliateOrder $row) {
			return $row->campaign->partner->name;
		})
			->setFilterSelect(['' => ''] + $this->affiliatePartners->getForSelectOption())
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->innerJoin('ao.campaign', 'ac')
					->andWhere('IDENTITY(ac.partner) = :partner')
					->setParameter('partner', $value);
			});
		$grid->addColumnText('campaign', 'eshopOrders.affiliate.campaign')->setRenderer(function(AffiliateOrder $row) use ($campaigns) {
			return $campaigns[$row->campaign->getId()];
		})
			->setFilterSelect(['' => ''] + $this->affiliateCampaigns->getForSelectOption())
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('IDENTITY(ao.campaign) = :campaign')
					->setParameter('campaign', $value);
			});
		$grid->addColumnText('email', 'eshopOrders.affiliate.email')
			->setFilterText();
		$grid->addColumnNumber('value', 'eshopOrders.affiliate.commission')->setRenderer(function(AffiliateOrder $row) {
			return $this->priceFilter->format((float) $row->value, null, null, null, null, 2);
		});
		$grid->addColumnNumber('orderValue', 'eshopOrders.affiliate.orderValue')->setRenderer(function(AffiliateOrder $row) {
			return $this->priceFilter->format((float) $row->orderValue, null, null, null, null, 2);
		});
		$grid->addColumnDateTime('created', 'eshopOrders.affiliate.created')
			->setFormat('j. n. Y H:i');
		$grid->addColumnDateTime('cancelled', 'eshopOrders.affiliate.cancelled')
			->setFormat('j. n. Y H:i');
		//		$grid->addColumnText('name', 'eshopOrders.affiliate.name');

		// Actions

		// Columns prototype
		$grid->getColumn('order')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('order')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('orderValue')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('created')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('cancelled')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}
}
