<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Affiliate;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\AdminModule\Model\AffiliatePartners;
use EshopOrders\Model\Entities\Affiliate\AffiliatePartner;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Html;

class PartnersGrid extends BaseControl
{
	protected AffiliatePartners         $affiliatePartners;
	protected IPartnerFormFactory       $partnerFormFactory;
	protected IPartnersStatsGridFactory $partnersStatsGridFactory;

	public function __construct(
		AffiliatePartners         $affiliatePartners,
		IPartnerFormFactory       $partnerFormFactory,
		IPartnersStatsGridFactory $partnersStatsGridFactory
	)
	{
		$this->affiliatePartners        = $affiliatePartners;
		$this->partnerFormFactory       = $partnerFormFactory;
		$this->partnersStatsGridFactory = $partnersStatsGridFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->affiliatePartners->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.affiliate.addPartner');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->setPartner($id);

		$partner = $this['form']->affiliatePartner;

		$this->template->modalTitle = $this->t('eshopOrders.affiliate.editPartner') . ' ' . ($partner->name);

		$this->template->modal = 'form';
		$this->redrawControl('modal');
	}

	public function handleStats(int $id): void
	{
		$partner = $this->affiliatePartners->get($id);

		$this->template->partnerId  = $id;
		$this->template->modalTitle = $this->t('eshopOrders.affiliate.statistics') . ' - ' . ($partner->name ?? '');
		$this->template->modal      = 'statsGrid';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->affiliatePartners->getEr()->createQueryBuilder('ap')
			->orderBy('ap.created', 'DESC');
		$grid->setDataSource($qb);

		//Columns
		$grid->addColumnText('ident', 'eshopOrders.affiliate.ident');
		$grid->addColumnText('name', 'eshopOrders.affiliate.name');

		// Actions
		$grid->addAction('exportOrders', '', ':EshopOrders:Cron:Affiliate:exportPartnerOrders', ['hash' => 'hash'])
			->setIcon('file-export')
			->setTitle('eshopOrders.affiliate.exportOrders')
			->setOpenInNewTab()
			->setBsType('default');
		$grid->addAction('exportClicks', '', ':EshopOrders:Cron:Affiliate:exportPartnerClicks', ['hash' => 'hash'])
			->setIcon('mouse')
			->setTitle('eshopOrders.affiliate.exportClicks')
			->setOpenInNewTab()
			->setBsType('default');
		$grid->addAction('stats', '', 'stats!')
			->setIcon('chart-line')
			->setBsType('default')
			->addClass('ajax');
		$grid->addAction('edit', '', 'edit!')
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setConfirm('default.reallyDelete')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax');

		// Columns prototype
		$grid->getColumn('ident')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('ident')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentForm(): PartnerForm
	{
		$control = $this->partnerFormFactory->create();

		if ($this->getParameter('id')) {
			$control->setPartner((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[] = function() {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentStatsGrid(): Multiplier
	{
		return new Multiplier(function($id): PartnersStatsGrid {
			return $this->partnersStatsGridFactory->create((int) $id);
		});
	}
}
