<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\SellerInvoiceConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use Nette\Utils\Html;

class InvoiceConfigsGrid extends BaseControl
{
	protected InvoiceConfigRepository $invoiceConfigs;
	protected Invoices                $invoices;

	public function __construct(
		InvoiceConfigRepository $invoiceConfigs,
		Invoices                $invoices
	)
	{
		$this->invoiceConfigs = $invoiceConfigs;
		$this->invoices       = $invoices;
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDataSource($this->invoiceConfigs->getQueryBuilder());

		$grid->addColumnText('sellers', 'eshopOrders.invoiceConfigsGrid.sellers')
			->setRenderer(static function(InvoiceConfig $config): Html {
				$html = Html::el();

				foreach ($config->sellerInvoiceConfigs->toArray() as $sellerConfig) {
					$html->addHtml(Html::el('div')->setText($sellerConfig->seller->title));
				}

				return $html;
			});

		$grid->addColumnText('maturity', 'eshopOrders.invoiceConfigsGrid.maturity');
		$grid->addColumnText('prefix', 'eshopOrders.invoiceConfigsGrid.numericalSeries.default.prefix')
			->setRenderer(static function(InvoiceConfig $config) {
				return $config->numericalSeries->prefix;
			});
		$grid->addColumnText('digitsCount', 'eshopOrders.invoiceConfigsGrid.numericalSeries.default.digitsCount')
			->setRenderer(static function(InvoiceConfig $config) {
				return $config->numericalSeries->digitsCount;
			});
		$grid->addColumnText('startNumber', 'eshopOrders.invoiceConfigsGrid.numericalSeries.default.startNumber')
			->setRenderer(static function(InvoiceConfig $config) {
				return $config->numericalSeries->startNumber;
			});

		$grid->addAction('edit', '', ':invoiceSettings!', ['itemId' => 'id'])
			->addClass('ajax')
			->setIcon('edit')->setBsType('primary');

		$grid->addAction('remove', '', 'remove!', ['itemId' => 'id'])
			->setRenderCondition(function(InvoiceConfig $config) {
				return $this->invoices->getIvoicesCount($this->getSellersId($config->sellerInvoiceConfigs->toArray())) === 0;
			})
			->addClass('ajax')
			->setIcon('trash')
			->setBsType('danger')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	/**
	 * @param mixed $itemId
	 */
	public function handleRemove($itemId): void
	{
		$this->invoiceConfigs->remove((int) $itemId);
		$this['grid']->reload();
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function getSellersId(array $sellers): array
	{
		return array_map(static function(SellerInvoiceConfig $config): int {
			return $config->seller->getId();
		}, $sellers);
	}
}
