<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopSales\FrontModule\Model\Dao\OrderSale;
use Exception;
use Nette\DI\Container;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class OrderDiscountCodeForm extends BaseControl
{
	protected Order     $order;
	protected Container $container;

	public function __construct(
		Order     $order,
		Container $container
	)
	{
		$this->order     = $order;
		$this->container = $container;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addText('code', 'eshopOrders.orderDiscountCodeForm.code')
			->setRequired();

		$form->addSaveCancelControl();
		unset($form->getComponent('saveControl')['save']);
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			if ($this->container->hasService('eshopSales.admin.orderSales')) {
				$service = $this->container->getService('eshopSales.front.orderSales');
				/** @var OrderSale $discount */
				$discount = $service->isValidDiscountCode($values->code, $this->order->getPriceItems(true), true);

				if ($discount) {
					$od = new OrderDiscount($discount->code, $this->order);
					$od->setName('');
					$value      = (float) $discount->value;
					$totalPrice = $this->order->getPriceItems(true);

					if ($discount->type === OrderDiscount::TYPE_PERCENT) {
						$price = $totalPrice * ($value / 100);
						$od->setPrice($price);
					} else if ($discount->type === OrderDiscount::TYPE_FIX) {
						$od->setPrice($value);
					}

					$od->setValue($value);
					$od->setType($discount->type);

					$this->order->addOrderDiscount($od);

					$this->eventDispatcher->dispatch(new OrderEvent($this->order), 'eshopOrders.admin.addDiscountToOrder');

					$this->em->persist($od);
					$this->em->flush();

					$this->presenter->flashMessageSuccess('default.saved');
					$form->addCustomData('isOk', true);

					return;
				}

				$this->presenter->flashMessageDanger('eshopSalesFront.discountForm.discountCodeInvalid');
			} else {
				$this->presenter->flashMessageDanger('default.error');
			}
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->presenter->flashMessageDanger('default.error');
		}

		$form->addCustomData('isOk', false);
	}
}
