<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\AdminModule\Model\OrderItemReviews;
use EshopOrders\Model\Entities\OrderItemReview;
use Nette\Utils\Html;

class OrderItemReviewsGrid extends BaseControl
{
	protected OrderItemReviews $reviews;

	public function __construct(
		OrderItemReviews $reviews
	)
	{
		$this->reviews = $reviews;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$qb   = $this->em->getRepository(OrderItemReview::class)->createQueryBuilder('oir');
		$qb->addSelect('oi, o, c, u, p, pt')
			->addSelect('CASE WHEN oir.status = \'new\' THEN 0 ELSE 1 END as hidden sort')
			->join('oir.orderItem', 'oi')
			->join('oi.order', 'o')
			->leftJoin('o.customer', 'c')
			->leftJoin('c.user', 'u')
			->join('oi.product', 'p')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->addOrderBy('sort')
			->addOrderBy('oir.created', 'desc');

		$prQb = $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.id');
		$prQb->select('p.id, pt.name')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->addOrderBy('pt.name')
			->setParameter('lang', $this->translator->getLocale());

		$productPairs = [null => null];
		foreach ($prQb->getQuery()->getArrayResult() as $r) {
			$productPairs[$r['id']] = $r['name'];
		}

		$grid->setDataSource($qb);

		$grid->addColumnDateTime('created', 'eshopOrders.orderItemReviewsGrid.created')
			->setFormat('d.m.Y')
			->setFitContent();
		$grid->addColumnText('customer', 'eshopOrders.orderItemReviewsGrid.customer')
			->setRenderer(fn(OrderItemReview $review) => $review->customer ?? $review->orderItem->order->getCustomer()->getUser()->getFullName())
			->setFitContent();
		$grid->addColumnText('product', 'eshopOrders.orderItemReviewsGrid.product')
			->setRenderer(fn(OrderItemReview $review) => $review->orderItem->getProduct()->getText()->name)
			->setFitContent()
			->setFilterSelect($productPairs);
		$grid->addColumnText('rating', 'eshopOrders.orderItemReviewsGrid.rating')
			->setFitContent()
			->setTemplateEscaping(true)
			->setRenderer(function(OrderItemReview $review) {
				$html = Html::el();
				foreach ($review->getRatingAsArray(true) as $star) {
					if ($star === 1) {
						$html->addHtml('<i class="fas fa-star" style="color: gold"></i>');
					} else {
						if ($star === 0.5) {
							$html->addHtml('<i class="fas fa-star-half-alt" style="color: gold"></i>');
						} else {
							$html->addHtml('<i class="far fa-star" style="color: gold"></i>');
						}
					}
				}

				return $html;
			});
		$grid->addColumnText('description', 'eshopOrders.orderItemReviewsGrid.description');
		$grid->addColumnStatus('status', 'eshopOrders.orderItemReviewsGrid.status')->setAlign('center')
			->addOption(OrderItemReview::STATUS_NEW, 'eshopOrders.orderItemReviewsGrid.statusOptions.new')
			->setClass('bg-warning')->setShowTitle(true)->endOption()
			->addOption(OrderItemReview::STATUS_NOT_PUBLISHED, 'eshopOrders.orderItemReviewsGrid.statusOptions.notPublished')
			->setClass('bg-danger')->setShowTitle(true)->endOption()
			->addOption(OrderItemReview::STATUS_PUBLISHED, 'eshopOrders.orderItemReviewsGrid.statusOptions.published')
			->setClass('bg-success')->setShowTitle(true)->endOption()
			->addOption(OrderItemReview::STATUS_TOP, 'eshopOrders.orderItemReviewsGrid.statusOptions.top')
			->setClass('bg-info')->setShowTitle(true)->endOption()
			->onChange[] = [$this, 'gridStatusChange'];

		return $grid;
	}

	public function gridStatusChange(string $id, string $newStatus): void
	{
		if ($this->reviews->changeStatus($newStatus, (int) $id)) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}
}
