<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\Event;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\OrderItems;
use EshopOrders\Model\Orders;
use Nette\Utils\Html;

class OrderItemsGrid extends BaseControl
{
	protected int         $orderId;
	protected Orders      $ordersService;
	protected OrderItems  $orderItemsService;
	protected PriceFilter $priceFilter;

	public function __construct(
		int         $orderId,
		Orders      $orders,
		OrderItems  $orderItems,
		PriceFilter $priceFilter
	)
	{
		$this->orderId           = $orderId;
		$this->ordersService     = $orders;
		$this->orderItemsService = $orderItems;
		$this->priceFilter       = $priceFilter;
	}

	public function render(): void
	{
		if (EshopOrdersConfig::load('orderForm.enableReadyToDeliveryCheckboxes')) {
			$this->template->order = $this->ordersService->get($this->orderId);
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([]);
		$grid->hideTableFooter();

		$qb = $this->orderItemsService->getEr()->createQueryBuilder('oi')
			->addSelect('gifts, sales, txts, children')
			->leftJoin('oi.gifts', 'gifts')
			->leftJoin('oi.sales', 'sales')
			->leftJoin('oi.children', 'children')
			->innerJoin('oi.orderItemTexts', 'txts')
			->andWhere('oi.order = :order')->setParameter('order', $this->orderId)
			->orderBy('oi.id', 'ASC');

		$totalPrice = [
			'price1'          => 0,
			'code1'           => null,
			'price2'          => null,
			'currency2'       => null,
			'decimals2'       => null,
			'priceWithoutVat' => null,
			'noVatText'       => $this->t('eshopOrders.orderForm.noVat'),
		];

		$prodIds = $this->orderItemsService->getEr()->createQueryBuilder('oi')
			->select('IDENTITY(oi.product) as productId')
			->andWhere('oi.order = :order')->setParameter('order', $this->orderId)
			->getQuery()->getResult();
		$prodIds = array_map(function($v) { return $v['productId']; }, $prodIds);


		$images = $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.id')
			->select('p, gal, imgs')//jen nejnutnejsi info - p.id, gal.basePath, img name
			->leftJoin('p.gallery', 'gal')
			->leftJoin('gal.images', 'imgs', 'WITH', 'imgs.isCover = 1 OR imgs.position = 0')
			->andWhere('p.id IN (:ids)')->setParameter('ids', $prodIds)
			->groupBy('p.id')
			->getQuery()->getResult();

		$dataSource = new DoctrineDataSource($qb, 'p.id');
		$grid->setDataSource($dataSource);

		$dataSource->onDataLoaded[] = function($items) {
			/** @var OrderItem[] $items */
			$event                     = new Event();
			$event->data['orderItems'] = &$items;
			$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');
		};

		//Columns
		if (EshopOrdersConfig::load('orderForm.enableReadyToDeliveryCheckboxes')) {
			$grid->addColumnText('isReadyToDelivery', 'eshopOrders.itemsGrid.readyToDelivery')
				->addCellAttributes(['style' => 'text-align: center !important;'])
				->setTemplateEscaping(false)
				->setRenderer(function(OrderItem $row) {
					$el = Html::el('input', [
						'type'    => 'checkbox',
						'name'    => 'isReadyToDelivery_' . $row->getId(),
						'checked' => (bool) $row->isReadyToDelivery,
						'style'   => 'transform: scale(1.5); cursor: pointer; width: 15px;',
					]);

					return Html::el('a', ['class' => 'ajax', 'href' => $this->link(':gridReadyToDeliveryChange!', ['id' => $row->getId(), 'newStatus' => (int) !$row->isReadyToDelivery])])->setHtml($el);
				});
		}
		$grid->addColumnText('image', 'eshopOrders.defaultGrid.image', '')
			->setRenderer(function(OrderItem $row) use ($images) {
				if (!isset($images[$row->getProductId()]) || !$images[$row->getProductId()]->getGallery() || !$images[$row->getProductId()]->getGallery()->getImages()) {
					return '';
				}
				$image = $images[$row->getProductId()]->getGallery()->getImages()->first();

				if ($image) {
					return Html::el('')
						->addHtml(Html::el('img', [
							'onMouseOver' => "showPicture('spt-{$row->getId()}', 1)",
							'onMouseOut'  => "showPicture('spt-{$row->getId()}', 0)",
							'src'         => $this->imagePipe->request($image->getFilePath(), '50x40'),
						]))
						->addHtml(Html::el('img', [
							'id'    => 'spt-' . $row->getId(),
							'class' => 'show-picture-target',
							'src'   => $this->imagePipe->request($image->getFilePath(), '400x400'),
						]));
				}

				return '';
			})->setAlign('center');
		$grid->addColumnText('code1', 'eshopOrders.itemsGrid.code1')->setRenderer(function(OrderItem $row) {
			return $row->getCode1() . ($row->getMoreDataValue('stockSupplyAssigned') ? '*' : '');
		});
		if (EshopOrdersConfig::load('orderForm.showSupplierColumn')) {
			$grid->addColumnText('supplier', 'eshopOrders.itemsGrid.supplier')
				->setRenderer(function(OrderItem $row) {
					return $row->getSuppliersLinks();
				});
			$grid->getColumn('supplier')->getElementPrototype('th')->class[] = 'w1nw';
			$grid->getColumn('supplier')->getElementPrototype('td')->class[] = 'w1nw';
		}
		$grid->addColumnText('productName', 'eshopOrders.itemsGrid.productName')
			->setRenderer(function(OrderItem $row) {
				$html = Html::el('div');

				$prodId = $row->getProductId();
				$link   = $prodId && $row->getProduct() && !$row->getProduct()->isDeleted
					? Html::el('a', [
							'href'   => $this->getPresenter()->link(':EshopCatalog:Admin:Products:edit', [$prodId]),
							'target' => '_blank',
						]
					)
					: Html::el('span');

				$productName = $row->getOrderItemText()->getName();

				if ($row->getParent()) {
					$html->addClass('ml-3');
					$html->addHtml($link->setHtml('<strong style="color: #f44336; margin-right: 5px">― ― ―</strong> ' . $productName));
				} else {
					$html->addHtml($link->setText($productName));
				}

				if ($row->getMoreDataValue('note')) {
					$html->addHtml(Html::el('div')
						->setHtml($this->t('eshopOrders.itemsGrid.note') . ' - ' . nl2br($row->getMoreDataValue('note'))));
				}

				foreach ($row->getGifts() as $gift) {
					$html->addHtml(Html::el('div')
						->setText($this->t('eshopOrders.itemsGrid.gift', ['name' => $gift->getName()])));
				}

				return $html;
			});

		$grid->addColumnText('quantity', 'eshopOrders.itemsGrid.quantity')
			->setRenderer(fn(OrderItem $row) => Html::el('strong')->setText($row->getQuantity()));

		if (EshopOrdersConfig::load('orderForm.productsVatRateAfterPrice')) {
			$grid->addColumnText('price', 'eshopOrders.itemsGrid.price')
				->setRenderer(function(OrderItem $row) {
					return $this->gridRenderPrice($row, 'getPrice');
				})->setAlign('right');
			$grid->addColumnNumber('vatRate', 'eshopOrders.itemsGrid.vatRate')->setFormat(0, ",", " ");
		} else {
			$grid->addColumnNumber('vatRate', 'eshopOrders.itemsGrid.vatRate')->setFormat(0, ",", " ");
			$grid->addColumnText('price', 'eshopOrders.itemsGrid.price')
				->setRenderer(function(OrderItem $row) {
					return $this->gridRenderPrice($row, 'getPrice');
				})->setAlign('right');
		}

		$grid->addColumnText('sale', 'eshopOrders.itemsGrid.sale')
			->setRenderer(function(OrderItem $item) {
				$compounded = '';
				foreach ($item->sales as $sale) {
					$compounded .= $sale->getFormatted();
					if ($sale !== end($item->sales)) {
						$compounded . ' | ';
					}
				}

				return $compounded;
			});
		$grid->addColumnText('priceTotal', 'eshopOrders.itemsGrid.priceTotal')
			->setRenderer(function(OrderItem $row) use (&$totalPrice) {
				$totalPrice['price1'] += $row->getPriceTotal();
				$totalPrice['code1']  = $row->getOrder()->getDefaultCurrency();

				if ($row->getOrder()->currency) {
					$totalPrice['price2']           += $row->getPriceTotal(true);
					$totalPrice['currency2']        = $row->getOrder()->currency->code;
					$totalPrice['decimals2']        = $row->getOrder()->currency->decimals;
					$totalPrice['price2WithoutVat'] += $row->getPriceTotalWithoutVat(true);
				}

				$totalPrice['priceWithoutVat'] += $row->getPriceTotalWithoutVat();

				return $this->gridRenderPrice($row, 'getPriceTotal');
			})->setAlign('right');

		// Filter

		// Actions
		$grid->addAction('front', '', ':EshopCatalog:Front:Default:product', ['id' => 'productId'])
			->setOpenInNewTab()
			->setIcon('link')->setBsType('secondary')
			->setRenderCondition(function(OrderItem $oi) {
				return $oi->getProductId() && $oi->getProduct() && !$oi->getProduct()->isDeleted ? true : false;
			});
		$grid->addAction('edit', '', ':orderItemForm!', ['itemId' => 'id'])
			->addClass('ajax')
			->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'orderItemForm:delete!', ['itemId' => 'id'])
			->addClass('ajax')
			->setIcon('times')->setBsType('danger')
			->setConfirm('eshopOrders.itemsGrid.deleteConfirm');

		$grid->setColumnsSummary(['priceTotal'])
			->setRenderer(function($row) use (&$totalPrice) {
				/** @phpstan-ignore-next-line */
				return (string) $this->priceFilter->renderTwoPrices(...array_values($totalPrice));
			});

		// Columns prototype
		$grid->getColumn('image')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('code1')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('quantity')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('vatRate')->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */
	public function handleGridReadyToDeliveryChange(string $id, string $newStatus): void
	{
		if ($this->orderItemsService->setReadyToDelivery((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');

		$parent = $this->getParent();
		if ($parent && method_exists($parent, 'redrawControl')) {
			$parent->redrawControl('expeditionForm');
		}
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	protected function gridRenderPrice(OrderItem $item, string $method): Html
	{
		$order = $item->getOrder();

		if ($order->currency) {
			$price2    = $item->$method(true);
			$code2     = $order->getCurrencyCode();
			$decimals2 = $order->currency->decimals;
		}

		return $this->priceFilter
			->renderTwoPrices($item->$method(), $order->getDefaultCurrency(),
				$price2 ?? null, $code2 ?? null, $decimals2 ?? null,
				$item->{$method . 'WithoutVat'}(), $this->t('eshopOrders.orderForm.noVat'),
				$item->{$method . 'WithoutVat'}(true),
			);
	}
}
