<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Statuses;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\Entities\Status;
use Ublaboo\DataGrid\Column\Action\Confirmation\StringConfirmation;

class StatusesGrid extends BaseControl
{
	protected Statuses           $statuses;
	protected IStatusFormFactory $formFactory;

	public function __construct(
		Statuses           $statuses,
		IStatusFormFactory $formFactory
	)
	{
		$this->statuses    = $statuses;
		$this->formFactory = $formFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.addStatus');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(string $ident): void
	{
		$this['form']->setStatus($ident);
		$this->template->modalTitle = $this->t('eshopOrders.title.editStatus', ['status' => $this['form']->status->getName()]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(string $id): void
	{
		if ($this->statuses->remove($id)) {
			$this->presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		} else {
			$this->presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([], true);

		$grid->setDataSource($this->statuses->getEr()->createQueryBuilder('s')->orderBy('s.position', 'ASC'));

		// Columns
		$grid->addColumnText('id', 'eshopOrders.status.id');
		$grid->addColumnText('name', 'eshopOrders.status.name');
		$grid->addColumnStatus('sendToCustomer', 'eshopOrders.status.sendToCustomerShort')
			->addOption(0, 'default.no')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->addOption(1, 'default.yes')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridSendToCustomerChange'];

		// Prototype
		$grid->getColumn('id')->getElementPrototype('td')->class[]             = 'w1nw';
		$grid->getColumn('sendToCustomer')->getElementPrototype('td')->class[] = 'w1nw';

		// Actions
		$grid->addAction('edit', '', 'edit!', ['ident' => 'id'])
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setConfirmation(
				new StringConfirmation('eshopOrders.orderForm.reallyDelete')
			)
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setRenderCondition(static fn(Status $status) => !$status->isLocked);

		return $grid;
	}

	protected function createComponentForm(): StatusForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('ident')) {
			$control->setStatus($this->getParameter('ident'));
		}

		$control['form']->onSuccessSave[] = function() {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	public function gridSendToCustomerChange(string $id, string $newStatus): void
	{
		if ($this->statuses->changeSendToClient($id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}
}
