<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\WebColors;
use DateTime;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Invoice\Discount;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCurrency;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Statuses;

/**
 * @method Order|null getReference($id)
 * @method Order[] getAll()
 * @method Order|null get($id)
 */
class Orders extends BaseEntityService
{
	protected                 $entityClass = Order::class;
	protected EventDispatcher $eventDispatcher;
	protected WebColors       $webColors;
	protected Invoices        $invoices;
	protected Statuses        $statusesService;
	protected Payments        $payments;

	protected ?array $cNewOrders = null;

	public function __construct(
		EventDispatcher $eventDispatcher,
		WebColors       $webColors,
		Invoices        $invoices,
		Statuses        $statusesService,
		Payments        $payments
	)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->webColors       = $webColors;
		$this->invoices        = $invoices;
		$this->statusesService = $statusesService;
		$this->payments        = $payments;
	}

	/**
	 * @return OrderExpedition[]
	 */
	public function getOrdersForExpedition(): array
	{
		$data  = [];
		$stats = [];

		$qb = $this->em->createQueryBuilder()->select('IDENTITY(os.status) as ident, IDENTITY(os.order) as order')
			->from(OrderStatus::class, 'os')
			->andWhere("os.status NOT IN ('" . implode("','", array_diff(array_keys(OrderStatus::$statuses), EshopOrdersConfig::load('expeditionOrdersGrid.status'))) . "')")
			->andWhere('(os.deleted IS NULL OR os.deleted = 0)')
			->setMaxResults(600)
			->orderBy('os.id', 'desc');

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$stats[$row['order']][] = $row['ident'];
		}

		$processStats = EshopOrdersConfig::load('expeditionOrdersGrid.status');
		$ids          = [];
		foreach ($stats as $orderId => $v) {
			if (in_array($v[0], $processStats)) {
				$ids[] = $orderId;
			}
		}

		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(os.order) as id, oss.ident')
			         ->from(OrderSpedition::class, 'os')
			         ->leftJoin('os.spedition', 'oss')
			         ->andWhere('os.order IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->groupBy('os.id')
			         ->orderBy('os.id', 'asc')->getQuery()->getArrayResult() as $row) {
			$data[] = [
				'id'      => $row['id'],
				'service' => $row['ident'],
			];
		}

		$event = new Event([
			'data'       => $data,
			'expedition' => [],
		]);
		$this->eventDispatcher->dispatch($event, Orders::class . '::processOrdersForExpedition');

		foreach ($event->data['expedition'] as &$expedition) {
			/** @var OrderExpedition $expedition */
			$expedition->siteColors = $this->webColors->getColors($expedition->order->site->getIdent());
		}

		$expeditions = $event->data['expedition'];
		$event       = null;
		$data        = null;

		uasort($expeditions, static function(OrderExpedition $a, OrderExpedition $b) {
			$now   = (new DateTime())->getTimestamp();
			$aDate = $a->order->demandedExpeditionDate;
			$aId   = $a->getOrderId();

			$bDate = $b->order->demandedExpeditionDate;
			$bId   = $b->getOrderId();

			$aCategory = $aDate ? ($aDate->getTimestamp() < $now ? -1 : 1) : 0;
			$bCategory = $bDate ? ($bDate->getTimestamp() < $now ? -1 : 1) : 0;

			return $aId <=> $bId;
		});

		return $expeditions;
	}

	public function sendExpeditionData(array $ids, int $quantity = 1, array $advancedOptions = []): array
	{
		$event = new Event([
			'data'            => $this->getOrderIdsWithSpeditionCode($ids),
			'quantity'        => $quantity,
			'advancedOptions' => $advancedOptions,
			'ok'              => [],
			'okCount'         => 0,
			'errors'          => [],
			'errorCount'      => 0,
		]);

		$this->eventDispatcher->dispatch($event, Orders::class . '::sendExpeditionData');

		$arr = [
			'okCount'    => $event->data['okCount'],
			'errorCount' => $event->data['errorCount'],
		];

		if (isset($event->data['message'])) {
			$arr['message'] = $event->data['message'];
		}

		return $arr;
	}

	public function generateLabel(array $ids): array
	{
		$event = new Event([
			'data'       => $this->getOrderIdsWithSpeditionCode($ids),
			'ok'         => [],
			'okCount'    => 0,
			'errors'     => [],
			'errorCount' => 0,
			'files'      => [],
		]);

		$this->eventDispatcher->dispatch($event, Orders::class . '::generateLabel');
		$this->eventDispatcher->dispatch($event, Orders::class . '::afterGenerateLabel');

		return [
			'okCount'    => $event->data['okCount'],
			'errorCount' => $event->data['errorCount'],
			'files'      => $event->data['files'],
		];
	}

	protected function getOrderIdsWithSpeditionCode(array $ids): array
	{
		$result = [];
		$qb     = $this->getEr()->createQueryBuilder('o')
			->select('o.id, oss.ident')
			->innerJoin('o.spedition', 'os')
			->leftJoin('os.spedition', 'oss')
			->andWhere('o.id IN (:ids)')
			->setParameter('ids', $ids);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$result[] = [
				'id'      => $row['id'],
				'service' => $row['ident'],
			];
		}

		return $result;
	}

	/**
	 * @param array<int, array{price: string|float, quantity: string|int, oiId: string|int}> $products
	 * @param array<int, array{odId: string|int, price: string|float}>                       $orderDiscounts
	 * @param string|float                                                                   $paymentPrice
	 * @param string|float                                                                   $speditionPrice
	 * @param \DateTime|\DateTimeImmutable|null                                              $duzp
	 * @param \DateTime|\DateTimeImmutable|null                                              $createdAt
	 */
	public function createCorrectiveTaxDocument(
		Order $fromOrder,
		array $products,
		array $orderDiscounts,
		      $paymentPrice,
		      $speditionPrice,
		      $duzp = null,
		      $createdAt = null
	): void
	{
		$order      = clone $fromOrder;
		$orderItems = [];
		/** @var OrderItem $item */
		foreach ($fromOrder->getOrderItems()->toArray() as $item) {
			if (in_array($item->getId(), array_keys($products))) {
				$newOi = clone $item;
				$newOi->setQuantity((int) $products[$item->getId()]['quantity']);
				$newOi->setPrice(-((float) Strings::formatEntityDecimal($products[$item->getId()]['price'])));
				$oit = clone $item->getOrderItemText();
				$oit->setId($newOi);
				$this->em->persist($oit);
				$newOi->setOrderItemText($oit);
				$newOi->order = $order;
				$orderItems[] = $newOi;
			}
		}

		$order->setOrderItems($orderItems);

		$orderCurrency = null;
		if ($fromOrder->currency) {
			$oc                      = $fromOrder->currency;
			$orderCurrency           = new OrderCurrency($order, $oc->code);
			$orderCurrency->rate     = $oc->rate;
			$orderCurrency->decimals = $oc->decimals;
		}

		$op           = $fromOrder->getPayment();
		$orderPayment = clone $op;

		$paymentIdent = EshopOrdersConfig::load('correctiveTaxDocument.payment');
		if ($paymentIdent) {
			/** @var Payment|null $payment */
			$payment = $this->payments->getEr()->findOneBy(['ident' => $paymentIdent], ['id' => 'asc']);
			if ($payment) {
				$orderPayment->setName($payment->getName());
				$orderPayment->setPayment($payment);
			}
		}

		$orderPayment->setPrice(-((float) $paymentPrice));
		$orderPayment->order = $order;

		$os             = $fromOrder->getSpedition();
		$orderSpedition = clone $os;
		$orderSpedition->setPrice(-((float) $speditionPrice));
		$orderSpedition->order = $order;

		$ai             = $fromOrder->getAddressInvoice();
		$addressInvoice = $ai ? clone $ai : null;

		$ad              = $fromOrder->getAddressDelivery();
		$addressDelivery = $ad ? clone $ad : null;

		$hasStatusCreated   = false;
		$hasStatusSpedition = false;
		$hasStatusFinished  = false;
		foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() == OrderStatus::STATUS_CREATED) {
				$hasStatusCreated = true;
			} else if ($orderStatus->getStatus()->getId() == OrderStatus::STATUS_SPEDITION) {
				$hasStatusSpedition = true;
			} else if ($orderStatus->getStatus()->getId() == OrderStatus::STATUS_FINISHED) {
				$hasStatusFinished = true;
			}
		}

		if (!$hasStatusCreated) {
			$statusCreated      = $this->statusesService->get(OrderStatus::STATUS_CREATED);
			$orderActionCreated = new OrderStatus($order, $statusCreated);
			$this->em->persist($orderActionCreated);
		}

		if (!$hasStatusSpedition) {
			$statusSpedition      = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
			$orderActionSpedition = new OrderStatus($order, $statusSpedition);
			$this->em->persist($orderActionSpedition);
		}

		if (!$hasStatusFinished) {
			$statusFinished      = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
			$orderActionFinished = new OrderStatus($order, $statusFinished);
			$this->em->persist($orderActionFinished);
		}

		$order->isCorrectiveTaxDocument = 1;
		$order->correctiveTaxDocumentOf = $fromOrder;
		$order->isPaid                  = 1;
		$order->currency                = $orderCurrency;
		$order->setPayment($orderPayment);
		$order->setSpedition($orderSpedition);
		if ($addressInvoice) {
			$order->setAddressInvoice($addressInvoice);
		}
		if ($addressDelivery) {
			$order->setAddressDelivery($addressDelivery);
		}
		$order->setInvoice(null);

		/** @var OrderItem $oi */
		foreach ($order->getOrderItems()->toArray() as $oi) {
			$this->em->persist($oi);
		}
		$orderStatuses = [];
		foreach ($order->getOrderStatuses()->toArray() as $item) {
			/** @var OrderStatus $os */
			$os = clone $item;
			$os->setOrder($order);
			$this->em->persist($os);
		}
		$order->setOrderStatuses($orderStatuses);

		$newOrderDiscounts = [];
		foreach ($orderDiscounts as $id => $orderDiscount) {
			/** @var OrderDiscount $od */
			foreach ($order->getOrderDiscounts()->toArray() as $od) {
				if ($od->getId() === (int) $orderDiscount['odId']) {
					$newOrderDiscount = clone $od;

					$amount = -abs((float) Strings::formatEntityDecimal($orderDiscount['price']));
					$newOrderDiscount->setPrice($amount);
					$newOrderDiscount->setOrder($order);
					$this->em->persist($newOrderDiscount);
					$newOrderDiscounts[] = $newOrderDiscount;
				}
			}
		}

		$this->em->persist($order);
		$this->em->persist($orderSpedition);
		$this->em->persist($orderPayment);
		if ($addressInvoice) {
			$this->em->persist($addressInvoice);
		}
		if ($addressDelivery) {
			$this->em->persist($addressDelivery);
		}
		$this->em->flush();

		$invoice = $this->invoices->getOneByOrder($order, true, InvoiceConfig::TYPE_CORRECTIVE);

		if ($duzp && $invoice) {
			$invoice->duzp = $duzp;
		}

		if ($createdAt && $invoice) {
			$invoice->createdAt = $createdAt;
		}

		$order->setInvoice($invoice);

		foreach ($invoice->invoiceData->discounts->toArray() as $od) {
			$this->em->remove($od);
		}

		$invoice->invoiceData->discounts->clear();
		foreach ($newOrderDiscounts as $orderDiscount) {
			$newDiscount        = new Discount($invoice->invoiceData);
			$newDiscount->name  = $orderDiscount->getName();
			$newDiscount->type  = $orderDiscount->getType();
			$newDiscount->code  = $orderDiscount->getCode();
			$newDiscount->value = $orderDiscount->getValue();
			$newDiscount->price = (string) Strings::formatEntityDecimal($orderDiscount->getPrice(true));

			$this->em->persist($newDiscount);
		}

		$this->em->flush();

		$event                        = new OrderEvent($order);
		$event->orderPayment          = $orderPayment;
		$event->orderSpedition        = $orderSpedition;
		$event->addrDelivery          = $addressDelivery;
		$event->addrInvoice           = $addressInvoice;
		$event->orderItems            = $orderItems;
		$event->formData['fromOrder'] = $fromOrder;

		$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.correctiveTaxDocumentCreated');
	}

	public function getNewOrders(): array
	{
		if ($this->cNewOrders === null) {
			$this->cNewOrders = $this->em->getConnection()->fetchAllAssociative("SELECT `order_id`
					FROM `eshop_orders__order_status`
					GROUP BY `order_id`
					HAVING MIN(`created`) = MAX(`created`) AND MIN(`status_id`) = '" . OrderStatus::STATUS_CREATED . "'") ?: [];
		}

		return $this->cNewOrders;
	}

	public function getOrdersNotFinishedOrCancelled(): array
	{
		if ($this->cNewOrders === null) {
			$this->cNewOrders = $this->em->getConnection()->fetchAllAssociative("SELECT o.order_id
					FROM eshop_orders__order_status o
					WHERE o.order_id NOT IN (
						SELECT `order_id`
						FROM `eshop_orders__order_status`
						WHERE `status_id` IN ('" . OrderStatus::STATUS_FINISHED . "', '" . OrderStatus::STATUS_CANCELED . "') 
						AND deleted IS NULL
					)
					group by o.order_id;") ?: [];
		}

		return $this->cNewOrders;
	}
}
