<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Components\Customer\CustomerForm;
use EshopOrders\AdminModule\Components\Customer\CustomersGrid;
use EshopOrders\AdminModule\Components\Customer\GroupCustomersForm;
use EshopOrders\AdminModule\Components\Customer\GroupsCustomersGrid;
use EshopOrders\AdminModule\Components\Customer\ICustomerFormFactory;
use EshopOrders\AdminModule\Components\Customer\ICustomersGridFactory;
use EshopOrders\AdminModule\Components\Customer\IGroupCustomersFormFactory;
use EshopOrders\AdminModule\Components\Customer\IGroupsCustomersGridFactory;
use EshopOrders\AdminModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\GroupCustomers;

class CustomersPresenter extends BasePresenter
{
	protected Customers $customers;

	public function __construct(
		Customers $customers
	)
	{
		$this->customers = $customers;
		parent::__construct();
	}

	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('eshopOrders.menu.customers'));
		$this->setHeader('eshopOrders.menu.customers', 'fa fa-user-friends');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionJsonList(): void
	{
		$output = [];

		foreach ($this->customers->getAllArray() as $row) {
			$output[$row['id']] = [
				'id'    => (string) $row['id'],
				'email' => $row['email'],
				'name'  => trim($row['email'] . ' - ' . $row['name'] . ' ' . $row['lastname']),
			];
		}

		$this->sendJson($output);
	}

	public function actionDefault(): void
	{
		$this['navigation']->setData(['header' => [
		]]);

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopOrders.menu.addGroupCustomers',
				'link'  => 'Customers:addGroup',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionEditCustomer(int $id): void
	{
		$customer = $this->em->getRepository(Customer::class)->find($id);

		if (!$customer) {
			$this->error('customer not found');
		}

		$this->setTitle($this->translator->translate('eshopOrders.title.editCustomer'), $customer->getUser()->getName() . ' ' . $customer->getUser()->getLastname());
		$this->setHeader('eshopOrders.title.editCustomer', 'fa fa-user-friends');
	}

	public function actionAddGroup(): void
	{
		$this->setTitle($this->translator->translate('eshopOrders.title.addGroupCustomers'));
		$this->setHeader('eshopOrders.title.addGroupCustomers', 'fa fa-users');
	}

	public function actionEditGroup(int $id): void
	{
		$group = $this->em->getRepository(GroupCustomers::class)->find($id);

		if (!$group) {
			$this->error('group not found');
		}

		$this->setTitle($this->translator->translate('eshopOrders.title.editGroupCustomers'));
		$this->setHeader('eshopOrders.title.editGroupCustomers', 'fa fa-users');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentCustomersGrid(ICustomersGridFactory $factory): CustomersGrid { return $factory->create(); }

	protected function createComponentGroupsCustomersGrid(IGroupsCustomersGridFactory $factory): GroupsCustomersGrid { return $factory->create(); }

	protected function createComponentCustomerForm(ICustomerFormFactory $factory): CustomerForm
	{
		$control = $factory->create();

		if ($this->getParameter('id')) {
			$control->setCustomer((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('editCustomer', (int) $form->getCustomData('customerId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('default');
		};

		return $control;
	}

	protected function createComponentGroupCustomersForm(IGroupCustomersFormFactory $factory): GroupCustomersForm
	{
		$control = $factory->create();

		if ($this->getParameter('id')) {
			$control->setGroupCustomers((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('editGroup', (int) $form->getCustomData('groupCustomersId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('default');
		};

		return $control;
	}

}
