<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Presenters;

use Contributte\Scheduler\Helpers\Debugger;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Components\Invoice\IInvoiceEditFormFactory;
use EshopOrders\AdminModule\Components\Invoice\InvoiceEditForm;
use EshopOrders\AdminModule\Components\Order\CorrectiveTaxDocumentForm;
use EshopOrders\AdminModule\Components\Order\CustomerInfoForm;
use EshopOrders\AdminModule\Components\Order\ICorrectiveTaxDocumentFormFactory;
use EshopOrders\AdminModule\Components\Order\ICustomerInfoFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderAddressFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderDiscountCodeFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderDiscountFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderItemAddFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderItemFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderPaymentFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrdersGridFactory;
use EshopOrders\AdminModule\Components\Order\IOrderSpeditionFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderStatusFormFactory;
use EshopOrders\AdminModule\Components\Order\IUploadExternalInvoiceFactory;
use EshopOrders\AdminModule\Components\Order\OrderAddressForm;
use EshopOrders\AdminModule\Components\Order\OrderDiscountCodeForm;
use EshopOrders\AdminModule\Components\Order\OrderDiscountForm;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrderItemAddForm;
use EshopOrders\AdminModule\Components\Order\OrderItemForm;
use EshopOrders\AdminModule\Components\Order\OrderPaymentForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusForm;
use EshopOrders\AdminModule\Components\Order\UploadExternalInvoice;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Subscribers\OrderSubscriber;
use Exception;

class DefaultPresenter extends BasePresenter
{
	public Orders $ordersService;

	public function __construct(
		Orders $ordersService
	)
	{
		$this->ordersService = $ordersService;
	}

	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('eshopOrders.title.orders'));
		$this->setHeader('eshopOrders.title.orders', 'fa fa-shopping-cart');
		OrderSubscriber::$enableSendInvoice = true;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/** Vyrenderuje vyskakovaci formular pro pridani stavu k objednavce */
	public function handleOrderStatusForm(): void
	{
		$this->template->modal = 'orderStatusForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.addOrderStatus'));
		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleSendCustomerInfo(): void
	{
		$this->template->modal = 'customerInfo';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.sendCustomerInfo'));
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci položky (cena a pocet ks)
	 */
	public function handleOrderItemForm(?int $itemId = null): void
	{
		$this->template->modal = 'orderItemForm';
		$this->redrawControl('modal');

		$control = $this['orderItemForm'];
		if ($itemId) {
			$control->setOrderItem($itemId);
			$this->setTitle($this->t('eshopOrders.title.editOrderItem'), (string) $itemId);
		} else {
			$this->setTitle($this->t('eshopOrders.title.addOrderItem'));
		}
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro pridani stavu k objednavce */
	public function handleOrderItemAddForm(): void
	{
		$this->template->modal = 'orderItemAddForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.addOrderItem'), ' ');
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dopravy v objednavce */
	public function handleOrderSpeditionForm(): void
	{
		$this->template->modal = 'orderSpeditionForm';
		$this->redrawControl('modal');

		$control = $this['orderSpeditionForm'];

		$this->setTitle($this->t('eshopOrders.title.editOrderSpedition'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci platby v objednavce */
	public function handleOrderPaymentForm(): void
	{
		$this->template->modal = 'orderPaymentForm';
		$this->redrawControl('modal');

		$control = $this['orderPaymentForm'];

		$this->setTitle($this->t('eshopOrders.title.editOrderPayment'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dorucovaci adresy
	 */
	public function handleAddressDeliveryForm(): void
	{
		$this->template->modal = 'orderAddressDeliveryForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editAddressDelivery'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dorucovaci adresy
	 */
	public function handleAddressInvoiceForm(): void
	{
		$this->template->modal = 'orderAddressInvoiceForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editAddressInvoice'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro urceni produtku v odd
	 */
	public function handleCorrectiveTaxDocumentForm(): void
	{
		$order = $this->ordersService->get((int) $this->getParameter('id'));

		if ($order->getOrderForCorrectiveTaxDocument() !== null) {
			$this->redirect(':EshopOrders:Admin:Default:editOrder', $order->getOrderForCorrectiveTaxDocument()->getId());
		}

		$this->template->modal = 'correctiveTaxDocumentForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.correctiveTaxDocumentForm'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleUploadExternalInvoice(): void
	{
		$order = $this->ordersService->get((int) $this->getParameter('id'));

		$this->template->modal = 'UploadExternalInvoice';
		$this->redrawControl('modal');

		$this->setTitle($order->getInvoice()
			? $this->t('eshopOrders.uploadExternalInvoiceForm.titleInvoiceExists', null, ['invoice' => $order->getInvoice()->ident])
			: $this->t('eshopOrders.uploadExternalInvoiceForm.title')
		);

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleOrderDiscountForm(): void
	{
		$this->template->modal = 'orderDiscountForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.orderDiscountForm'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleOrderDiscountCodeForm(): void
	{
		$this->template->modal = 'orderDiscountCodeForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.orderDiscountCodeForm'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleInvoiceEditForm(): void
	{
		$this->template->modal = 'invoiceEditForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editInvoice'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleNavigationCallback(string $key): void
	{
		$this->eventDispatcher->dispatch(new OrderNavigationCallbackEvent($key, $this), 'eshopOrders.admin.ordersDefaultNavigationCallback');
	}

	public function handleSendActualOrderToClient(int $id): void
	{
		$order = $this->ordersService->getFullOrder($id);
		if ($order) {
			$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendActualOrder');
		}
	}

	public function handleSendInvoice(int $id): void
	{
		$order = $this->ordersService->getFullOrder($id);
		if ($order) {
			$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendInvoice');
		}
	}

	public function handleSendInvoiceReminder(int $id): void
	{
		$order = $this->ordersService->getFullOrder($id);
		if ($order) {
			$order->increaseInvoiceReminderCount();
			$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendInvoiceReminder');
			$this->em->persist($order)->flush();
		}

		$this->flashMessageSuccess('default.sended');
		$this->redirect('this');
	}

	public function handleGridChangeStatus(): void
	{
		$status = $this->getParameter('status');
		$ids    = $this->getParameter('ids');

		if ($status && $ids) {
			$this['ordersGrid']->gridChangeStatus($ids, $status);
		}
	}

	public function handleRemoveOrderDiscount(int $discountId): void
	{
		try {
			$orderDiscount = $this->em->getRepository(OrderDiscount::class)->find($discountId);

			if ($orderDiscount) {
				$this->em->remove($orderDiscount);
				$this->em->flush();
			}
			$this->flashMessageSuccess('default.removed');
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->flashMessageDanger('default.error');
		}

		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$navigation = [];

		foreach (EshopOrdersConfig::load('ordersGridNavButtonStatuses', []) as $status) {
			if (is_string($status)) {
				$navigation[] = [
					'title'            => $this->t('eshopOrders.ordersGrid.statusesLong.' . $status),
					'link'             => $this->link('gridChangeStatus!'),
					'data-status'      => $status,
					'data-grid-action' => $this['ordersGrid']['grid']->getFullName(),
				];
			}
		}

		$this->eventDispatcher->dispatch(new OrdersNavigationEvent($navigation), 'eshopOrders.admin.ordersDefaultNavigation');
		$this->transformNavigation($navigation);

		$this['navigation']->setData(['header' => $navigation]);
	}

	public function actionEditOrder(int $id): void
	{
		/** @var Order $order */
		$order = $this->em->getRepository(Order::class)->find($id);

		if (!$order) {
			$this->error('Order ' . $id . ' not found');
		}

		$navigation = [
			[
				'title' => 'eshopOrders.menu.editAddressDelivery',
				'link'  => $this->link('addressDeliveryForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.editAddressInvoice',
				'link'  => $this->link('addressInvoiceForm!', ['type' => 'invoice']),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.addItem',
				'link'  => $this->link('orderItemAddForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.addStatus',
				'link'  => $this->link('orderStatusForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.sendCustomerInfo',
				'link'  => $this->link('sendCustomerInfo!'),
				'ico'   => 'envelope',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.sendActualOrder',
				'link'  => $this->link('sendActualOrderToClient!'),
				'ico'   => 'envelope',
				'class' => 'ajax',
			],
		];

		if ((EshopOrdersConfig::load('invoice.enable') && !in_array($order->getPaymentIdent(), ['storeCash',
					'storeCard'])) || $order->isCorrectiveTaxDocument) {
			if (!$order->getInvoice()) {
				if (!EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile') || $order->enableInvoiceGeneration) {
					$navigation[] = [
						'link'   => $this->link('Orders:invoice', $order->getId()),
						'ico'    => 'fas fa-sync',
						'title'  => 'eshopOrders.menu.generateInvoice',
						'target' => '_blank',
					];
				}
			} else {
				$navigation[] = [
					'link'   => $this->link('Orders:invoice', $order->getId()),
					'ico'    => 'fas fa-file-invoice',
					'title'  => 'eshopOrders.menu.showInvoice',
					'target' => '_blank',
				];
				$navigation[] = [
					'link'  => $this->link('sendInvoice!'),
					'ico'   => 'envelope',
					'title' => 'eshopOrders.menu.sendInvoice',
					'class' => 'ajax',
				];
				if ($order->getInvoice()->getId() && !$order->getInvoice()->isExternal() && $this->getUser()->isAllowed('EshopOrders:Admin:Invoice', 'regenerateInvoices')) {
					$navigation[] = [
						'link'  => $this->link('regenerate!', $order->getInvoice()->getId()),
						'ico'   => 'sync',
						'title' => 'eshopOrders.menu.regenerateInvoice',
						'class' => 'ajax',
					];
				}
			}

			if (EshopOrdersConfig::load('customer.allowInvoiceReminder')) {
				$class = '';
				if ($order->invoiceReminderCount === Customer::WARNING_INVOICE_REMINDER) {
					$class = '!btn-warning';
				} else if ($order->invoiceReminderCount === Customer::DANGER_INVOICE_REMINDER) {
					$class = '!btn-danger';
				}

				$navigation[] = [
					'link'    => $this->link('sendInvoiceReminder!'),
					'ico'     => 'far fa-bell',
					'title'   => 'eshopOrders.menu.sendInvoiceReminder',
					'class'   => trim(sprintf('%s ajax', $class)),
					'confirm' => $this->t('eshopOrders.customer.invoiceReminder.confirm'),
				];
			}

			if (EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile') && !$order->enableInvoiceGeneration) {
				$navigation[] = [
					'link'  => $this->link('uploadExternalInvoice!'),
					'ico'   => 'fas fa-file-upload',
					'title' => 'eshopOrders.menu.uploadExternalInvoice',
					'class' => 'ajax',
				];
			}
		}

		if ($order->getInvoice() && !$order->getInvoice()->isExternal()) {
			$navigation[] = [
				'link'  => $this->link('invoiceEditForm!'),
				'ico'   => 'far fa-sticky-note',
				'title' => 'eshopOrders.menu.editInvoice',
				'class' => 'ajax',
			];
		}

		if (EshopOrdersConfig::load('enableDeliveryListPrint')) {
			$defaultLang  = $this->langsService->getLangs()[$this->translator->getDefaultLocale()];
			$navigation[] = [
				'link'   => $this->link('Orders:printDeliveryList', ['ids' => [$order->getId()]]),
				'ico'    => 'file-pdf',
				'title'  => $order->lang === $defaultLang->getTag()
					? 'eshopOrders.menu.showDeliveryList'
					: $this->t('eshopOrders.menu.showDeliveryListLang', [
						'lang' => $this->langsService->getLangs(false)[$order->lang]->getShortTitle(),
					]),
				'target' => '_blank',
			];

			if ($order->lang !== $defaultLang->getTag()) {
				$navigation[] = [
					'link'   => $this->link('Orders:printDeliveryList', [
						'ids'  => [$order->getId()],
						'lang' => $defaultLang->getTag(),
					]),
					'ico'    => 'file-pdf',
					'title'  => $this->t('eshopOrders.menu.showDeliveryListLang', ['lang' => $defaultLang->getShortTitle()]),
					'target' => '_blank',
				];
			}
		}

		if (EshopOrdersConfig::load('correctiveTaxDocument.enable')) {
			if (!$order->isCorrectiveTaxDocument && $order->getOrderForCorrectiveTaxDocument() === null) {
				$navigation[] = [
					'link'  => $this->link('correctiveTaxDocumentForm!'),
					'ico'   => 'fas fa-copy',
					'title' => 'eshopOrders.menu.createCorrectiveTaxDocument',
					'class' => 'ajax',
				];
			} else if ($order->getOrderForCorrectiveTaxDocument() !== null) {
				$navigation[] = [
					'link'  => $this->link('editOrder', $order->getOrderForCorrectiveTaxDocument()->getId()),
					'ico'   => 'fas fa-copy',
					'title' => 'eshopOrders.menu.showCorrectiveTaxDocument',
				];
			}
		}

		if (EshopOrdersConfig::load('orderForm.enableOrderDiscounts')) {
			$navigation[] = [
				'link'  => $this->link('orderDiscountForm!'),
				'ico'   => 'fas fa-percentage',
				'title' => 'eshopOrders.menu.addOrderDiscount',
				'class' => 'ajax',
			];
		}

		if (class_exists('EshopSales\DI\EshopSalesExtension') && EshopOrdersConfig::load('orderForm.enableOrderDiscountCodes')) {
			$navigation[] = [
				'link'  => $this->link('orderDiscountCodeForm!'),
				'ico'   => 'fas fa-percent',
				'title' => 'eshopOrders.menu.addOrderDiscountCode',
				'class' => 'ajax',
			];
		}

		$e        = new OrdersNavigationEvent($navigation);
		$e->order = $order;
		$this->eventDispatcher->dispatch($e, 'eshopOrders.admin.onBeforeRenderEditOrderNavigation');

		$this->transformNavigation($navigation);

		$this['navigation']->setData(['header' => $navigation]);

		if (!$order->isCorrectiveTaxDocument) {
			$title = $this->t('eshopOrders.title.editOrder', ['id' => $order->getId()]);
		} else if ($order->correctiveTaxDocumentOf !== null) {
			$title = $this->t('eshopOrders.title.correctiveTaxDocumentFormWithOrder', ['order' => $order->correctiveTaxDocumentOf->getId()]);
		} else if ($order->isCorrectiveTaxDocument) {
			$title = $this->t('eshopOrders.title.correctiveTaxDocumentForm');
		}

		if (isset($title)) {
			$this->setHeader($title, 'fa fa-shopping-cart');
		}

		$this->template->order = $order;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersGrid(IOrdersGridFactory $factory): OrdersGrid { return $factory->create(); }

	protected function createComponentOrderForm(IOrderFormFactory $factory): OrderForm
	{
		$control = $factory->create();

		if ($this->getParameter('id')) {
			$control->setOrder((int) $this->getParameter('id'));
		}

		return $control;
	}

	protected function createComponentOrderStatusForm(IOrderStatusFormFactory $factory): OrderStatusForm
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id')) {
			$control->setOrder((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawStatuses();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawStatuses();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderItemForm(IOrderItemFormFactory $factory): OrderItemForm
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id')) {
			$control->setOrder((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderItemAddForm(IOrderItemAddFormFactory $factory): OrderItemAddForm
	{
		$control = $factory->create((int) $this->getParameter('id'));

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function() {
				$this->redrawControl('flashes');
				$this['orderForm']->redrawControl('items');
				$this['orderForm']->redrawControl('priceTotal');
			};
			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
				$this['orderForm']->redrawControl('items');
				$this['orderForm']->redrawControl('priceTotal');
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentOrderSpeditionForm(IOrderSpeditionFormFactory $factory): OrderSpeditionForm
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id')) {
			$control->setOrder((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('spedition');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('spedition');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderPaymentForm(IOrderPaymentFormFactory $factory): OrderPaymentForm
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id')) {
			$control->setOrder((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('payment');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('payment');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderAddressDeliveryForm(IOrderAddressFormFactory $factory): OrderAddressForm
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id')) {
			$control->setOrderAddress((int) $this->getParameter('id'), 'delivery');
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderAddressInvoiceForm(IOrderAddressFormFactory $factory): OrderAddressForm
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id')) {
			$control->setOrderAddress((int) $this->getParameter('id'), 'invoice');
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentCorrectiveTaxDocumentForm(ICorrectiveTaxDocumentFormFactory $factory): CorrectiveTaxDocumentForm
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentUploadExternalInvoice(IUploadExternalInvoiceFactory $factory): UploadExternalInvoice
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderDiscountForm(IOrderDiscountFormFactory $factory): OrderDiscountForm
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSave[]         = function() {
			$this->redirect('this');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->redirect('this');
		};

		return $control;
	}

	protected function createComponentOrderDiscountCodeForm(IOrderDiscountCodeFormFactory $factory): OrderDiscountCodeForm
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			if ($form->getCustomData('isOk')) {
				$this->redirect('this');
			} else {
				$this->redrawControl('flashes');
			}
		};

		return $control;
	}

	protected function createComponentInvoiceEditForm(IInvoiceEditFormFactory $factory): InvoiceEditForm
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id'))->getInvoice());

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentCustomerInfoForm(ICustomerInfoFormFactory $factory): CustomerInfoForm
	{
		return $factory->create($this->ordersService->getFullOrder((int) $this->getParameter('id')));
	}

	/*******************************************************************************************************************
	 * ============================== Others
	 */

	protected function transformNavigation(array &$navigation): void
	{
		foreach ($navigation as $k => $nav) {
			if (is_array($nav['link'])) {
				$navigation[$k]['link'] = $this->link($nav['link'][0], $nav['link'][1]);
			}
		}
	}
}
