<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Model;

use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\EntityHelpers;
use Core\Model\ORM\Query;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\ApiModule\Api\V1\BodyEntity;
use EshopOrders\ApiModule\Api\V1\Model\Helpers\CustomerHelper;
use EshopOrders\FrontModule\Model\Customers as FrontCustomers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\Spedition;
use Exception;
use Tracy\Debugger;
use Users\Model\Entities\User;

class Customers
{
	protected EntityManagerDecorator $em;
	protected Sites                  $sites;
	protected Countries              $countries;
	protected CustomerHelper         $customerHelper;
	protected CustomerGroups         $customerGroups;
	protected FrontCustomers         $customers;
	protected Payments               $payments;
	protected Speditions             $speditions;

	public function __construct(
		EntityManagerDecorator $em,
		Sites                  $sites,
		Countries              $countries,
		CustomerHelper         $customerHelper,
		CustomerGroups         $customerGroups,
		FrontCustomers         $customers,
		Payments               $payments,
		Speditions             $speditions
	)
	{
		$this->em             = $em;
		$this->sites          = $sites;
		$this->countries      = $countries;
		$this->customerHelper = $customerHelper;
		$this->customerGroups = $customerGroups;
		$this->customers      = $customers;
		$this->payments       = $payments;
		$this->speditions     = $speditions;
	}

	public function filterIds(array $filter): array
	{
		$result = [];

		$qb = $this->em->getRepository(Customer::class)->createQueryBuilder('c')
			->select('c.id')
			->innerJoin('c.user', 'user', Join::WITH, 'user.isActive = 1');

		if (isset($filter['term'])) {
			$qb->andWhere('user.name LIKE :term OR user.lastname LIKE :term OR user.email LIKE :term')
				->setParameter('term', $filter['term']);
		}

		if (isset($filter['id'])) {
			if (!is_array($filter['id'])) {
				$filter['id'] = [$filter['id']];
			}

			$qb->andWhere('c.id IN (:ids)')
				->setParameter('ids', $filter['id']);
		}

		if (isset($filter['lastModified'])) {
			$tmp = explode(' ', $filter['lastModified'], 2);

			if (isset($tmp[0], $tmp[1])) {
				$qb->andWhere("user.lastModified {$tmp[0]} :lastModified")
					->setParameter('lastModified', $tmp[1]);
			}
		}

		if (isset($filter['customerIds'])) {
			$qb->andWhere('c.id IN (' . implode(',', $filter['customerIds']) . ')');
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$result[] = $row['id'];
		}

		return $result;
	}

	public function getCustomer(int $customerId): ?Dao\Customer
	{
		$rowC = $this->em->getRepository(Customer::class)->createQueryBuilder('c')
			->select('c.id, user.id as userId, c.phone, user.name as firstName, user.lastname as lastName, user.email, 
				IDENTITY(c.addressDelivery) as delivery, IDENTITY(c.addressInvoice) as invoice, 
				IDENTITY(c.groupCustomers) as group')
			->innerJoin('c.user', 'user', Join::WITH, 'user.isActive = 1')
			->where('c.id = :id')
			->setParameter('id', $customerId)
			->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

		if (!$rowC) {
			return null;
		}

		$customer            = new Dao\Customer($customerId, $rowC['userId']);
		$customer->firstName = $rowC['firstName'];
		$customer->lastName  = $rowC['lastName'];
		$customer->email     = $rowC['email'];
		$customer->phone     = $rowC['phone'];

		$customer->group = $this->customerGroups->getAll()[$rowC['group']] ?? null;

		$addresses = [];
		foreach ($this->em->getRepository(CustomerAddress::class)->createQueryBuilder('a')
			         ->select('a, IDENTITY(a.country) as country')
			         ->innerJoin('a.customers', 'c', Join::WITH, 'c.id =:id')
			         ->setParameter('id', $customerId)
			         ->getQuery()->getArrayResult() as $row) {
			$row += $row[0];

			$addr              = new Dao\Address($row['id']);
			$addr->firstName   = $row['firstName'];
			$addr->lastName    = $row['lastName'];
			$addr->email       = $row['email'];
			$addr->phone       = $row['phone'];
			$addr->company     = $row['company'];
			$addr->street      = $row['street'];
			$addr->city        = $row['city'];
			$addr->postal      = $row['postal'];
			$addr->idNumber    = $row['idNumber'];
			$addr->vatNumber   = $row['vatNumber'];
			$addr->countryCode = $row['country'];

			$addresses[$addr->id] = $addr;
		}

		$customer->deliveryAddress = $addresses[$rowC['delivery']] ?? null;
		$customer->invoiceAddress  = $addresses[$rowC['invoice']] ?? null;
		$customer->addresses       = $addresses;

		return $customer;
	}

	public function getAddresses(int $customerId): array
	{
		$result = [];

		foreach ($this->em->getRepository(CustomerAddress::class)->createQueryBuilder('ca')
			         ->addSelect('IDENTITY(ca.country) as countryCode')
			         ->innerJoin('ca.customers', 'c', Join::WITH, 'c.id = :customer')
			         ->setParameter('customer', $customerId)
			         ->getQuery()->getArrayResult() as $row) {
			$row += $row[0];
			$dao = new Dao\Address($row['id']);
			Arrays::setObjectVars($dao, $row);

			$result[$row['id']] = $dao;
		}

		return $result;
	}

	public function setDisabledPayment(int $customerId, int $paymentId, int $value): bool
	{
		try {
			/** @var Customer $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if ($customer) {
				if ($value === 1) {
					if (!$customer->disabledPayments->containsKey($paymentId)) {
						$customer->disabledPayments->add($this->em->getReference(Payment::class, $paymentId));
					}
				} else {
					$customer->disabledPayments->remove($paymentId);
				}
			}

			$this->em->persist($customer);
			$this->em->flush();

			return true;
		} catch (Exception $e) {
			Debugger::log('Disable customer payment: ' . $e->getMessage(), 'api-eshoporders-customers');
			Debugger::log($e, 'api-eshoporders-customers');
		}

		return false;
	}

	public function setDisabledSpedition(int $customerId, int $speditionId, int $value): bool
	{
		try {
			/** @var Customer $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if ($customer) {
				if ($value === 1) {
					if (!$customer->disabledSpeditions->containsKey($speditionId)) {
						$customer->disabledSpeditions->add($this->em->getReference(Spedition::class, $speditionId));
					}
				} else {
					$customer->disabledSpeditions->remove($speditionId);
				}
			}

			$this->em->persist($customer);
			$this->em->flush();

			return true;
		} catch (Exception $e) {
			Debugger::log('Disable customer spedition: ' . $e->getMessage(), 'api-eshoporders-customers');
			Debugger::log($e, 'api-eshoporders-customers');
		}

		return false;
	}

	public function setCustomerGroup(int $customerId, ?int $groupId): bool
	{
		try {
			/** @var Customer $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if ($customer) {
				if ($groupId) {
					/** @var GroupCustomers $groupRef */
					$groupRef = $this->em->getReference(GroupCustomers::class, $groupId);

					$customer->setGroupCustomers($groupRef);
				} else {
					$customer->removeGroupCustomers();
				}

				$this->em->persist($customer);
				$this->em->flush();

				return true;
			}

			Debugger::log('Customer id ' . $customerId . ' not found', 'api-eshoporders-customers');
		} catch (Exception $e) {
			Debugger::log('Set Customer group: ' . $e->getMessage(), 'api-eshoporders-customers');
			Debugger::log($e, 'api-eshoporders-customers');
		}

		return false;
	}

	public function setAddress(int $customerId, BodyEntity\Address $address, ?int $addressId): ?int
	{
		try {
			/** @var Customer|null $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if (!$customer) {
				Debugger::log("Customer $customerId", 'api-eshoporders-customers');

				return null;
			}

			if ($addressId) {
				/** @var CustomerAddress|null $customerAddress */
				$customerAddress = $this->em->getRepository(CustomerAddress::class)->find($addressId);
				if (!$customerAddress) {
					Debugger::log("Customer $customerId addressId {$addressId} not found", 'api-eshoporders-customers');

					return null;
				}
			} else {
				$customerAddress = new CustomerAddress($customer);
			}

			$customerAddress->setFirstName($address->firstName);
			$customerAddress->setLastName($address->lastName);
			$customerAddress->company   = $address->company;
			$customerAddress->idNumber  = $address->idNumber;
			$customerAddress->vatNumber = $address->vatNumber;
			$customerAddress->email     = $address->email;
			$customerAddress->phone     = $address->phone;
			$customerAddress->street    = $address->street;
			$customerAddress->city      = $address->city;
			$customerAddress->postal    = $address->postal;
			$customerAddress->country   = $address->countryCode ? $this->countries->get($address->countryCode) : null;

			$this->em->persist($customerAddress);
			$this->em->flush();

			$this->em->getConnection()->executeQuery("INSERT IGNORE INTO eshop_orders__customer_customer_address (customer_id, address_id) VALUES (:customer, :address)", [
				'customer' => $customerId,
				'address'  => $customerAddress->getId(),
			]);

			return $customerAddress->getId();
		} catch (Exception $e) {
			Debugger::log('Set Address: ' . $e->getMessage(), 'api-eshoporders-customers');
			Debugger::log($e, 'api-eshoporders-customers');
		}

		return null;
	}

	public function deleteAddress(int $customerId, int $addressId): bool
	{
		try {
			$address = $this->em->getRepository(CustomerAddress::class)->createQueryBuilder('ca')
				->where('ca.id = :id')
				->innerJoin('ca.customers', 'caC', Join::WITH, 'caC.id = :customer')
				->setParameters([
					'id'       => $addressId,
					'customer' => $customerId,
				])->getQuery()->getOneOrNullResult();

			if ($address) {
				$this->em->remove($address);
				$this->em->flush();
			} else {
				Debugger::log("Customer {$customerId} address {$addressId} not found for remove", 'api-eshoporders-customers');
			}

			return true;
		} catch (Exception $e) {
			Debugger::log('Delete Address: ' . $e->getMessage(), 'api-eshoporders-customers');
			Debugger::log($e, 'api-eshoporders-customers');
		}

		return false;
	}

	public function setCustomerInvoiceAddress(int $customerId, int $addressId): bool
	{
		try {
			/** @var Customer|null $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if (!$customer) {
				return false;
			}

			/** @var CustomerAddress|null $address */
			$address = $this->em->getRepository(CustomerAddress::class)->find($addressId);
			if (!$address) {
				return false;
			}

			$customer->setAddressInvoice($address);

			if (!$customer->addressesAvailable->containsKey($addressId)) {
				$customer->addressesAvailable->set($addressId, $address);
			}

			$this->em->persist($customer);
			$this->em->flush();

			return true;
		} catch (Exception $e) {
			Debugger::log($e);
		}

		return false;
	}

	public function setCustomerDeliveryAddress(int $customerId, int $addressId): bool
	{
		try {
			/** @var Customer|null $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if (!$customer) {
				return false;
			}

			/** @var CustomerAddress|null $address */
			$address = $this->em->getRepository(CustomerAddress::class)->find($addressId);
			if (!$address) {
				return false;
			}

			if (!$customer->addressesAvailable->containsKey($addressId)) {
				$customer->addressesAvailable->set($addressId, $address);
			}

			$customer->setAddressDelivery($address);
			$this->em->persist($customer);
			$this->em->flush();

			return true;
		} catch (Exception $e) {
			Debugger::log($e);
		}

		return false;
	}

	public function updateCustomer(int $customerId, BodyEntity\Customer $data): ?int
	{
		/** @var Customer|null $customer */
		$customer = $this->em->getRepository(Customer::class)->find($customerId);
		if (!$customer) {
			return null;
		}

		$customer->setPhone($data->phone ?: '');

		$user           = $customer->user;
		$user->name     = $data->firstName;
		$user->lastname = $data->lastName;
		$user->email    = $data->email;
		$user->phone    = $data->phone;

		if ($data->parentCustomer) {
			$parentUserId = $this->em->getRepository(Customer::class)->createQueryBuilder('c')
				->select('IDENTITY(c.user) as userId')
				->where('c.id = :id')
				->setParameter('id', $data->parentCustomer)
				->setMaxResults(1)
				->getQuery()->getArrayResult()[0]['userId'] ?? null;

			if ($parentUserId) {
				$user->parent = $this->em->getRepository(User::class)->find((int) $parentUserId);
			}
		} else {
			$user->parent = null;
		}

		if (!empty($data->canManageUsers)) {
			$childUsers = [];

			foreach ($this->em->getRepository(Customer::class)->createQueryBuilder('c')
				         ->select('IDENTITY(c.user) as userId')
				         ->where('c.id IN (:ids)')
				         ->setParameter('ids', $data->canManageUsers)
				         ->getQuery()->getArrayResult() as $row) {
				$childUsers[] = (int) $row['userId'];
			}

			$user->setParam('childUsers', $childUsers);
		} else {
			if ($user->getParam('childUsers')) {
				$user->setParam('childUsers', null);
			}
		}

		$this->em->persist($customer);
		$this->em->persist($user);
		$this->em->flush();

		return $customer->getId();
	}

	public function createCustomer(BodyEntity\Customer $data): ?int
	{
		try {
			$this->em->beginTransaction();

			$customer = $this->customers->getOrCreateCustomer(null, $data->email, $data->firstName, $data->lastName, $data->phone);

			$efAllowMinimalOrderPrice = EntityHelpers::setExtraField($customer, 'allowMinimalOrderPrice', (string) (int) $data->allowOrderPrice);
			$efMinimalOrderPrice      = EntityHelpers::setExtraField($customer, 'minimalOrderPrice', (string) (int) $data->minimalOrderPrice);
			$efShowPricesWithoutVat   = EntityHelpers::setExtraField($customer, 'showPricesWithoutVat', (string) $data->showPricesWithoutVat);
			$this->em->persist($efShowPricesWithoutVat);

			$this->em->persist($efAllowMinimalOrderPrice);
			$this->em->persist($efMinimalOrderPrice);

			if ($data->disableSpeditions !== null) {
				foreach ($data->disableSpeditions as $disableSpeditionId) {
					$customer->disabledSpeditions->set($disableSpeditionId, $this->speditions->getReference($disableSpeditionId));
				}
			}

			if ($data->disablePayments !== null) {
				foreach ($data->disablePayments as $disablePaymentId) {
					$customer->disabledPayments->set($disablePaymentId, $this->payments->getReference($disablePaymentId));
				}
			}

			if ($data->group) {
				/** @var GroupCustomers $groupRef */
				$groupRef = $this->em->getReference(GroupCustomers::class, $data->group);

				$customer->setGroupCustomers($groupRef);
			}

			$this->em->persist($customer);

			if ($data->addressDelivery) {
				$customer->setAddressDelivery(new CustomerAddress($customer));
				$customer->addressDelivery->setFirstName($data->addressDelivery->firstName);
				$customer->addressDelivery->setLastName($data->addressDelivery->lastName);
				$customer->addressDelivery->setCompany($data->addressDelivery->company);
				$customer->addressDelivery->email              = $data->addressDelivery->email;
				$customer->addressDelivery->phone              = $data->addressDelivery->phone;
				$customer->addressDelivery->street             = $data->addressDelivery->street;
				$customer->addressDelivery->city               = $data->addressDelivery->city;
				$customer->addressDelivery->postal             = $data->addressDelivery->postal;
				$customer->addressDelivery->country            = $data->addressDelivery->countryCode ? $this->countries->get($data->addressDelivery->countryCode) : null;
				$customer->addressDelivery->idNumber           = $data->addressDelivery->idNumber;
				$customer->addressDelivery->vatNumber          = $data->addressDelivery->vatNumber;
				$customer->addressDelivery->validatedVatNumber = (int) $data->addressDelivery->validatedVatNumber;
				$this->em->persist($customer->addressDelivery);
			}

			if ($data->addressInvoice) {
				$customer->setAddressInvoice(new CustomerAddress($customer));
				$customer->addressInvoice->setFirstName($data->addressInvoice->firstName);
				$customer->addressInvoice->setLastName($data->addressInvoice->lastName);
				$customer->addressInvoice->setCompany($data->addressInvoice->company);
				$customer->addressInvoice->email              = $data->addressInvoice->email;
				$customer->addressInvoice->phone              = $data->addressInvoice->phone;
				$customer->addressInvoice->street             = $data->addressInvoice->street;
				$customer->addressInvoice->city               = $data->addressInvoice->city;
				$customer->addressInvoice->postal             = $data->addressInvoice->postal;
				$customer->addressInvoice->country            = $data->addressInvoice->countryCode ? $this->countries->get($data->addressInvoice->countryCode) : null;
				$customer->addressInvoice->idNumber           = $data->addressInvoice->idNumber;
				$customer->addressInvoice->vatNumber          = $data->addressInvoice->vatNumber;
				$customer->addressInvoice->validatedVatNumber = (int) $data->addressInvoice->validatedVatNumber;
				$this->em->persist($customer->addressInvoice);
			}

			$this->em->flush();
			$this->em->commit();

			return $customer->user->getId();
		} catch (Exception $ex) {
			Debugger::log($ex);

			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}
		}

		return null;
	}

	public function deleteCustomer(int $customerId): bool
	{
		try {
			/** @var Customer|null $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if (!$customer) {
				return true;
			}

			$this->em->remove($customer->user);
			$this->em->flush();

			return true;
		} catch (Exception $e) {
			Debugger::log($e);
		}

		return false;
	}
}
