<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\Helpers\Arrays;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Nette\Utils\ArrayHash;

class CartDetailItem extends BaseControl
{
	protected ?int $itemId = null;

	protected Carts $cartsService;
	protected Sites $sites;

	public function __construct(
		Carts $carts,
		Sites $sites
	)
	{
		$this->cartsService = $carts;
		$this->sites        = $sites;
	}

	public function render(): void
	{
		$this->template->item = $this->getItem();
		$this->template->render($this->getTemplateFile(null, 'CartDetailItem'));
	}

	public function handleRemove(): void
	{
		$cartItems = $this->cartsService->getCurrentCart()->getCartItems();
		/** @var CartItem|null $item */
		$item = $this->getItem() && $cartItems[$this->getItem()->getId()] ? $cartItems[$this->getItem()->getId()] : null;

		if (!$item) {
			return;
		}

		$event = new UpdatedCartItemEvent(
			(string) $item->getId(),
			0,
			(int) $item->quantity,
			[],
			$item->getProductId(),
		);

		$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
	}

	protected function createComponentForm(): ?BaseForm
	{
		$form          = $this->createForm();
		$item          = $this->getItem();
		$minimumAmount = $item->getProduct() ? $item->getProduct()->minimumAmount : 1;
		$maximumAmount = $item->getProduct() ? $item->getProduct()->getMaximumAmount() : null;
		$site          = $this->sites->getCurrentSite();

		$form->addText('quantity', null)
			->addRule($form::FILLED)
			->setHtmlAttribute('data-add-to-cart-quantity-input', 'instant')
			->setHtmlAttribute('data-item-id', $item->getId())
			->setDefaultValue($item->quantity);
		$form->getComponent('quantity')
			->setHtmlAttribute('data-minimum-amount', $minimumAmount)
			->addRule(
				BaseForm::MIN,
				'eshopCatalogFront.cart.minimumQuantity',
				$minimumAmount > 1 ? $minimumAmount : 0
			);
		$form->addHidden('itemId', (string) $item->getId());

		$product                       = $item->product;
		$cartProductMaxQuantityByEshop = Config::load('cartProductMaxQuantityByEshop', []) ?? [];

		if (
			(($product
					&& Config::load('pseudoWarehouse')
					&& $product->unlimitedQuantity === 0
					&& $product->getQuantity() > 0
				)
				&& (!$maximumAmount || $maximumAmount > $product->getQuantity() + $product->getQuantityExternal())
				&& (!Config::load('allowWarehouseOverdraft') || !$product->getAvailability()->warehouseOverdraft))
			|| (
				$product
				&& Arrays::contains($cartProductMaxQuantityByEshop, $site->getIdent())
				&& (!$product->availabilityAfterSoldOut || !$product->availability->canAddToCart())
			)
		) {
			$maximumAmount = $product->getQuantity() + $product->getQuantityExternal();
		}

		if ($maximumAmount) {
			$form->getComponent('quantity')->addRule(BaseForm::MAX, 'eshopCatalogFront.cart.maximumQuantity', $maximumAmount);
			$form->getComponent('quantity')->setHtmlAttribute('data-max', $maximumAmount);
		}

		if (
			($product && $product->isAssort)
			|| $item->getData('note')
			|| (Config::load('product.allowNote') && $product && $product->getExtraField('addNote'))
		) {
			$lines = explode("\n", (string) $item->getData('note'));
			$form->addTextArea('note', $item->getData('noteTitle'))
				->setHtmlAttribute('rows', max(2, count($lines)))
				->setDefaultValue($item->getData('note'));
		}

		$childsContainer = $form->addContainer('childs');
		foreach ($item->getChilds() as $child) {
			$childContainer = $childsContainer->addContainer('child' . $child->getId());

			if ($child->getData('note')) {
				$childContainer->addTextArea('note', $child->getData('noteTitle'))
					->setDefaultValue($child->getData('note'));
			}
		}

		$this->monitor(FrontPresenter::class, function(FrontPresenter $presenter): void {
			$presenter['jsTranslates']->addTranslate(
				'maximumQuantityOfThisProductInCart',
				['eshopCatalogFront.cart.maximumQuantityOfThisProductInCart', ['quantity' => '__QUANTITY__',]]
			);
		});

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$cartItems = $this->cartsService->getCurrentCart()->getCartItems();
			$event     = new UpdatedCartItemEvent(
				(string) $values->itemId,
				(int) $values->quantity,
				(int) $cartItems[$values->itemId]->quantity,
				[],
				(int) $cartItems[$values->itemId]->productId,
			);

			if ($values->note) {
				$event->moreData['note'] = $values->note;
			}

			$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');

			foreach ($values->childs ?? [] as $childId => $childVals) {
				if (empty((array) $childVals)) {
					continue;
				}

				$childId = substr($childId, 5);
				$child   = $cartItems[$values->itemId]->getChilds()[$childId];

				$event = new UpdatedCartItemEvent(
					(string) $child->getId(),
					(int) $child->quantity,
					(int) $child->quantity,
					[],
					(int) $child->productId,
				);

				if ($childVals->note) {
					$event->moreData['note'] = $childVals->note;
				}

				$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
			}
		} catch (\Exception $e) {
			$this->presenter->flashMessageWarning('eshopOrders.cart.countUpdateErr');
			$this->presenter->redrawControl('flashes');

			return false;
		}

		return true;
	}

	protected function getItem(): ?CartItem
	{
		return $this->cartsService->getCartItem($this->itemId);
	}

	public function setItem(int $itemId): self
	{
		$this->itemId = $itemId;

		return $this;
	}
}

