<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\OrderFacade;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use Nette\Utils\Paginator;

class OrdersList extends BaseControl
{
	/** @persistent */
	public int $page = 1;

	public int $limit = 5;

	protected Orders      $ordersService;
	protected Customers   $customersService;
	protected OrderFacade $orderFacade;

	protected ?Customer  $customer  = null;
	protected ?Paginator $paginator = null;

	public function __construct(
		Orders      $orders,
		Customers   $customers,
		OrderFacade $orderFacade
	)
	{
		$this->ordersService    = $orders;
		$this->customersService = $customers;
		$this->orderFacade      = $orderFacade;
	}

	public function render(): void
	{
		$this->template->setFile($this->getTemplateFile());

		$customer = $this->getCustomer();

		if (!$customer) {
			return;
		}

		$paginator = $this->getPaginator();
		$orders    = $this->ordersService->getByCustomer($customer, $paginator->getItemsPerPage(), $paginator->getOffset());

		$this->template->orders    = $orders;
		$this->template->paginator = $paginator;
		$this->template->showUser  = EshopOrdersConfig::load('showOrdersOfChildrenUsers');

		$this->template->render($this->getTemplateFile());
	}

	public function handleRepeatOrder(int $id): void
	{
		$presenter = $this->presenter;

		$result = $this->orderFacade->repeatOrder($id);
		if ($result['added'] === 0) {
			$presenter->flashMessageDanger('eshopOrdersFront.list.repeatOrderNoProducts');
		} else if ($result['notQuantity'] > 0) {
			$presenter->flashMessageInfo('eshopOrdersFront.list.repeatOrderNoQuantity');
		} else {
			$presenter->flashMessageSuccess('eshopOrdersFront.list.repeatOrderSuccess');
		}

		if ($result['status'] === 'ok') {
			$event = new Event(['orderId' => $id]);
			$this->eventDispatcher->dispatch($event, 'eshopOrders.repeatOrder');

			$presenter->redirect(':EshopOrders:Front:Default:order');
		} else {
			if ($presenter->isAjax()) {
				$presenter->redrawControl('flashes');
			} else {
				$presenter->redirect('this');
			}
		}
	}

	public function handlePaginator(int $page): void
	{
		$this->page = $page;

		$this->redrawControl('list');
		$this->redrawControl('paginator');
	}

	public function getPaginator(): Paginator
	{
		if ($this->paginator === null) {
			$itemsPerPage = $this->limit;

			$paginator = new Paginator();
			$paginator->setItemCount($this->getCustomer() ? $this->ordersService->getOrdersCountByCustomer($this->getCustomer()) : 0);
			$paginator->setItemsPerPage($itemsPerPage);
			$paginator->setPage($this->page);

			$this->paginator = $paginator;
		}

		return $this->paginator;
	}

	protected function getCustomer(): ?Customer
	{
		if ($this->customer === null) {
			$userId         = (int) $this->getPresenter()->getUser()->id;
			$this->customer = $this->customersService->getByUserId($userId);
		}

		return $this->customer;
	}
}

