<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use Core\Model\Application\AppState;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopCatalog\Model\Entities\Availability;
use EshopOrders\Model\Helpers\OrderHelper;

class Cart
{
	public int        $id;
	public ?string    $ident                = null;
	public ?array     $cartItems            = null;
	public ?Spedition $spedition            = null;
	public ?Payment   $payment              = null;
	public array      $discounts            = [];
	public array      $gifts                = [];
	public ?int       $orderGiftId          = null;
	public float      $minimalOrderPrice    = 0;
	protected array   $cCalculatedDiscounts = [];

	/** @var PaymentSpedition[] */
	public array $paymentSpeditionFreeCombinations = [];

	public function setId(int $id): self
	{
		$this->id = $id;

		return $this;
	}

	public function setIdent(?string $ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	public function setCartItems(array $cartItems): self
	{
		$this->cartItems = $cartItems;

		return $this;
	}

	public function getId(): int
	{
		return $this->id;
	}

	public function getIdent(): ?string
	{
		return $this->ident;
	}

	/** @return CartItem[] */
	public function getCartItems(): array
	{
		return $this->cartItems === null ? [] : $this->cartItems;
	}

	public function getTotalQuantity(): int
	{
		$result = 0;

		foreach ($this->getCartItems() as $item) {
			$result += $item->getQuantity();
		}

		return $result;
	}

	public function getCartItemByProductId(int $id): ?CartItem
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->productId === $id) {
				return $item;
			}
		}

		return null;
	}

	public function getCartItemByIdent(string $ident): ?CartItem
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->ident === $ident) {
				return $item;
			}
		}

		return null;
	}

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function addGift(Gift $gift): self
	{
		$this->gifts[$gift->id] = $gift;

		return $this;
	}

	public function getCartItemsPrice(bool $withVat = true, bool $useBaseCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			$priceTotal += $withVat ? $item->getTotalPrice($useBaseCurrency) : $item->getTotalPriceWithoutVat($useBaseCurrency);
		}

		return $priceTotal;
	}

	/**
	 * @return CartItem[]
	 */
	public function getCartItemsForGifts(): array
	{
		$items = [];
		foreach ($this->cartItems as $item) {
			if ($item->product && !$item->product->orderGiftsAllowed) {
				continue;
			}

			$items[] = $item;
		}

		return $items;
	}

	public function getPriceForFreeSpedition(): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			if (EshopCatalogConfig::load('product.allowModifySpedition', false) && $item->disableCalculateFreeSpedition) {
				continue;
			}
			$priceTotal += $item->getTotalPrice();
		}

		return $priceTotal;
	}

	public function getPriceTotal(): float
	{
		$speditionPrice = $this->spedition ? $this->spedition->getPriceActual($this) : 0;
		$paymentPrice   = $this->payment ? $this->payment->getPriceActual($this) : 0;

		return $this->getPriceTotalWithoutSpedition(true) + $speditionPrice + $paymentPrice;
	}

	public function getPriceTotalWithoutVat(): float
	{
		$speditionPrice = $this->spedition ? $this->spedition->getPriceActualWithoutVat($this) : 0;
		$paymentPrice   = $this->payment ? $this->payment->getPriceActualWithoutVat($this) : 0;

		return $this->getPriceTotalWithoutSpedition(false) + $speditionPrice + $paymentPrice;
	}

	public function getPriceTotalWithoutSpedition(bool $withVat = true): float
	{
		$resultPrice = $this->getCartItemsPrice($withVat);

		if ($withVat) {
			return $resultPrice + $this->calculateDiscounts($withVat);
		}

		$itemsPrice = 0;
		$dphRates   = [];

		foreach ($this->getCartItems() as $item) {
			if ($item->discountDisabled) {
				continue;
			}

			$itemsPrice += $item->getTotalPrice(false);
		}

		$zeroPrice = $itemsPrice === 0;

		foreach ($this->getCartItems() as $item) {
			if ($item->discountDisabled) {
				continue;
			}

			$vat = $item->getVatRate();
			if (!isset($dphRates[$vat])) {
				$dphRates[$vat] = 0;
			}

			if (!$zeroPrice) {
				$dphRates[$vat] += $item->getTotalPrice(false);
			}
		}

		$totalSale = 0;
		foreach ($this->discounts as $discount) {
			$totalSale += abs($discount->calculateDiscount(false));
		}

		foreach (OrderHelper::calculateVatWithSales($itemsPrice, $totalSale, $dphRates) as $v) {
			$resultPrice += round($v['withoutVat'], AppState::getState('eshopOrdersCartDecimals') ?? 0);
		}

		return round($resultPrice, AppState::getState('eshopOrdersCartDecimals') ?? 0);
	}

	public function getItemsCount(): int
	{
		$count = 0;
		foreach ($this->cartItems as $item) {
			$count += $item->getQuantity();
		}

		return $count;
	}

	public function addDiscount(string $key, Discount $discount): self
	{
		$this->discounts[$key] = $discount;

		return $this;
	}

	public function removeDiscount(string $key): self
	{
		unset($this->discounts[$key]);

		return $this;
	}

	public function getDiscount(string $key): ?Discount { return $this->discounts[$key] ?? null; }

	/** @return Discount[] */
	public function getDiscounts(): iterable
	{
		// Přepočítá slevu, může tam být uložena sleva bez DPH
		$this->calculateDiscounts();

		$discounts = [];
		foreach ($this->discounts as $k => $discount) {
			$discounts[$k] = $discount;
		}

		return $discounts;
	}

	public function setSpedition(Spedition $spedition): self
	{
		$spedition->setCart($this);
		$this->spedition = $spedition;

		return $this;
	}

	public function setPayment(Payment $payment): self
	{
		$payment->setCart($this);
		$this->payment = $payment;

		return $this;
	}

	public function calculateDiscounts(bool $withVat = true): float
	{
		$k = (string) $withVat;

		if (!isset($this->cCalculatedDiscounts[$k])) {
			$price = 0;
			foreach ($this->discounts as $discount) {
				$price += $discount->calculateDiscount();
			}

			$this->cCalculatedDiscounts[$k] = $price;
		}

		return $this->cCalculatedDiscounts[$k];
	}

	public function calculateDiscount(Discount $discount, bool $withVat = true, bool $useBaseCurrency = false): float
	{
		return $discount->calculateDiscount($useBaseCurrency);
	}

	/** @return CartItem[] */
	public function getProductsForDiscount(): array
	{
		$items = [];

		foreach ($this->cartItems as $k => $item) {
			if ($item->discountDisabled) {
				continue;
			}

			$items[$k] = $item;
		}

		return $items;
	}

	public function hasOversizedProduct(): bool
	{
		foreach ($this->cartItems as $item) {
			if (
				($item->product && $item->product->isOversize === true)
				|| $item->getData('isOversize') === true
			) {
				return true;
			}
		}

		return false;
	}

	public function hasDisabledPickUpSpedition(): bool
	{
		foreach ($this->cartItems as $item) {
			if ($item->product && $item->product->hasDisabledPickupPoints()) {
				return true;
			}
		}

		return false;
	}

	public function getOrderGift(): ?Gift
	{
		if (!$this->orderGiftId) {
			return null;
		}

		return $this->gifts[$this->orderGiftId] ?? null;
	}

	public function getRemainingValueToCompleteOrder(): float
	{
		return $this->minimalOrderPrice > 0 ? $this->minimalOrderPrice - $this->getCartItemsPrice() : 0;
	}

	public function hasPreorderProduct(): bool
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->getProduct() && $item->getProduct()->getAvailability()->getIdent() === Availability::PREORDER) {
				return true;
			}
		}

		return false;
	}

	public function getTotalWeight(): int
	{
		$weight = 0;

		foreach ($this->getCartItems() as $item) {
			$weight += $item->getTotalWeight();
		}

		return $weight;
	}

	public function getTotalWeightInKG(): float
	{
		$totalWeight = $this->getTotalWeight();

		if ($totalWeight > 0) {
			return round($totalWeight / 1000, 2);
		}

		return 0;
	}
}
