<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Mailing\TemplateFactory;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Core\Model\Settings;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\EmailEvent;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use EshopOrders\Model\Utils\Helpers;
use EshopOrders\Model\Utils\Strings;
use Exception;
use Mpdf\Output\Destination;
use Nette\Application\LinkGenerator;
use Nette\Bridges\ApplicationLatte\Template;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Mail\Mailer;
use Nette\Mail\Message;
use Nette\Utils\Validators;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderMailerSubscriber implements EventSubscriberInterface
{
	protected Translator                $translator;
	protected TemplateFactory           $templateFactory;
	protected Mailer                    $mailer;
	protected SessionSection            $orderSuccessMessages;
	protected Invoices                  $invoices;
	protected IInvoicePdfBuilderFactory $invoicePdfBuilderFactory;
	protected EventDispatcher           $eventDispatcher;
	protected Sellers                   $sellers;
	protected Payments                  $paymentsService;
	protected QrGenerator               $qrGenerator;
	protected LinkGenerator             $linkGenerator;
	protected Settings                  $settings;

	public function __construct(
		Translator                $translator,
		TemplateFactory           $templateFactory,
		Mailer                    $mailer,
		Session                   $session,
		Invoices                  $invoices,
		IInvoicePdfBuilderFactory $invoicePdfBuilderFactory,
		EventDispatcher           $eventDispatcher,
		Sellers                   $sellers,
		Payments                  $payments,
		QrGenerator               $qrGenerator,
		LinkGenerator             $linkGenerator,
		Settings                  $settings
	)
	{
		$this->translator               = $translator;
		$this->templateFactory          = $templateFactory;
		$this->orderSuccessMessages     = $session->getSection('eshopOrders/orderSuccessMessages');
		$this->invoices                 = $invoices;
		$this->invoicePdfBuilderFactory = $invoicePdfBuilderFactory;
		$this->eventDispatcher          = $eventDispatcher;
		$this->sellers                  = $sellers;
		$this->paymentsService          = $payments;
		$this->qrGenerator              = $qrGenerator;
		$this->linkGenerator            = $linkGenerator;
		$this->settings                 = $settings;
		$this->mailer                   = $mailer;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'          => ['orderOnSuccess', 100],
			'eshopOrders.sendActualOrder'         => ['sendActualOrder', 100],
			'eshopOrders.sendCardPaymentReminder' => ['sendCardPaymentReminder', 100],
			'eshopOrders.sendInvoice'             => ['sendInvoice', 100],
			'eshopOrders.sendInvoiceReminder'     => ['sendInvoiceReminder', 100],
			'eshopOrders.sendSurvey'              => ['sendSurvey', 100],
			'eshopOrders.sendRequestsForReview'   => ['sendRequestsForReview', 100],
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!EshopOrdersConfig::load('orderForm.send')) {
			return;
		}

		try {
			$order = $event->order;
			$this->translator->setLocale($order->lang);
			Sites::$currentIdentOverride = $order->lang;

			$seller     = $this->sellers->getSellerForSite($order->site->getIdent());
			$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/orderSend.latte';
			$originFile = __DIR__ . '/../../_emails/orderSend.latte';

			if (!file_exists($file)) {
				$file = $originFile;
			}

			$siteIdent   = $order->site->getIdent();
			$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
			$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

			if (!$sellerEmail) {
				return;
			}

			$template = $this->templateFactory->create($siteIdent, $order->lang);

			$subject = $this->translator->translate('eshopOrders.emails.subject', [
				'orderId'  => $order->getId(),
				'siteName' => $template->siteName,
			]);

			$country = $order->getAddressInvoice() && $order->getAddressInvoice()->getCountry()
				? Strings::upper($order->getAddressInvoice()->getCountry()->getId())
				: null;

			$customer      = $order->getCustomer();
			$canShowQrCode = EshopOrdersConfig::load('emails.showQrPayment')
				&& $country && Arrays::contains(EshopOrdersConfig::load('showQrForCountries'), $country)
				&& $seller
				&& ($order->getPaymentIdent() === 'transfer' || EshopOrdersConfig::load('emails.showQrPaymentInAllEmails'))
				&& (
					!$customer
					|| !$customer->getGroupCustomers()
					|| !Arrays::contains(EshopOrdersConfig::load('emails.hideQrPaymentForCustomerGroup'), $customer->getGroupCustomers()->getId())
				);

			$template->setFile($file);
			$template->setParameters([
				'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
				'subject'        => $subject,
				'order'          => $order,
				'originTemplate' => $originFile,
				'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
					&& in_array($order->getPaymentIdent(), ['card', 'essox'])
					&& $this->paymentsService->getByIdent('card', true),
				'qrCode'         => $canShowQrCode ? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller) : null,
			]);

			$message = new Message();
			$message->setSubject($subject);
			$message->setFrom($sellerEmail, $sellerName);

			if (EshopOrdersConfig::load('enableEditTermsAndConditionPdfFile')) {
				$termsAndConditionsPdf = $this->settings->get('eshopOrdersTermsAndConditionsPdf')[$this->translator->getLocale()] ?? null;
				if ($termsAndConditionsPdf && file_exists(WWW_DIR . $termsAndConditionsPdf)) {
					$message->addAttachment(
						$this->translator->translate('eshopOrdersFront.cart.termsAndConditions') . '.pdf',
						(string) file_get_contents(WWW_DIR . $termsAndConditionsPdf),
					);
				}
			}

			if (EshopOrdersConfig::load('enableEditComplaintReportPdfFile')) {
				$complaintReportPdf = $this->settings->get('eshopOrdersComplaintReportPdf')[$this->translator->getLocale()] ?? null;
				if ($complaintReportPdf && file_exists(WWW_DIR . $complaintReportPdf)) {
					$message->addAttachment(
						$this->translator->translate('eshopOrdersFront.cart.complaintReport') . '.pdf',
						(string) file_get_contents(WWW_DIR . $complaintReportPdf),
					);
				}
			}

			$bcc = EshopOrdersConfig::load('orderForm.bcc');
			if ($bcc) {
				foreach (explode(',', $bcc) as $v) {
					if ($v && Validators::isEmail(trim($v))) {
						$message->addBcc(trim($v));
					}
				}
			}

			$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailCreateOrder');

			$message->setHtmlBody($template->renderToString());
		} catch (Exception $e) {
			Debugger::log('Order email build failed - ' . $event->order->getId(), 'eshopOrderEmail');
			Debugger::log($e, 'eshopOrderEmail');
			LogNotifier::toDevelopers('Order email build failed - ' . $event->order->getId());

			return;
		}

		if ($message->getSubject()) {
			// Customer
			try {
				if (
					EshopOrdersConfig::load('orderForm.sendTo') === 'customer'
					&& $order->getCustomer()
				) {
					$user = $order->getCustomer()->getUser();
					$message->addTo(
						$user->getEmail(),
						$user->getName() . ' ' . $user->getLastname(),
					);
				} else {
					$message->addTo(
						$order->getAddressInvoice()->getEmail(),
						$order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName(),
					);
				}

				$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailCreateOrderOnlyCustomer');

				$this->mailer->send($message);
			} catch (Exception $e) {
				Debugger::log('Order email customer send failed - ' . $event->order->getId(), 'eshopOrderEmail');
				Debugger::log($e, 'eshopOrderEmail');
				LogNotifier::toDevelopers('Order email customer send failed - ' . $event->order->getId());
			}

			// Seller
			try {
				$message->clearHeader('To');
				$message->clearHeader('Cc');
				$message->clearHeader('Bcc');

				if ($seller->sendOrderToSeller) {
					$message->addTo($sellerEmail, $sellerName);

					$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailCreateOrderOnlySeller');

					$this->mailer->send($message);
				}
			} catch (Exception $e) {
				Debugger::log('Order email seller send failed - ' . $event->order->getId(), 'eshopOrderEmail');
				Debugger::log($e, 'eshopOrderEmail');
				LogNotifier::toDevelopers('Order email seller send failed - ' . $event->order->getId());
			}
		} else {
			Debugger::log('Order email message missing - ' . $event->order->getId(), 'eshopOrderEmail');
			LogNotifier::toDevelopers('Order email message missing - ' . $event->order->getId());
		}
	}

	public function sendActualOrder(OrderEvent $event): void
	{
		$order = $event->order;
		$this->translator->setLocale($order->lang);
		Sites::$currentIdentOverride = $order->lang;

		$siteIdent  = $order->site->getIdent();
		$seller     = $this->sellers->getSellerForSite($siteIdent);
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/actualOrder.latte';
		$originFile = __DIR__ . '/../../_emails/actualOrder.latte';

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

		if (!file_exists($file)) {
			$file = $originFile;
		}

		if (!$sellerEmail) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.actualOrder', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$country = $order->getAddressInvoice() && $order->getAddressInvoice()->getCountry()
			? Strings::upper($order->getAddressInvoice()->getCountry()->getId())
			: null;

		$customer      = $order->getCustomer();
		$canShowQrCode = EshopOrdersConfig::load('emails.showQrPayment')
			&& $country && Arrays::contains(EshopOrdersConfig::load('showQrForCountries'), $country)
			&& $seller
			&& ($order->getPaymentIdent() === 'transfer' || EshopOrdersConfig::load('emails.showQrPaymentInAllEmails'))
			&& (
				!$customer
				|| !$customer->getGroupCustomers()
				|| !Arrays::contains(EshopOrdersConfig::load('emails.hideQrPaymentForCustomerGroup'), $customer->getGroupCustomers()->getId())
			);

		$template->setFile($file);
		$template->setParameters([
			'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
			'subject'        => $subject,
			'order'          => $order,
			'originTemplate' => $originFile,
			'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
				&& in_array($order->getPaymentIdent(), ['card', 'essox'])
				&& $this->paymentsService->getByIdent('card', true),
			'qrCode'         => $canShowQrCode ? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller) : null,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName);

		if (
			EshopOrdersConfig::load('orderForm.sendTo') === 'customer'
			&& $order->getCustomer()
		) {
			$user = $order->getCustomer()->getUser();
			$message->addTo(
				$user->getEmail(),
				$user->getName() . ' ' . $user->getLastname(),
			);
		} else {
			$message->addTo(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName(),
			);
		}

		$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailActualStatus');

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendCardPaymentReminder(OrderEvent $event): void
	{
		$order = $event->order;
		$this->translator->setLocale($order->lang);
		Sites::$currentIdentOverride = $order->lang;

		$siteIdent  = $order->site->getIdent();
		$seller     = $this->sellers->getSellerForSite($siteIdent);
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/cardPaymentReminder.latte';
		$originFile = __DIR__ . '/../../_emails/cardPaymentReminder.latte';

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang) ?: null;

		if (!file_exists($file)) {
			$file = $originFile;
		}

		if (!$sellerEmail) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.cardPaymentReminder', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$country = $order->getAddressInvoice() && $order->getAddressInvoice()->getCountry()
			? Strings::upper($order->getAddressInvoice()->getCountry()->getId())
			: null;

		$customer      = $order->getCustomer();
		$canShowQrCode = EshopOrdersConfig::load('emails.showQrPayment')
			&& $country && Arrays::contains(EshopOrdersConfig::load('showQrForCountries'), $country)
			&& $seller
			&& (
				!$customer
				|| !$customer->getGroupCustomers()
				|| !Arrays::contains(EshopOrdersConfig::load('emails.hideQrPaymentForCustomerGroup'), $customer->getGroupCustomers()->getId())
			);

		$template->setFile($file);
		$template->setParameters([
			'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
			'subject'        => $subject,
			'order'          => $order,
			'originTemplate' => $originFile,
			'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
				&& in_array($order->getPaymentIdent(), ['card', 'essox'])
				&& $this->paymentsService->getByIdent('card', true),
			'qrCode'         => $canShowQrCode ? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller) : null,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName);

		if (
			EshopOrdersConfig::load('orderForm.sendTo') === 'customer'
			&& $order->getCustomer()
		) {
			$user = $order->getCustomer()->getUser();
			$message->addTo(
				$user->getEmail(),
				$user->getName() . ' ' . $user->getLastname(),
			);
		} else {
			$message->addTo(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName(),
			);
		}

		$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailCardPaymentReminder');

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendSurvey(OrderEvent $event): void
	{
		$order = $event->order;
		$this->translator->setLocale($order->lang);
		Sites::$currentIdentOverride = $order->lang;

		$seller     = $this->sellers->getSellerForSite($order->site->getIdent());
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/survey.latte';
		$originFile = __DIR__ . '/../../_emails/survey.latte';

		if (!file_exists($file)) {
			$file = $originFile;
		}

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $order->site->getIdent(), $order->lang);

		if (!$sellerEmail) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.survey', [
			'siteName' => $template->siteName,
		]);

		$template->setFile($file);
		$template->setParameters([
			'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
			'subject'        => $subject,
			'order'          => $order,
			'originTemplate' => $originFile,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName);

		if (
			EshopOrdersConfig::load('orderForm.sendTo') === 'customer'
			&& $order->getCustomer()
		) {
			$user = $order->getCustomer()->getUser();
			$message->addTo(
				$user->getEmail(),
				$user->getName() . ' ' . $user->getLastname(),
			);
		} else {
			$message->addTo(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName(),
			);
		}

		$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailSurvey');

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendInvoice(OrderEvent $event): void
	{
		$order = $event->order;

		if (!EshopOrdersConfig::load('invoice.enable')) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);
		$this->setTemplateFile($template, 'invoice');

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.invoice', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$message = $this->prepareEmail($event, $template, $subject);

		$invoice    = $this->invoices->getOneByOrder($order);
		$pdfBuilder = $this->invoicePdfBuilderFactory->create($invoice);
		$message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));
		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendInvoiceReminder(OrderEvent $event): void
	{
		$order = $event->order;

		if (!EshopOrdersConfig::load('customer.allowInvoiceReminder')) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);
		$this->setTemplateFile($template, 'invoiceReminder');

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.invoiceReminder.' . $order->invoiceReminderCount, [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$invoice    = $this->invoices->getOneByOrder($order);
		$template->add('invoice', $invoice);
		$message = $this->prepareEmail($event, $template, $subject);

		$pdfBuilder = $this->invoicePdfBuilderFactory->create($invoice);
		$message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));
		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendRequestsForReview(OrderEvent $event): void
	{
		$order = $event->order;

		if (!EshopOrdersConfig::load('orderItemReviews.enable')) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);
		$this->setTemplateFile($template, 'requestForReview');

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.review', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$message        = $this->prepareEmail($event, $template, $subject);
		$template->link = $this->linkGenerator->link('EshopOrders:Front:OrderItemReviews:giveReviews', ['orderIdent' => $order->getIdent()]);

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	protected function setTemplateFile(Template $template, string $fileName): Template
	{
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/' . $fileName . '.latte';
		$originFile = __DIR__ . '/../../_emails/' . $fileName . '.latte';

		if (!file_exists($file)) {
			$file = $originFile;
		}

		$template->originTemplate = $originFile;
		$template->setFile($file);

		return $template;
	}

	protected function prepareEmail(OrderEvent $event, Template $template, string $subject): ?Message
	{
		/** @var \Core\Model\Templating\Template $template */
		$order     = $event->order;
		$siteIdent = $order->site->getIdent();
		$seller    = $this->sellers->getSellerForSite($siteIdent);

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

		if (!$sellerEmail) {
			return null;
		}

		$template->setParameters($template->getParameters() + [
				'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
				'subject'        => $subject,
				'order'          => $order,
				'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
					&& in_array($order->getPaymentIdent(), ['card', 'essox'])
					&& $this->paymentsService->getByIdent('card', true),
			]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName);

		if (
			EshopOrdersConfig::load('orderForm.sendTo') === 'customer'
			&& $order->getCustomer()
		) {
			$user = $order->getCustomer()->getUser();
			$message->addTo(
				$user->getEmail(),
				$user->getName() . ' ' . $user->getLastname(),
			);
		} else {
			$message->addTo(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName(),
			);
		}

		return $message;
	}
}
