<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use Core\Model\Event\PresenterTemplateEvent;
use Core\Model\Helpers\Arrays;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Utils\Strings;

class FinishedPresenter extends BasePresenter
{
	/** @persistent */
	public ?string $orderIdent = null;

	public ?Order $order = null;

	protected Orders              $ordersService;
	protected Statuses            $statusesService;
	protected CardsPaymentService $cardsPaymentService;
	protected Sellers             $sellers;
	protected QrGenerator         $qrGenerator;

	protected array $includeTemplates = [];

	public function __construct(
		Orders              $orders,
		Statuses            $statuses,
		CardsPaymentService $cardsPaymentService,
		Sellers             $sellers,
		QrGenerator         $qrGenerator
	)
	{
		$this->ordersService       = $orders;
		$this->statusesService     = $statuses;
		$this->cardsPaymentService = $cardsPaymentService;
		$this->sellers             = $sellers;
		$this->qrGenerator         = $qrGenerator;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionOrderFinished(): void
	{
		$this['meta']->setMeta('robots', 'noindex, nofollow');

		if ($this->getParameter('orderIdent')) {
			$this->order           = $this->ordersService->getByIdent($this->getParameter('orderIdent'));
			$this->template->order = $this->order;
		}

		if (!$this->order) {
			$this->error();
		}

		$siteIdent = $this->order->site->getIdent();
		$seller    = $this->sellers->getSellerForSite($siteIdent);
		if ($seller->sendSellerEmailType === Seller::EMAIL_TYPE_SELLER) {
			$sellerEmail = EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$siteIdent}.email", null) ?: $seller->email;
		} else {
			$sellerEmail = $this->order->site->email;
		}
		$sellerName = EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$siteIdent}.name", null) ?: $seller->name;

		$country = $this->order->getAddressInvoice() && $this->order->getAddressInvoice()->getCountry()
			? Strings::upper($this->order->getAddressInvoice()->getCountry()->getId())
			: null;

		$customer = $this->order->getCustomer();
		if (
			$seller
			&& EshopOrdersConfig::load('orderForm.showOrderFinishedQrPayment')
			&& $country && Arrays::contains(EshopOrdersConfig::load('showQrForCountries'), $country)
			&& $this->order->getPaymentIdent() === 'transfer'
			&& (
				!$customer
				|| !$customer->getGroupCustomers()
				|| !Arrays::contains(EshopOrdersConfig::load('orderForm.hideOrderFinishedQrPaymentForCustomerGroup'), $customer->getGroupCustomers()->getId())
			)
		) {
			$this->template->qrPayment = $this->qrGenerator->getQrCodeByOrderAndSeller($this->order, $seller);
		}

		$this->template->sellerEmail = $sellerEmail;
		$this->template->sellerName  = $sellerName;

		$this->eventDispatcher->dispatch(new PresenterTemplateEvent($this->template, $this), 'eshopOrders.orderFinishedRender');
		$this->template->includeTemplates = $this->includeTemplates;
	}

	/*******************************************************************************************************************
	 * ============================== Renders
	 */

	public function renderOrderFinished(): void
	{
		if ($this->order) {
			$this->template->order = $this->order;
		}

		//flash zpravy zobrazime v textu, ne jako prekryvne
		$paymentType                  = $this->order ? $this->order->getPayment()->getPayment()->getIdent() : null;
		$this->template->paymentType  = $paymentType;
		$this->template->messages     = $this->getTemplate()->flashes;
		$this->getTemplate()->flashes = [];
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	/*******************************************************************************************************************
	 * ============================== Other
	 */

	public function addIncludeTemplate(string $file): self
	{
		$this->includeTemplates[] = $file;

		return $this;
	}
}
