<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice\InvoiceData;
use EshopOrders\Model\EshopOrdersConfig;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @ORM\Table("eshop_orders__invoice")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopOrders\Model\Listeners\InvoiceListener"})
 */
class Invoice
{
	use TId;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public ?\DateTime $dueDate = null;

	/**
	 * @var \DateTime|\DateTimeImmutable|null $duzp
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public $duzp = null;

	/**
	 * @ORM\Column(type="string", nullable=false)
	 */
	public string $ident;

	/**
	 * @ORM\OneToOne(targetEntity="Order", mappedBy="invoice", cascade={"persist"})
	 */
	public ?Order $order = null;

	/**
	 * @var \DateTime|\DateTimeImmutable $createdAt
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(type="datetime", nullable=false)
	 */
	public $createdAt;

	/**
	 * @ORM\OneToOne(targetEntity="EshopOrders\Model\Entities\Invoice\InvoiceData", mappedBy="invoice", cascade={"persist"})
	 */
	public ?InvoiceData $invoiceData = null;

	/**
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\Seller")
	 * @ORM\JoinColumn(name="seller_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Seller $seller;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	protected ?string $externalInvoiceFile = null;

	/**
	 * @ORM\Column(type="text", nullable=true, options={"default": null})
	 */
	public ?string $note = null;

	public function __construct(?\DateTime $dueDate, string $ident, Seller $seller)
	{
		$this->dueDate   = $dueDate;
		$this->ident     = $ident;
		$this->seller    = $seller;
		$this->createdAt = new \DateTime();
	}

	public function setExternalInvoiceFile(?string $externalInvoiceFile): void
	{
		if ($externalInvoiceFile === null) {
			$this->externalInvoiceFile = null;

			return;
		}

		if (file_exists(WWW_DIR . $externalInvoiceFile)) {
			$this->externalInvoiceFile = $externalInvoiceFile;
		}
	}

	public function getExternalInvoiceFile(): ?string
	{
		if ($this->externalInvoiceFile !== null && file_exists(WWW_DIR . $this->externalInvoiceFile)) {
			return $this->externalInvoiceFile;
		}

		return null;
	}

	public function isExternal(): bool
	{
		return $this->externalInvoiceFile !== null;
	}

	public function getDUZP(): ?\DateTimeInterface
	{
		if ($this->duzp) {
			return $this->duzp;
		}

		if (EshopOrdersConfig::load('invoice.document.separatedDUZPCreated')) {
			return $this->order->getCreatedTime() ?: $this->createdAt;
		}

		return $this->createdAt;
	}
}
