<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Arrays;
use Core\Model\Parameters;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Currency\DI\CurrencyExtension;
use Currency\Model\Config as CurrencyConfig;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use Nette\Utils\Validators;

/**
 * @ORM\Table("eshop_orders__order")
 * @ORM\Entity
 * @ORM\EntityListeners({"OrderListener"})
 */
class Order
{
	public const paramInternalNote = 'internalNote';

	use TId;

	/**
	 * @ORM\Column(name="ident", type="string", length=255, nullable=false)
	 */
	protected string $ident = '';

	/**
	 * @ORM\ManyToOne(targetEntity="Customer")
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected ?Customer $customer = null;

	/**
	 * @ORM\Column(name="message", type="text", nullable=true)
	 */
	protected ?string $message = null;

	/**
	 * @ORM\OneToOne(targetEntity="OrderPayment", cascade={"persist"})
	 * @ORM\JoinColumn(name="payment", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?OrderPayment $payment = null;

	/**
	 * @ORM\OneToOne(targetEntity="OrderSpedition", cascade={"persist"})
	 * @ORM\JoinColumn(name="spedition", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?OrderSpedition $spedition = null;

	/**
	 * @var ArrayCollection<OrderItem>
	 * @ORM\OneToMany(targetEntity="OrderItem", mappedBy="order", indexBy="id")
	 */
	protected Collection $orderItems;

	/**
	 * @var ArrayCollection<OrderDiscount>
	 * @ORM\OneToMany(targetEntity="OrderDiscount", mappedBy="order", indexBy="id")
	 */
	public Collection $orderDiscounts;

	/**
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_delivery_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?OrderAddress $addressDelivery = null;

	/**
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?OrderAddress $addressInvoice = null;

	/**
	 * @var ArrayCollection<OrderStatus>
	 * @ORM\OneToMany(targetEntity="OrderStatus", mappedBy="order", cascade={"persist"})
	 */
	protected Collection $orderStatuses;

	/**
	 * @var ArrayCollection<OrderPaidHistory>
	 * @ORM\OneToMany(targetEntity="OrderPaidHistory", mappedBy="order", cascade={"persist"})
	 */
	protected Collection $orderPaidHistory;

	/**
	 * @ORM\Column(name="agreed_terms", type="boolean", nullable=true, options={"default": 0})
	 */
	protected int $agreedTerms = 0;

	/**
	 * @var ArrayCollection<OrderFlag>
	 * @ORM\OneToMany(targetEntity="OrderFlag", mappedBy="order", indexBy="type")
	 */
	protected Collection $orderFlags;

	/**
	 * @ORM\OneToOne(targetEntity="Invoice", inversedBy="order")
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected ?Invoice $invoice = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Site")
	 * @ORM\JoinColumn(name="site_id", referencedColumnName="ident", onDelete="SET NULL", nullable=true)
	 */
	public ?Site $site = null;

	/**
	 * @var OrderCurrency|null $currency
	 *
	 * @ORM\OneToOne(targetEntity="OrderCurrency", mappedBy="order", cascade={"persist"})
	 */
	public $currency = null;

	/**
	 * @var ArrayCollection<OrderGift>
	 * @ORM\OneToMany(targetEntity="OrderGift", mappedBy="order")
	 */
	protected Collection $gifts;

	/**
	 * @ORM\Column(name="is_paid", type="smallint", options={"default": 0})
	 */
	public int $isPaid = 0;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public ?\DateTimeInterface $paid = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderForCorrectiveTaxDocument")
	 * @ORM\JoinColumn(referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	public ?Order $correctiveTaxDocumentOf = null;

	/**
	 * @var ArrayCollection<self>
	 * @ORM\OneToMany(targetEntity="Order", mappedBy="correctiveTaxDocumentOf")
	 */
	public Collection $orderForCorrectiveTaxDocument;

	/**
	 * @ORM\Column(type="smallint", options={"default": 0})
	 */
	public int $isCorrectiveTaxDocument = 0;

	/**
	 * @ORM\Column(name="lang", type="string", nullable=false, options={"default": "cs"})
	 */
	public string $lang;

	/**
	 * @ORM\Column(type="string", nullable=true, options={"default": null})
	 */
	public ?string $receiptIdent = null;

	/**
	 * @ORM\Column(type="smallint", options={"default": 1})
	 */
	public int $enableInvoiceGeneration = 1;

	public ?PriceFilter $priceFilter = null;

	/**
	 * @ORM\Column(type="array", nullable=true)
	 */
	protected ?array $params = null;

	/**
	 * @ORM\Column(type="smallint", options={"default": 0})
	 */
	public int $isMessageReadyToDelivery = 0;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public ?\DateTime $demandedExpeditionDate = null;

	/**
	 * @ORM\Column(type="integer", options={"default": 0})
	 */
	public int $invoiceReminderCount = 0;

	public function __construct(Site $site)
	{
		$this->ident                         = $this->genUuid();
		$this->site                          = $site;
		$this->orderItems                    = new ArrayCollection();
		$this->orderDiscounts                = new ArrayCollection();
		$this->orderFlags                    = new ArrayCollection();
		$this->gifts                         = new ArrayCollection();
		$this->orderStatuses                 = new ArrayCollection();
		$this->orderPaidHistory              = new ArrayCollection();
		$this->addressDelivery               = null;
		$this->addressInvoice                = null;
		$this->orderForCorrectiveTaxDocument = new ArrayCollection();
		$this->lang                          = (string) Parameters::load('translation.default', 'cs');
	}

	public function getIdent(): string { return $this->ident; }

	public function getOrderForCorrectiveTaxDocument(): ?self
	{
		if ($this->orderForCorrectiveTaxDocument instanceof self) {
			return $this->orderForCorrectiveTaxDocument;
		}

		$first = $this->orderForCorrectiveTaxDocument->first();
		if ($first !== false) {
			return $first;
		}

		return null;
	}

	public function getCustomer(): ?Customer { return $this->customer; }

	public function getMessage(): ?string { return $this->message; }

	public function getPrice(bool $useCurrency = false): float
	{
		$priceTotal = $this->getPriceItems($useCurrency);

		foreach ($this->orderDiscounts as $discount) {
			if ($priceTotal < 0) {
				$priceTotal += abs($discount->getPrice($useCurrency));
			} else {
				$priceTotal -= abs($discount->getPrice($useCurrency));
			}
		}

		$priceTotal += ($this->spedition ? $this->getSpedition()->getPrice($useCurrency) : 0)
			+ ($this->payment ? $this->getPayment()->getPrice($useCurrency) : 0);

		return $priceTotal;
	}

	public function getPriceWithoutDiscounts(bool $useCurrency = false): float
	{
		$priceTotal = $this->getPriceItems($useCurrency);

		$priceTotal += ($this->spedition ? $this->getSpedition()->getPrice($useCurrency) : 0)
			+ ($this->payment ? $this->getPayment()->getPrice($useCurrency) : 0);

		return $priceTotal;
	}

	public function getPriceItems(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->orderItems as $item) {
			$priceTotal += $item->getPriceTotal($useCurrency);
		}

		return $priceTotal;
	}

	public function getPriceItemsDiscount(bool $useCurrency = false): float
	{
		$priceTotal = $this->getPriceItems($useCurrency);
		foreach ($this->orderDiscounts as $discount) {
			$priceTotal -= abs($discount->getPrice($useCurrency));
		}

		return $priceTotal;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$itemsPrice   = $this->getPriceTotalWithoutSpedition(false, $useCurrency);
		$paySpedPrice = $this->getPaySpedPriceWithoutVat($useCurrency);

		return $itemsPrice + $paySpedPrice;
	}

	public function getItemsPriceWithoutVat(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getPriceTotalWithoutVat($useCurrency);
		}

		return $priceTotal;
	}

	public function getPriceTotalWithoutSpedition(bool $withVat = true, bool $useCurrency = false): float
	{
		if ($withVat) {
			return $this->getPriceItems($useCurrency) + $this->calculateDiscounts($this->getPriceItems($useCurrency));
		}

		$resultPrice = $this->getItemsPriceWithoutVat($useCurrency);
		$itemsPrice  = 0;
		$dphRates    = [];

		foreach ($this->getOrderItems() as $item) {
			if ($item->getMoreDataValue('discountDisabled')) {
				continue;
			}

			$itemsPrice += $item->getPriceTotalWithoutVat($useCurrency);
		}

		$zeroPrice = $itemsPrice === 0;

		foreach ($this->getOrderItems() as $item) {
			if ($item->getMoreDataValue('discountDisabled')) {
				continue;
			}

			$vat = $item->getVatRate();
			if (!isset($dphRates[$vat])) {
				$dphRates[$vat] = 0;
			}

			if (!$zeroPrice) {
				$dphRates[$vat] += $item->getPriceTotalWithoutVat($useCurrency);
			}
		}

		$totalSale = 0;
		foreach ($this->orderDiscounts as $discount) {
			$totalSale += abs($discount->getPrice($useCurrency));
		}

		foreach (OrderHelper::calculateVatWithSales($itemsPrice, $totalSale, $dphRates) as $v) {
			$resultPrice += $useCurrency
				? round($v['withoutVat'], $this->currency->decimals ?? 2)
				: round($v['withoutVat'], 2);
		}

		return $useCurrency
			? round($resultPrice, $this->currency->decimals ?? 2)
			: round($resultPrice, 2);
	}

	public function getItemsVatPrice(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getPriceTotal($useCurrency) - $item->getPriceTotalWithoutVat($useCurrency);
		}

		return $priceTotal;
	}

	public function getPaySpedPrice(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition() ? $this->getSpedition()->getPrice($useCurrency) : 0;
		$priceTotal += $this->getPayment() ? $this->getPayment()->getPrice($useCurrency) : 0;

		return $priceTotal;
	}

	public function getPaySpedPriceWithoutVat(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition() ? $this->getSpedition()->getPriceWithoutVat($useCurrency) : 0;
		$priceTotal += $this->getPayment() ? $this->getPayment()->getPriceWithoutVat($useCurrency) : 0;

		return $priceTotal;
	}

	public function getPaySpedId(): string { return $this->getSpedition()->getId() . '-' . $this->getPayment()->getId(); }

	public function getPaySpedName(): string { return $this->getSpedition()->getName() . ' - ' . $this->getPayment()->getName(); }

	public function getPayment(): ?OrderPayment
	{
		if ($this->payment && $this->payment->order === null) {
			$this->payment->order = $this;
		}
		
		return $this->payment;
	}

	public function getPaymentIdent(): ?string
	{
		return $this->getPayment() && $this->getPayment()->getPayment() ? $this->getPayment()->getPayment()->getIdent() : null;
	}

	public function getEshopPaymentId(): ?int { return $this->getPayment() ? $this->getPayment()->getPaymentId() : null; }

	public function getPaySpedVatRate(): int
	{
		if ($this->isZeroVat()) {
			return 0;
		}

		if ($this->getSpedition()) {
			return $this->getSpedition()->getVatRate();
		}

		return (int) EshopOrdersConfig::load('speditions.vat');
	}

	public function getSpedition(): ?OrderSpedition
	{
		if ($this->spedition && $this->spedition->order === null) {
			$this->spedition->order = $this;
		}

		return $this->spedition;
	}

	public function getSpeditionIdent(): ?string { return $this->getSpedition() ? $this->getSpedition()->getIdent() : null; }

	public function getEshopSpeditionId(): ?int { return $this->getSpedition() ? $this->getSpedition()->getSpeditionId() : null; }

	/**
	 * @return ArrayCollection<OrderItem>
	 */
	public function getOrderItems(): Collection
	{
		$items   = [];
		$parents = [];

		foreach ($this->orderItems as $k => $item) {
			if ($item->getId()) {
				if ($item->getParent()) {
					$parents[$item->getParent()->getId()][$item->getId()] = $item;
				} else {
					$items[$item->getId()] = $item;
				}
			} else {
				$items[$k] = $item;
			}
		}

		foreach ($parents as $k => $list) {
			Arrays::insertAfter($items, $k, $list);
		}

		$parents = null;

		return new ArrayCollection($items);
	}

	public function getOrderItemByProductId(int $productId): ?OrderItem
	{
		foreach ($this->getOrderItems() as $item) {
			if ($item->getProductId() === $productId) {
				return $item;
			}
		}

		return null;
	}

	public function getAddressDelivery(): ?OrderAddress { return $this->addressDelivery ?: $this->addressInvoice; }

	public function getAddressDeliveryRaw(): ?OrderAddress { return $this->addressDelivery; }

	public function getAddressInvoice(): ?OrderAddress { return $this->addressInvoice ?: ($this->addressDelivery ?: null); }

	public function getAddressInvoiceRaw(): ?OrderAddress { return $this->addressInvoice; }

	public function setCustomer(Customer $customer): Order
	{
		$this->customer = $customer;

		return $this;
	}

	public function setMessage(?string $message): self
	{
		$this->message = $message;

		return $this;
	}

	public function setPayment(OrderPayment $payment): self
	{
		$this->payment = $payment;

		return $this;
	}

	public function setSpedition(OrderSpedition $spedition): self
	{
		$this->spedition = $spedition;

		return $this;
	}

	/** @param OrderItem[] $orderItems */
	public function setOrderItems(array $orderItems): self
	{
		$this->orderItems = new ArrayCollection($orderItems);

		return $this;
	}

	public function setAddressDelivery(OrderAddress $addressDelivery): self
	{
		$this->addressDelivery = $addressDelivery;

		return $this;
	}

	public function setAddressInvoice(OrderAddress $addressInvoice): self
	{
		$this->addressInvoice = $addressInvoice;

		return $this;
	}

	/** @return ArrayCollection<OrderDiscount> */
	public function getOrderDiscounts(): Collection { return $this->orderDiscounts; }

	/** @param OrderDiscount[] $orderDiscounts */
	public function setOrderDiscounts(array $orderDiscounts): Order
	{
		$this->orderDiscounts = new ArrayCollection($orderDiscounts);

		return $this;
	}

	public function addOrderDiscount(OrderDiscount $orderDiscount): self
	{
		$this->orderDiscounts->add($orderDiscount);

		return $this;
	}

	protected function calculateDiscounts(float $price): float
	{
		foreach ($this->getOrderDiscounts() as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	public function getAgreedTerms(): bool { return (bool) $this->agreedTerms; }

	/** @param bool|int $agreedTerms */
	public function setAgreedTerms($agreedTerms): self
	{
		$this->agreedTerms = (int) $agreedTerms;

		return $this;
	}

	/** @return ArrayCollection<OrderFlag> */
	public function getOrderFlags(): Collection { return $this->orderFlags; }

	/** @param OrderFlag[] $orderFlags */
	public function setOrderFlags(array $orderFlags): self
	{
		$this->orderFlags = new ArrayCollection($orderFlags);

		return $this;
	}

	public function addFlag(OrderFlag $flag): self
	{
		if (!$this->orderFlags->containsKey($flag->getType())) {
			$this->orderFlags->set($flag->getType(), $flag);
		}

		return $this;
	}

	public function hasFlag(string $type): bool
	{
		return ($this->orderFlags->containsKey($type) && $this->orderFlags[$type] == true);
	}

	/** @return ArrayCollection<OrderStatus> */
	public function getOrderStatuses(): Collection { return $this->orderStatuses; }

	public function getNewestOrderStatus(): ?OrderStatus
	{
		$last = null;

		foreach ($this->orderStatuses as $status) {
			if (
				!$status->isDeleted()
				&& (!$last || $status->getCreated() > $last->getCreated())
			) {
				$last = $status;
			}
		}

		return $last ?: $this->getOrderStatuses()->last();
	}

	/** @param OrderStatus[] $orderStatuses */
	public function setOrderStatuses(array $orderStatuses): Order
	{
		$this->orderStatuses = new ArrayCollection($orderStatuses);

		return $this;
	}

	public function getCreatedTime(): ?\DateTime
	{
		foreach ($this->orderStatuses as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === 'created') {
				$createdStatus = $orderStatus;
				break;
			}
		}

		if (isset($createdStatus)) {
			return $createdStatus->getCreated();
		}

		return null;
	}

	public function getInvoice(): ?Invoice { return $this->invoice; }

	public function setInvoice(?Invoice $invoice): void { $this->invoice = $invoice; }

	public function getCurrencyCode(): string
	{
		if (class_exists(CurrencyExtension::class) && $this->currency) {
			return $this->currency->code;
		}

		return $this->getDefaultCurrency();
	}

	public function getDefaultCurrency(): string { return class_exists(CurrencyConfig::class) ? CurrencyConfig::load('default') : 'CZK'; }

	public function calculateCurrencyPrice(float $price): float
	{
		return $this->currency
			? OrderHelper::calculateCurrencyPrice($price, (float) $this->currency->rate, (int) $this->currency->decimals)
			: $price;
	}

	public ?int $renderPriceDecimals = null;

	public function renderPrice(float $price, bool $useCurrency = false, ?string $method = null, ?bool $currencyFromLeft = null, ?string $separator = null, ?string $decSep = null, ?int $decimals = null): string
	{
		if (!$method) {
			$method = 'format';
		}

		if (!$decimals && $this->renderPriceDecimals) {
			$decimals = $this->renderPriceDecimals;
		}

		return $this->priceFilter->$method($price, $useCurrency ? $this->getCurrencyCode() : $this->getDefaultCurrency(), $currencyFromLeft, $separator, $decSep, $decimals);
	}

	/*************************************
	 * == Gifts
	 */

	/** @return ArrayCollection<OrderGift> */
	public function getGifts(): Collection { return $this->gifts; }

	public function addGift(OrderGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}

	/** http://stackoverflow.com/a/2040279
	 * @return string UUID - unikatni ID, tezke na uhodnuti
	 */
	protected function genUuid(): string
	{
		return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff),

			// 16 bits for "time_mid"
			mt_rand(0, 0xffff),

			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand(0, 0x0fff) | 0x4000,

			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand(0, 0x3fff) | 0x8000,

			// 48 bits for "node"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
		);
	}

	public function getBaseVatRates(bool $useCurrency = false): array
	{
		$list = [];

		foreach ($this->orderItems as $item) {
			$vat = $item->getVatRate();

			if (!isset($list[$vat])) {
				$list[$vat] = [
					'withoutVat' => 0,
					'total'      => 0,
				];
			}

			$list[$vat]['withoutVat'] += $item->getPriceTotalWithoutVat($useCurrency);
			$list[$vat]['total']      += $item->getPriceTotal($useCurrency);
		}

		if (!isset($list[$this->getPaySpedVatRate()])) {
			$list[$this->getPaySpedVatRate()] = [
				'withoutVat' => 0,
				'total'      => 0,
			];
		}

		$list[$this->getPaySpedVatRate()]['withoutVat'] += $this->getPaySpedPriceWithoutVat($useCurrency);
		$list[$this->getPaySpedVatRate()]['total']      += $this->getPaySpedPrice($useCurrency);

		return $list;
	}

	public function getVatRates(bool $useCurrency = false): array
	{
		$list = $this->getBaseVatRates($useCurrency);

		$discountsByVat = $this->getDiscountsByVat($useCurrency);
		if ($discountsByVat) {
			foreach ($discountsByVat as $vat => $value) {
				$list[$vat]['withoutVat'] += $value['withoutVat'];
				$list[$vat]['total']      += $value['total'];
			}
		}

		foreach ($list as $vat => $prices) {
			if ($vat === 0) {
				unset($list[$vat]);
				continue;
			}

			$list[$vat]['rate'] = round($prices['total'] - $prices['withoutVat'], 2);
		}

		return $list;
	}

	/** @return OrderItem[] */
	public function getItemsForDiscount(): array
	{
		$items = [];

		foreach ($this->orderItems as $k => $item) {
			if ($item->getMoreDataValue('discountDisabled', false)) {
				continue;
			}

			$items[$k] = $item;
		}

		return $items;
	}

	public function getDiscountsByVat(bool $useCurrency = false): array
	{
		if (empty($this->orderDiscounts->toArray())) {
			return [];
		}

		$itemsPrice = 0;
		$dphRates   = [];

		foreach ($this->getItemsForDiscount() as $item) {
			$itemsPrice += $item->getPriceTotal($useCurrency);
		}

		$zeroPrice = $itemsPrice === 0;

		foreach ($this->getItemsForDiscount() as $item) {
			$vat = $item->getVatRate();
			if (!isset($dphRates[$vat])) {
				$dphRates[$vat] = 0;
			}

			if (!$zeroPrice) {
				$dphRates[$vat] += $item->getPriceTotal($useCurrency);
			}
		}

		$totalSale = 0;
		foreach ($this->orderDiscounts as $discount) {
			$totalSale += abs($discount->getPrice($useCurrency));
		}

		return OrderHelper::calculateVatWithSales($itemsPrice, $totalSale, $dphRates);
	}

	public function isZeroVat(): bool
	{
		$addrInv = $this->getAddressInvoice();

		if (!$addrInv || !$addrInv->getCountry()) {
			return false;
		}

		$vatRate = OrderHelper::checkCountryVatRate(
			21,
			$addrInv->getCountry()->getId(),
			(bool) $addrInv->validatedVatNumber,
			$addrInv->getIdNumber(),
			$addrInv->getVatNumber(),
		);

		return $vatRate === 0;
	}

	public function getPaidDate(): ?\DateTimeInterface { return $this->isPaid ? $this->paid : null; }

	public function __clone()
	{
		$this->id    = null;
		$this->ident = $this->genUuid();
	}

	public function isCanceled(): bool
	{
		$isCanceled = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$isCanceled = true;
			}
		}

		return $isCanceled;
	}

	public function isShipped(): bool
	{
		$isShipped = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_SPEDITION && !$orderStatus->isDeleted()) {
				$isShipped = true;
			}
		}

		return $isShipped;
	}

	public function isFinished(): bool
	{
		$isShipped = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_FINISHED && !$orderStatus->isDeleted()) {
				$isShipped = true;
			}
		}

		return $isShipped;
	}

	public function getItemsWeight(bool $inKg = true): float
	{
		$weight = 0;

		foreach ($this->getOrderItems() as $orderItem) {
			if ($orderItem->getProduct() && $orderItem->getProduct()->weight) {
				$weight += $orderItem->getProduct()->weight;
			}
		}

		return round($inKg ? $weight / 1000 : $weight, 2);
	}

	public function hasPreorderProduct(): bool
	{
		foreach ($this->getOrderItems() as $item)
			if ($item->getMoreDataValue('isPreorder', false))
				return true;

		return false;
	}

	public function getDiscountsCodes(): array { return array_map(static fn(OrderDiscount $d) => $d->getCode(), $this->getOrderDiscounts()->toArray()); }

	public function isOtherCountryVatFilled(): bool
	{
		$addrInv = $this->getAddressInvoice();
		$country = $addrInv && $addrInv->getCountry() ? mb_strtoupper($addrInv->getCountry()->getId()) : null;

		return OrderHelper::isOtherCountryVatFilled($country, $addrInv ? $addrInv->getVatNumber() : null);
	}

	public function isOrderItemsReadyToDelivery(): bool
	{
		foreach ($this->orderItems as $orderItem) {
			if (!$orderItem->isReadyToDelivery) {
				return false;
			}
		}

		foreach ($this->getGifts() as $gift) {
			if (!$gift->isReadyToDelivery) {
				return false;
			}
		}

		return true;
	}

	public function getParams(): array { return $this->params ?: []; }

	/** @return mixed */
	public function getParam(string $key) { return $this->getParams()[$key] ?? null; }

	/** @param mixed $value */
	public function setParam(string $key, $value): void
	{
		if (!is_array($this->params)) {
			$this->params = [];
		}

		$this->params[$key] = $value;
	}

	public function removeParam(string $key): void
	{
		if (is_array($this->params)) {
			unset($this->params[$key]);
		}
	}

	public function isMessageReadyToDelivery(): bool
	{
		return Validators::isNone($this->getMessage()) || $this->isMessageReadyToDelivery;
	}

	public function increaseInvoiceReminderCount(): void
	{
		$max = (int) max(Customer::$invoiceReminderOptions);
		$this->invoiceReminderCount = min($this->invoiceReminderCount + 1, $max);

		if ($this->getCustomer()) {
			$this->getCustomer()->invoiceReminderCount = $this->invoiceReminderCount;
		}
	}

}
