<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Helpers\OrderHelper;

/** podobne jako order_item, ale samostatna tabulka
 * @ORM\Table("eshop_orders__order_payment")
 * @ORM\Entity
 */
class OrderPayment
{
	use TId;

	/**
	 * @ORM\ManyToOne(targetEntity="Payment")
	 * @ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?Payment $payment = null;

	/**
	 * @ORM\Column(name="name", type="string")
	 */
	protected ?string $name = null;

	/**
	 * @var float|string
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @ORM\OneToOne(targetEntity="Order", inversedBy="payment")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public ?Order $order = null;

	/**
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	protected ?int $vatRate = null;

	public function __construct(?Payment $payment, Order $order)
	{
		$this->payment = $payment;
		$this->name    = $payment ? $payment->getName() : '';
		$this->order   = $order;

		if ($payment) {
			$this->price = $payment->getPrice();
		}
	}

	public function getIdent(): ?string { return $this->getPayment() ? $this->getPayment()->getIdent() : null; }

	public function getPaymentId(): ?int { return $this->payment ? $this->payment->getId() : null; }

	public function getPayment(): ?Payment { return $this->payment; }

	public function setPayment(Payment $payment): self
	{
		$this->payment = $payment;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		if ($this->getOrder() && $this->getOrder()->isZeroVat()) {
			$result = $this->getPriceWithoutVat($useCurrency);
		} else {
			$result = $useCurrency
				? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
				: (float) $this->price;
		}

		if ($useCurrency) {
			$result = round($result, $this->getOrder()->currency->decimals ?? 2);
		}

		return $result;
	}

	public function getRawPrice(): float
	{
		return (float) $this->price;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$price = $useCurrency
			? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
			: (float) $this->price;

		$price = OrderHelper::removeVat($price, $this->vatRate);

		if ($useCurrency) {
			$price = round($price, $this->getOrder()->currency->decimals ?? 2);
		} else {
			$price = round($price, 2);
		}

		return $price;
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getOrder(): ?Order { return $this->order; }

	public function setVatRate(?int $vatRate): void { $this->vatRate = $vatRate; }

	public function getVatRate(): int { return $this->getOrder()->isZeroVat() ? 0 : ($this->vatRate ?: 21); }
}
